/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Transpose  transxy         Transpose X/Y
*/

#include <cdi.h>


#include "dmemory.h"
#include "process_int.h"

void
transxy(int gridID, double *array1, double *array2)
{
  auto nx = gridInqXsize(gridID);
  auto ny = gridInqYsize(gridID);
  auto gridsize = nx * ny;

  if (gridsize > 0)
    {
      double **a2D1 = (double **) Malloc(ny * sizeof(double *));
      double **a2D2 = (double **) Malloc(nx * sizeof(double *));

      for (size_t j = 0; j < ny; ++j) a2D1[j] = array1 + j * nx;
      for (size_t i = 0; i < nx; ++i) a2D2[i] = array2 + i * ny;

      for (size_t j = 0; j < ny; ++j)
        for (size_t i = 0; i < nx; ++i) a2D2[i][j] = a2D1[j][i];

      Free(a2D1);
      Free(a2D2);
    }
  else
    {
      gridsize = gridInqSize(gridID);
      for (size_t i = 0; i < gridsize; ++i) array2[i] = array1[i];
    }
}

void *
Transpose(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto ngrids = vlistNgrids(vlistID1);
  for (int index = 0; index < ngrids; index++)
    {
      const auto gridID1 = vlistGrid(vlistID1, index);
      const auto nx = gridInqXsize(gridID1);
      const auto ny = gridInqYsize(gridID1);
      const auto gridsize = nx * ny;
      if (gridsize > 0)
        {
          const auto gridID2 = gridCreate(GRID_GENERIC, gridsize);
          gridDefXsize(gridID2, ny);
          gridDefYsize(gridID2, nx);
          vlistChangeGridIndex(vlistID2, index, gridID2);
        }
    }

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  std::vector<double> array1(gridsizemax);
  std::vector<double> array2(gridsizemax);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, array1.data(), &nmiss);

          auto gridID = vlistInqVarGrid(vlistID1, varID);
          transxy(gridID, array1.data(), array2.data());

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, array2.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
