/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

     Subtrend   subtrend        Subtract trend
*/

#include <cdi.h>


#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"


static void
subtrend(int tsID, const Field &field1, const Field &field2, const Field &field3, Field &field4)
{
  const auto gridsize = field1.size;
  const auto missval = field1.missval;
  const auto missval1 = missval;
  const auto missval2 = missval;
  for (size_t i = 0; i < gridsize; i++) field4.vec[i] = SUBMN(field1.vec[i], ADDMN(field2.vec[i], MULMN(field3.vec[i], tsID)));
}

void *
Subtrend(void *process)
{
  int varID, levelID;

  cdoInitialize(process);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);
  const auto streamID3 = cdoOpenRead(2);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = cdoStreamInqVlist(streamID3);
  const auto vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_DIM);
  vlistCompare(vlistID1, vlistID3, CMP_DIM);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID4 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID4, taxisID4);

  const auto streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  VarList varList;
  varListInit(varList, vlistID1);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field4;
  field1.resize(gridsizemax);
  field4.resize(gridsizemax);

  FieldVector2D vars2, vars3;
  fieldsFromVlist(vlistID1, vars2, FIELD_VEC);
  fieldsFromVlist(vlistID1, vars3, FIELD_VEC);

  int tsID = 0;
  auto nrecs = cdoStreamInqTimestep(streamID2, tsID);
  for (int recID = 0; recID < nrecs; recID++)
    {
      cdoInqRecord(streamID2, &varID, &levelID);
      cdoReadRecord(streamID2, vars2[varID][levelID].vec.data(), &vars2[varID][levelID].nmiss);
    }

  tsID = 0;
  nrecs = cdoStreamInqTimestep(streamID3, tsID);
  for (int recID = 0; recID < nrecs; recID++)
    {
      cdoInqRecord(streamID3, &varID, &levelID);
      cdoReadRecord(streamID3, vars3[varID][levelID].vec.data(), &vars3[varID][levelID].nmiss);
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID4, taxisID1);
      cdoDefTimestep(streamID4, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, field1.vec.data(), &field1.nmiss);

          field1.size = varList[varID].gridsize;
          field1.missval = varList[varID].missval;
          field4.size = varList[varID].gridsize;
          field4.missval = varList[varID].missval;

          subtrend(tsID, field1, vars2[varID][levelID], vars3[varID][levelID], field4);

          cdoDefRecord(streamID4, varID, levelID);
          cdoWriteRecord(streamID4, field4.vec.data(), fieldNumMiss(field4));
        }

      tsID++;
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
