/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

     Sorttimestamp    sorttimestamp         Sort all timesteps
*/

#include <cdi.h>


#include "cdo_options.h"
#include "dmemory.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"

#include "printinfo.h"
#include "timebase.h"

#define NALLOC_INC 1024

struct timeinfo_t
{
  int index;
  double datetime;
};

static int
cmpdatetime(const void *a, const void *b)
{
  const double x = ((const timeinfo_t *) a)->datetime;
  const double y = ((const timeinfo_t *) b)->datetime;
  return ((x > y) - (x < y)) * 2 + (x > y) - (x < y);
}

void *
Sorttimestamp(void *process)
{
  int nrecs;
  int gridID, varID, levelID;
  int tsID, lasttsID = -1;
  int nalloc = 0;
  int vlistID2 = -1, taxisID2 = -1;
  size_t nmiss;
  int nvars = 0;

  cdoInitialize(process);

  FieldVector3D vars;
  std::vector<int64_t> vdate;
  std::vector<int> vtime;

  const auto nfiles = cdoStreamCnt() - 1;

  int xtsID = 0;
  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      const auto streamID1 = cdoOpenRead((fileID));

      const auto vlistID1 = cdoStreamInqVlist(streamID1);
      const auto taxisID1 = vlistInqTaxis(vlistID1);

      if (fileID == 0)
        {
          vlistID2 = vlistDuplicate(vlistID1);
          taxisID2 = taxisDuplicate(taxisID1);
          if (taxisHasBounds(taxisID2))
            {
              cdoWarning("Time bounds unsupported by this operator, removed!");
              taxisDeleteBounds(taxisID2);
            }
        }
      else
        {
          vlistCompare(vlistID2, vlistID1, CMP_ALL);
        }

      nvars = vlistNvars(vlistID1);

      int tsID = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          if (xtsID >= nalloc)
            {
              nalloc += NALLOC_INC;
              vdate.resize(nalloc);
              vtime.resize(nalloc);
              vars.resize(nalloc);
            }

          vdate[xtsID] = taxisInqVdate(taxisID1);
          vtime[xtsID] = taxisInqVtime(taxisID1);

          fieldsFromVlist(vlistID1, vars[xtsID], FIELD_NONE);

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);
              gridID = vlistInqVarGrid(vlistID1, varID);
              const auto gridsize = gridInqSize(gridID);
              vars[xtsID][varID][levelID].resize(gridsize);
              cdoReadRecord(streamID1, vars[xtsID][varID][levelID].vec.data(), &nmiss);
              vars[xtsID][varID][levelID].nmiss = nmiss;
            }

          tsID++;
          xtsID++;
        }

      cdoStreamClose(streamID1);
    }

  int nts = xtsID;

  timeinfo_t *timeinfo = (timeinfo_t *) Malloc(nts * sizeof(timeinfo_t));
  const auto calendar = taxisInqCalendar(taxisID2);

  for (tsID = 0; tsID < nts; tsID++)
    {
      const auto julday = date_to_julday(calendar, vdate[tsID]);
      const auto secofday = time_to_sec(vtime[tsID]);
      const double vdatetime = julday + secofday / 86400.;
      timeinfo[tsID].index = tsID;
      timeinfo[tsID].datetime = vdatetime;
    }

  qsort(timeinfo, nts, sizeof(timeinfo_t), cmpdatetime);

  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite((nfiles));
  cdoDefVlist(streamID2, vlistID2);

  int tsID2 = 0;
  for (tsID = 0; tsID < nts; tsID++)
    {
      xtsID = timeinfo[tsID].index;

      if (tsID > 0)
        {
          if (IS_EQUAL(timeinfo[tsID].datetime, timeinfo[lasttsID].datetime))
            {
              if (Options::cdoVerbose)
                {
                  cdoPrint("Timestep %4d %s %s already exists, skipped!", xtsID,
                           dateToString(vdate[xtsID]).c_str(), timeToString(vtime[xtsID]).c_str());
                }
              continue;
            }
        }

      lasttsID = tsID;

      taxisDefVdate(taxisID2, vdate[xtsID]);
      taxisDefVtime(taxisID2, vtime[xtsID]);
      cdoDefTimestep(streamID2, tsID2++);

      for (varID = 0; varID < nvars; varID++)
        {
          const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              if (!vars[xtsID][varID][levelID].empty())
                {
                  nmiss = vars[xtsID][varID][levelID].nmiss;
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, vars[xtsID][varID][levelID].vec.data(), nmiss);
                }
            }
        }
    }

  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
