/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Setgatt    setgatt         Set global attribute
      Setgatt    setgatts        Set global attributes
*/

#include <cdi.h>


#include "process_int.h"
#include "readline.h"

void *
Setgatt(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;
  char *attname = nullptr, *attstring = nullptr, *attfile = nullptr;

  cdoInitialize(process);

  const int SETGATT = cdoOperatorAdd("setgatt", 0, 0, "attribute name and string");
  cdoOperatorAdd("setgatts", 0, 0, nullptr);

  const int operatorID = cdoOperatorID();

  if (operatorID == SETGATT)
    {
      operatorInputArg(cdoOperatorEnter(operatorID));
      attname = operatorArgv()[0];
      attstring = operatorArgv()[1];
    }
  else
    {
      attfile = operatorArgv()[0];
    }

  const auto streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);

  if (operatorID == SETGATT)
    {
      cdiDefAttTxt(vlistID2, CDI_GLOBAL, attname, (int) strlen(attstring), attstring);
    }
  else
    {
      char line[1024];
      int attlen = 0;

      FILE *fp = fopen(attfile, "r");
      if (fp == nullptr) cdoAbort("Open failed on %s", attfile);

      while (readline(fp, line, 1024))
        {
          attlen = 0;
          if (line[0] == '#') continue;
          if (line[0] == '\0') continue;
          attname = line;
          while (isspace((int) *attname)) attname++;
          if (attname[0] == '\0') continue;
          attstring = attname;
          while (*attstring != ' ' && *attstring != '\0' && *attstring != '=' && *attstring != '"') attstring++;
          if (*attstring == '\0')
            attstring = nullptr;
          else
            {
              *attstring = '\0';
              attstring++;
              while (isspace((int) *attstring) || (int) *attstring == '=' || (int) *attstring == '"' || (int) *attstring == '\'')
                attstring++;
              attlen = strlen(attstring);
              if (attstring[attlen - 1] == '"' || attstring[attlen - 1] == '\'') attstring[--attlen] = 0;
            }

          if (attstring && attlen) cdiDefAttTxt(vlistID2, CDI_GLOBAL, attname, attlen, attstring);
        }

      fclose(fp);
    }

  cdoDefVlist(streamID2, vlistID2);

  auto gridsizemax = vlistGridsizeMax(vlistID1);
  if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
  std::vector<double> array(gridsizemax);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoDefRecord(streamID2, varID, levelID);

          cdoReadRecord(streamID1, array.data(), &nmiss);
          cdoWriteRecord(streamID2, array.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
