/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Pardup     pardup          Duplicate parameters
      Pardup     parmul          Multiply parameters
*/

#include <cdi.h>


#include "process_int.h"
#include "param_conversion.h"

void *
Pardup(void *process)
{
  int nrecs;
  int varID, varID2, levelID;
  size_t offset;
  int nmul = 0;
  size_t nmiss;
  int nlevel;
  double *single;

  cdoInitialize(process);

  const auto PARDUP = cdoOperatorAdd("pardup", 0, 0, nullptr);
  const auto PARMUL = cdoOperatorAdd("parmul", 0, 0, nullptr);

  const auto operatorID = cdoOperatorID();

  if (operatorID == PARDUP)
    {
      nmul = 2;
    }
  else if (operatorID == PARMUL)
    {
      operatorInputArg("number of multiply");
      nmul = parameter2int(operatorArgv()[0]);
    }
  else
    cdoAbort("operator not implemented!");

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  std::vector<double> array(gridsizemax);
  std::vector<std::vector<double>> vardata(nvars);
  std::vector<std::vector<size_t>> varnmiss(nvars);

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
      nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      vardata[varID].resize(gridsize * nlevel);
      varnmiss[varID].resize(nlevel);
    }

  for (int i = 1; i < nmul; i++)
    {
      vlistCat(vlistID2, vlistID1);
      for (varID = 0; varID < nvars; varID++)
        vlistDefVarParam(vlistID2, varID + nvars * i, cdiEncodeParam(-(varID + nvars * i + 1), 255, 255));
    }

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          recList[recID].varID = varID;
          recList[recID].levelID = levelID;
          recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;

          const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
          offset = gridsize * levelID;
          single = &vardata[varID][offset];

          cdoReadRecord(streamID1, single, &nmiss);
          varnmiss[varID][levelID] = nmiss;
        }

      for (int i = 0; i < nmul; i++)
        for (int recID = 0; recID < nrecs; recID++)
          {
            varID = recList[recID].varID;
            levelID = recList[recID].levelID;
            varID2 = varID + i * nvars;

            const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
            offset = gridsize * levelID;
            single = &vardata[varID][offset];
            nmiss = varnmiss[varID][levelID];

            arrayCopy(gridsize, single, array.data());
            cdoDefRecord(streamID2, varID2, levelID);
            cdoWriteRecord(streamID2, array.data(), nmiss);
          }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
