/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>


#include "cdo_options.h"
#include "process_int.h"

#include "listarray.h"

void *
Histogram(void *process)
{
  int nrecs, varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  // clang-format off
  const auto HISTCOUNT = cdoOperatorAdd("histcount", 0, 0, nullptr);
  const auto HISTSUM   = cdoOperatorAdd("histsum",   0, 0, nullptr);
  const auto HISTMEAN  = cdoOperatorAdd("histmean",  0, 0, nullptr);
  const auto HISTFREQ  = cdoOperatorAdd("histfreq",  0, 0, nullptr);
  // clang-format on

  (void)(HISTSUM); //CDO_UNUSED

  const auto operatorID = cdoOperatorID();

  operatorInputArg("bins");

  ListArray<double> listArrayFlt;
  int nbins = listArrayFlt.argvToFlt(operatorArgc(), operatorArgv()) - 1;
  if (nbins < 1) cdoAbort("Too few arguments!");
  double *fltarr = listArrayFlt.data();

  if (Options::cdoVerbose)
    {
      printf("nbins = %d\n", nbins);
      for (int i = 0; i < nbins; i++) printf("flt %d = %g\n", i + 1, fltarr[i]);
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto taxisID1 = vlistInqTaxis(vlistID1);

  const auto vlistID2 = vlistDuplicate(vlistID1);

  /* create zaxis for output bins */
  const auto zaxisID2 = zaxisCreate(ZAXIS_GENERIC, nbins);

  {
    std::vector<double> bins(nbins);
    /* for ( int i = 0; i < nbins; i++ ) bins[i] = (fltarr[i]+fltarr[i+1])/2; */
    for (int i = 0; i < nbins; i++) bins[i] = fltarr[i];
    zaxisDefLevels(zaxisID2, bins.data());
  }

  zaxisDefLbounds(zaxisID2, fltarr);
  zaxisDefUbounds(zaxisID2, fltarr + 1);
  zaxisDefName(zaxisID2, "bin");
  zaxisDefLongname(zaxisID2, "histogram bins");
  zaxisDefUnits(zaxisID2, "level");

  /* check zaxis: only 2D fields allowed */
  int nzaxis = vlistNzaxis(vlistID1);
  for (int index = 0; index < nzaxis; index++)
    {
      const auto zaxisID = vlistZaxis(vlistID1, index);
      const auto nlevel = zaxisInqSize(zaxisID);
      if (nlevel > 1) cdoAbort("Found 3D field with %d levels. Only 2D fields allowed!", nlevel);
      vlistChangeZaxisIndex(vlistID2, index, zaxisID2);
    }

  const auto streamID2 = cdoOpenWrite(1);

  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  cdoDefVlist(streamID2, vlistID2);

  const auto nvars = vlistNvars(vlistID2);
  std::vector<std::vector<double>> vardata(nvars);
  std::vector<std::vector<double>> varcount(nvars);
  std::vector<std::vector<double>> vartcount(nvars);
  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID));
      vardata[varID].resize(nbins * gridsize, 0);
      varcount[varID].resize(nbins * gridsize, 0);
      vartcount[varID].resize(gridsize, 0);
    }

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  std::vector<double> array(gridsizemax);

  int tsID1 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID1)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, array.data(), &nmiss);
          const auto missval = vlistInqVarMissval(vlistID1, varID);

          const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));

          nmiss = 0;
          for (size_t i = 0; i < gridsize; i++)
            {
              if (!DBL_IS_EQUAL(array[i], missval))
                {
                  vartcount[varID][i] += 1;
                  int index = 0;
                  while (index < nbins)
                    {
                      const auto offset = gridsize * index;
                      if (!DBL_IS_EQUAL(vardata[varID][offset + i], missval) && array[i] >= fltarr[index]
                          && array[i] < fltarr[index + 1])
                        {
                          vardata[varID][offset + i] += array[i];
                          varcount[varID][offset + i] += 1;
                          break;
                        }
                      index++;
                    }
                }
              else
                { /* missing value */
                  nmiss++;
                }
            }
        }
      tsID1++;
    }

  cdoDefTimestep(streamID2, 0);

  for (varID = 0; varID < nvars; varID++)
    {
      const auto missval = vlistInqVarMissval(vlistID2, varID);
      const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID));

      /* fix mising values */

      for (int index = 0; index < nbins; index++)
        {
          nmiss = 0;
          const auto offset = gridsize * index;

          for (size_t i = 0; i < gridsize; i++)
            {
              if (vartcount[varID][i] > 0)
                {
                  if (operatorID == HISTMEAN || operatorID == HISTFREQ)
                    {
                      if (varcount[varID][offset + i] > 0)
                        {
                          if (operatorID == HISTMEAN)
                            vardata[varID][offset + i] /= varcount[varID][offset + i];
                          else
                            vardata[varID][offset + i] = varcount[varID][offset + i] / vartcount[varID][i];
                        }
                    }
                }
              else
                {
                  nmiss++;
                  varcount[varID][offset + i] = missval;
                  vardata[varID][offset + i] = missval;
                }
            }

          cdoDefRecord(streamID2, varID, index);

          if (operatorID == HISTCOUNT)
            cdoWriteRecord(streamID2, &varcount[varID][offset], nmiss);
          else
            cdoWriteRecord(streamID2, &vardata[varID][offset], nmiss);
        }
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
