/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Copy       copy            Copy datasets
*/

#include <cdi.h>


#include "dmemory.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "par_io.h"
#include "progress.h"
#include "cdo_options.h"

extern "C" int streamGrbInqDataScanningMode(void);

void *
Copy(void *process)
{
  bool lconstvars = true;
  CdoStreamID streamID2 = CDO_STREAM_UNDEF;
  int vlistID2 = CDI_UNDEFID;
  int taxisID2 = CDI_UNDEFID;
  int nrecs;
  int varID, levelID;
  size_t nmiss;
  void *array = nullptr;
  par_io_t parIO;

  cdoInitialize(process);

  const auto lcopy = unchangedRecord();

  // clang-format off
                       cdoOperatorAdd("copy",   0, 0, nullptr);
  const auto SELALL  = cdoOperatorAdd("selall", 0, 0, nullptr);
  const auto SZIP    = cdoOperatorAdd("szip",   0, 0, nullptr);
  // clang-format on

#ifdef HIRLAM_EXTENSIONS
  // KEEP in mind the difference between copy and selall with respect to
  // unpacking and repacking the GRIB information! Especially when setting the
  // DataScanningMode.
  printf("cdo copy/selall : UNCHANGED_RECORD=%d\n", unchangedRecord());
  // if (cdiGribDataScanningMode != -1) lcopy = false;
  printf("cdo copy/selall : cdiGribDataScanningMode=%d; lcopy=%d\n", streamGrbInqDataScanningMode(), lcopy);
#endif  //#ifdef HIRLAM_EXTENSIONS

  const auto operatorID = cdoOperatorID();
  if (operatorID == SZIP)
    {
      Options::cdoCompType = CDI_COMPRESS_SZIP;
      Options::cdoCompLevel = 0;
    }

  const auto streamCnt = cdoStreamCnt();
  const auto nfiles = streamCnt - 1;

  Progress::init();

  int tsID2 = 0;
  for (int indf = 0; indf < nfiles; ++indf)
    {
      if (Options::cdoVerbose) cdoPrint("Process file: %s", cdoGetStreamName(indf));

      const auto streamID1 = cdoOpenRead(indf);
      const auto vlistID1 = cdoStreamInqVlist(streamID1);
      const auto taxisID1 = vlistInqTaxis(vlistID1);

      if (indf == 0)
        {
          vlistID2 = vlistDuplicate(vlistID1);
          taxisID2 = taxisDuplicate(taxisID1);
          vlistDefTaxis(vlistID2, taxisID2);

          auto ntsteps = vlistNtsteps(vlistID1);
          const auto nvars = vlistNvars(vlistID1);

          if (ntsteps == 1)
            {
              for (varID = 0; varID < nvars; ++varID)
                if (vlistInqVarTimetype(vlistID1, varID) != TIME_CONSTANT) break;

              if (varID == nvars) ntsteps = 0;
            }

          if (ntsteps == 0 && nfiles > 1)
            {
              lconstvars = false;
              for (varID = 0; varID < nvars; ++varID) vlistDefVarTimetype(vlistID2, varID, TIME_VARYING);
            }

          const auto gridsizemax = vlistGridsizeMax(vlistID1);
          if (Options::CDO_Memtype == MEMTYPE_FLOAT)
            array = Malloc(gridsizemax * sizeof(float));
          else
            array = Malloc(gridsizemax * sizeof(double));

          if (Options::cdoParIO)
            {
              fprintf(stderr, "Parallel reading enabled!\n");
              parIO.array = (double *) Malloc(gridsizemax * sizeof(double));
              parIO.array_size = gridsizemax;
            }
        }
      else
        {
          vlistCompare(vlistID1, vlistID2, CMP_ALL);
        }

      if (streamID2 == CDO_STREAM_UNDEF)
        {
          streamID2 = cdoOpenWrite((nfiles));
          cdoDefVlist(streamID2, vlistID2);
        }

      const auto ntsteps = vlistNtsteps(vlistID1);

      int tsID1 = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID1)))
        {
          const double fstatus = (ntsteps > 1) ? indf + (tsID1 + 1.) / ntsteps : indf + 1.;
          if (!Options::cdoVerbose) Progress::update(0, 1, fstatus / nfiles);

          taxisCopyTimestep(taxisID2, taxisID1);
          cdoDefTimestep(streamID2, tsID2);

          for (int recID = 0; recID < nrecs; recID++)
            {
              if (lcopy && (operatorID == SELALL || operatorID == SZIP))
                {
                  cdoInqRecord(streamID1, &varID, &levelID);

                  if (lconstvars && tsID2 > 0 && tsID1 == 0)
                    if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

                  cdoDefRecord(streamID2, varID, levelID);
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  if (Options::cdoParIO)
                    {
                      parIO.recID = recID;
                      parIO.nrecs = nrecs;
                      /* fprintf(stderr, "in1 streamID %d varID %d levelID
                       * %d\n", streamID1, varID, levelID);*/
                      parReadRecord(streamID1, &varID, &levelID, (double *) array, &nmiss, &parIO);
                      /* fprintf(stderr, "in2 streamID %d varID %d levelID
                       * %d\n", streamID1, varID, levelID);*/
                    }
                  else
                    {
                      cdoInqRecord(streamID1, &varID, &levelID);

                      if (lconstvars && tsID2 > 0 && tsID1 == 0)
                        if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

                      if (Options::CDO_Memtype == MEMTYPE_FLOAT)
                        cdoReadRecordF(streamID1, (float *) array, &nmiss);
                      else
                        cdoReadRecord(streamID1, (double *) array, &nmiss);
                    }
                  /*
                  if ( Options::cdoParIO )
                    fprintf(stderr, "out1 %d %d %d\n", streamID2,  varID,
                  levelID);
                  */
                  cdoDefRecord(streamID2, varID, levelID);
                  if (Options::CDO_Memtype == MEMTYPE_FLOAT)
                    cdoWriteRecordF(streamID2, (float *) array, nmiss);
                  else
                    cdoWriteRecord(streamID2, (double *) array, nmiss);
                  /*
                  if ( Options::cdoParIO )
                    fprintf(stderr, "out2 %d %d %d\n", streamID2,  varID,
                  levelID);
                  */
                }
            }

          tsID1++;
          tsID2++;
        }

      cdoStreamClose(streamID1);
    }

  cdoStreamClose(streamID2);

  if (array) Free(array);
  if (vlistID2 != CDI_UNDEFID) vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
