/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Cond       ifthen          If then
      Cond       ifnotthen       If not then
*/

#include <cdi.h>


#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_fill.h"

void *
Cond(void *process)
{
  enum
  {
    FILL_NONE,
    FILL_TS,
    FILL_REC
  };
  int filltype = FILL_NONE;
  int nrecs, nrecs2;
  int varID, levelID;
  size_t nmiss1 = 0, nmiss2;
  double missval1 = -9.E33;
  std::vector<std::vector<size_t>> varnmiss1;
  std::vector<std::vector<double>> vardata1;

  cdoInitialize(process);

  // clang-format off
  const auto IFTHEN    = cdoOperatorAdd("ifthen",    0, 0, nullptr);
  const auto IFNOTTHEN = cdoOperatorAdd("ifnotthen", 0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = vlistDuplicate(vlistID2);

  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = taxisDuplicate(taxisID2);
  vlistDefTaxis(vlistID3, taxisID3);

  auto ntsteps1 = vlistNtsteps(vlistID1);
  auto ntsteps2 = vlistNtsteps(vlistID2);
  if (ntsteps1 == 0) ntsteps1 = 1;
  if (ntsteps2 == 0) ntsteps2 = 1;

  if (vlistNrecs(vlistID1) == 1 && vlistNrecs(vlistID2) != 1)
    {
      filltype = FILL_REC;
      cdoPrint("Filling up stream1 >%s< by copying the first record.", cdoGetStreamName(0));
    }

  if (filltype == FILL_NONE) vlistCompare(vlistID1, vlistID2, CMP_DIM);

  nospec(vlistID1);
  nospec(vlistID2);

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  const auto streamID3 = cdoOpenWrite(2);
  cdoDefVlist(streamID3, vlistID3);

  const auto gridsizemax = vlistGridsizeMax(vlistID2);

  if (filltype == FILL_REC && gridsizemax != gridInqSize(vlistGrid(vlistID1, 0)))
    cdoAbort("Stream1 >%s< has wrong gridsize!", cdoGetStreamName(0));

  std::vector<double> array1(gridsizemax);
  std::vector<double> array2(gridsizemax);
  std::vector<double> array3(gridsizemax);

  if (Options::cdoVerbose) cdoPrint("Number of timesteps: file1 %d, file2 %d", ntsteps1, ntsteps2);

  if (filltype == FILL_NONE)
    {
      if (ntsteps1 == 1 && ntsteps2 != 1)
        {
          filltype = FILL_TS;
          cdoPrint("Filling up stream1 >%s< by copying the first timestep.", cdoGetStreamName(0));

          cdoFillTs(vlistID1, vardata1, varnmiss1);
        }
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      if (tsID == 0 || filltype == FILL_NONE)
        {
          nrecs2 = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs2 == 0) cdoAbort("Input streams have different number of timesteps!");
        }

      taxisCopyTimestep(taxisID3, taxisID2);
      cdoDefTimestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, &array2[0], &nmiss2);

          if (tsID == 0 || filltype == FILL_NONE)
            {
              if (recID == 0 || filltype != FILL_REC)
                {
                  cdoInqRecord(streamID1, &varID, &levelID);
                  cdoReadRecord(streamID1, &array1[0], &nmiss1);
                }

              if (filltype == FILL_TS)
                {
                  const auto offset = varList1[varID].gridsize * levelID;
                  arrayCopy(varList1[varID].gridsize, &array1[0], &vardata1[varID][offset]);
                  varnmiss1[varID][levelID] = nmiss1;
                }
            }
          else if (filltype == FILL_TS)
            {
              const auto offset = varList1[varID].gridsize * levelID;
              arrayCopy(varList1[varID].gridsize, &vardata1[varID][offset], &array1[0]);
              nmiss1 = varnmiss1[varID][levelID];
            }

          const auto gridsize = varList2[varID].gridsize;
          const auto missval2 = varList2[varID].missval;
          if (recID == 0 || filltype != FILL_REC) missval1 = varList1[varID].missval;

          if (operatorID == IFTHEN)
            {
              for (size_t i = 0; i < gridsize; i++)
                array3[i] = !DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array1[i], 0.) ? array2[i] : missval2;
            }
          else if (operatorID == IFNOTTHEN)
            {
              for (size_t i = 0; i < gridsize; i++)
                array3[i] = !DBL_IS_EQUAL(array1[i], missval1) && DBL_IS_EQUAL(array1[i], 0.) ? array2[i] : missval2;
            }
          else
            {
              cdoAbort("Operator not implemented!");
            }

          const auto nmiss3 = arrayNumMV(gridsize, &array3[0], missval2);
          cdoDefRecord(streamID3, varID, levelID);
          cdoWriteRecord(streamID3, &array3[0], nmiss3);
        }

      tsID++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
