/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>


#include "cdo_options.h"
#include "param_conversion.h"
#include "process_int.h"
#include "timer.h"
#include "util_files.h"

const char *
filetypestr(int filetype)
{
  switch (filetype)
    {
    // clang-format off
    case CDI_FILETYPE_GRB:  return ("GRIB");
    case CDI_FILETYPE_GRB2: return ("GRIB2");
    case CDI_FILETYPE_NC:   return ("NetCDF");
    case CDI_FILETYPE_NC2:  return ("NetCDF2");
    case CDI_FILETYPE_NC4:  return ("NetCDF4");
    case CDI_FILETYPE_NC4C: return ("NetCDF4 classic");
    case CDI_FILETYPE_NC5:  return ("NetCDF5");
    case CDI_FILETYPE_SRV:  return ("SERVICE");
    case CDI_FILETYPE_EXT:  return ("EXTRA");
    case CDI_FILETYPE_IEG:  return ("IEG");
    default:                return ("");
      // clang-format on
    }
}

const char *
datatypestr(int datatype)
{
  static char str[20];
  str[0] = 0;
  snprintf(str, sizeof(str), "%d bit packed", datatype);

  // clang-format off
  if      ( datatype == CDI_DATATYPE_PACK   ) return ("P0");
  else if ( datatype > 0 && datatype <= 32  ) return (str);
  else if ( datatype == CDI_DATATYPE_CPX32  ) return ("C32");
  else if ( datatype == CDI_DATATYPE_CPX64  ) return ("C64");
  else if ( datatype == CDI_DATATYPE_FLT32  ) return ("32 bit floats");
  else if ( datatype == CDI_DATATYPE_FLT64  ) return ("64 bit floats");
  else if ( datatype == CDI_DATATYPE_INT8   ) return ("I8");
  else if ( datatype == CDI_DATATYPE_INT16  ) return ("I16");
  else if ( datatype == CDI_DATATYPE_INT32  ) return ("I32");
  else if ( datatype == CDI_DATATYPE_UINT8  ) return ("U8");
  else if ( datatype == CDI_DATATYPE_UINT16 ) return ("U16");
  else if ( datatype == CDI_DATATYPE_UINT32 ) return ("U32");
  else                                        return ("");
  // clang-format on
}

static void
print_stat(const char *sinfo, int memtype, int datatype, int filetype, off_t nvalues, double data_size, double file_size, double tw)
{
  nvalues /= 1000000;
  data_size /= 1024. * 1024. * 1024.;

  cdoPrint("%s Read %.1f GB of %d bit floats from %s %s, %.1f MVal/s", sinfo, data_size, memtype == MEMTYPE_FLOAT ? 32 : 64,
           datatypestr(datatype), filetypestr(filetype), tw > 0 ? nvalues / tw : -1);

  file_size /= 1024. * 1024. * 1024.;
  cdoPrint("%s Read %.1f GB in %.1f seconds, total %.1f MB/s", sinfo, file_size, tw, tw > 0 ? 1024 * file_size / tw : -1);
}

void *
CDIread(void *process)
{
  int memtype = Options::CDO_Memtype;
  int varID, levelID;
  size_t nmiss;
  int nrecs;
  int filetype = -1, datatype = -1;
  char sinfo[64];
  off_t nvalues = 0;
  double file_size = 0, data_size = 0;
  double twsum = 0;

  sinfo[0] = 0;

  cdoInitialize(process);

  if (Options::cdoVerbose) cdoPrint("parameter: <nruns>");

  if (operatorArgc() > 1) cdoAbort("Too many arguments!");

  auto nruns = (operatorArgc() == 1) ? parameter2int(operatorArgv()[0]) : 1;
  if (nruns < 0) nruns = 0;
  if (nruns > 99) nruns = 99;

  if (Options::cdoVerbose) cdoPrint("nruns      : %d", nruns);

  // vlistDefNtsteps(vlistID, 1);

  for (int irun = 0; irun < nruns; ++irun)
    {
      auto tw0 = timer_val(timer_read);
      data_size = 0;
      nvalues = 0;

      const auto streamID = cdoOpenRead(0);
      const auto vlistID = cdoStreamInqVlist(streamID);

      filetype = cdoInqFiletype(streamID);
      datatype = vlistInqVarDatatype(vlistID, 0);

      auto gridsize = vlistGridsizeMax(vlistID);

      std::vector<float> farray;
      std::vector<double> darray;
      if (memtype == MEMTYPE_FLOAT)
        farray.resize(gridsize);
      else
        darray.resize(gridsize);

      auto t0 = timer_val(timer_read);

      int tsID = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID, tsID)))
        {

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID, &varID, &levelID);

              gridsize = gridInqSize(vlistInqVarGrid(vlistID, varID));
              nvalues += gridsize;

              if (memtype == MEMTYPE_FLOAT)
                {
                  cdoReadRecordF(streamID, farray.data(), &nmiss);
                  //  for ( size_t i = 0; i < gridsize; ++i ) darray[i] = farray[i];
                  data_size += gridsize * 4;
                }
              else
                {
                  cdoReadRecord(streamID, darray.data(), &nmiss);
                  data_size += gridsize * 8;
                }
            }

          if (Options::cdoVerbose)
            {
              const auto tw = timer_val(timer_read) - t0;
              t0 = timer_val(timer_read);
              cdoPrint("Timestep %d: %.2f seconds", tsID + 1, tw);
            }

          tsID++;
        }

      cdoStreamClose(streamID);

      const auto tw = timer_val(timer_read) - tw0;
      twsum += tw;

      file_size = (double) fileSize(cdoGetStreamName(0));

      if (nruns > 1) snprintf(sinfo, sizeof(sinfo), "(run %d)", irun + 1);

      print_stat(sinfo, memtype, datatype, filetype, nvalues, data_size, file_size, tw);
    }

  if (nruns > 1) print_stat("(mean)", memtype, datatype, filetype, nvalues, data_size, file_size, twsum / nruns);

  cdoFinish();

  return nullptr;
}
