/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Arith      add             Add two fields
      Arith      sub             Subtract two fields
      Arith      mul             Multiply two fields
      Arith      div             Divide two fields
      Arith      min             Minimum of two fields
      Arith      max             Maximum of two fields
      Arith      atan2           Arc tangent of two fields
*/

#include <cdi.h>
#include "cdo_options.h"


#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_options.h"
#include "cdo_fill.h"


static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("add",     func_add,     1, nullptr);
  cdoOperatorAdd("sub",     func_sub,     1, nullptr);
  cdoOperatorAdd("mul",     func_mul,     1, nullptr);
  cdoOperatorAdd("div",     func_div,     1, nullptr);
  cdoOperatorAdd("min",     func_min,     0, nullptr);
  cdoOperatorAdd("max",     func_max,     0, nullptr);
  cdoOperatorAdd("atan2",   func_atan2,   0, nullptr);
  cdoOperatorAdd("setmiss", func_setmiss, 0, nullptr);
  // clang-format on
}

void *
Arith(void *process)
{
  enum
  {
    FILL_NONE,
    FILL_TS,
    FILL_VAR,
    FILL_VARTS,
    FILL_FILE
  };
  int filltype = FILL_NONE;
  int varID, levelID;
  int nlevels2 = 1;
  int levelID2 = -1;
  std::vector<std::vector<size_t>> varnmiss;
  std::vector<std::vector<double>> vardata;
  std::vector<size_t> varnmiss2;
  std::vector<double> vardata2;

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);
  const bool opercplx = cdoOperatorF2(operatorID);
  operatorCheckArgc(0);

  auto streamID1 = cdoOpenRead(0);
  auto streamID2 = cdoOpenRead(1);

  auto streamID2x = streamID2;

  Field field1, field2;
  Field *fieldx1 = &field1;
  Field *fieldx2 = &field2;

  auto vlistID1 = cdoStreamInqVlist(streamID1);
  auto vlistID2 = cdoStreamInqVlist(streamID2);
  auto vlistID1x = vlistID1;

  if (Options::cdoVerbose) vlistPrint(vlistID1);
  if (Options::cdoVerbose) vlistPrint(vlistID2);

  auto taxisID1 = vlistInqTaxis(vlistID1);
  auto taxisID2 = vlistInqTaxis(vlistID2);

  auto ntsteps1 = vlistNtsteps(vlistID1);
  auto ntsteps2 = vlistNtsteps(vlistID2);
  if (ntsteps1 == 0) ntsteps1 = 1;
  if (ntsteps2 == 0) ntsteps2 = 1;

  const auto nvars1 = vlistNvars(vlistID1);
  const auto nvars2 = vlistNvars(vlistID2);

  bool lfill1 = false, lfill2 = false, lfill1x = false;
  if (nvars1 == 1 && nvars2 == 1)
    {
      lfill2 = vlistNrecs(vlistID1) != 1 && vlistNrecs(vlistID2) == 1;
      lfill1 = vlistNrecs(vlistID1) == 1 && vlistNrecs(vlistID2) != 1;
      if (lfill1 && ntsteps1 != 1 && ntsteps2 == 1)
        {
          lfill1 = false;
          lfill2 = true;
          lfill1x = true;
        }
    }
  else
    {
      lfill2 = nvars1 != 1 && nvars2 == 1;
      lfill1 = nvars1 == 1 && nvars2 != 1;
    }

  if (lfill1x)
    {
      nlevels2 = vlistCompareX(vlistID2, vlistID1, CMP_DIM);

      filltype = FILL_NONE;
      cdoPrint("Filling up stream1 >%s< by copying the first variable of each timestep.", cdoGetStreamName(0));
    }
  else if (lfill2)
    {
      nlevels2 = vlistCompareX(vlistID1, vlistID2, CMP_DIM);

      if (ntsteps1 != 1 && ntsteps2 == 1)
        {
          filltype = FILL_VAR;
          cdoPrint("Filling up stream2 >%s< by copying the first variable.", cdoGetStreamName(1));
        }
      else
        {
          filltype = FILL_VARTS;
          cdoPrint("Filling up stream2 >%s< by copying the first variable of each timestep.", cdoGetStreamName(1));
        }
    }
  else if (lfill1)
    {
      nlevels2 = vlistCompareX(vlistID2, vlistID1, CMP_DIM);

      if (ntsteps1 == 1 && ntsteps2 != 1)
        {
          filltype = FILL_VAR;
          cdoPrint("Filling up stream1 >%s< by copying the first variable.", cdoGetStreamName(0));
        }
      else
        {
          filltype = FILL_VARTS;
          cdoPrint("Filling up stream1 >%s< by copying the first variable of each timestep.", cdoGetStreamName(0));
        }

      std::swap(streamID1, streamID2);
      std::swap(vlistID1, vlistID2);
      std::swap(taxisID1, taxisID2);
      fieldx1 = &field2;
      fieldx2 = &field1;
    }

  if (lfill1x == false && filltype == FILL_NONE) vlistCompare(vlistID1, vlistID2, CMP_ALL);

  const size_t nwpv = (vlistNumber(vlistID1) == CDI_COMP && vlistNumber(vlistID2) == CDI_COMP) ? 2 : 1;
  if (nwpv == 2 && !opercplx) cdoAbort("Fields with complex numbers are not supported by this operator!");
  const auto gridsizemax = nwpv * vlistGridsizeMax(vlistID1);

  field1.resize(gridsizemax);
  field2.resize(gridsizemax);

  if (lfill1x || filltype == FILL_VAR || filltype == FILL_VARTS)
    {
      vardata2.resize(gridsizemax * nlevels2);
      varnmiss2.resize(nlevels2);
    }

  if (Options::cdoVerbose) cdoPrint("Number of timesteps: file1 %d, file2 %d", ntsteps1, ntsteps2);

  if (filltype == FILL_NONE)
    {
      if (ntsteps1 != 1 && ntsteps2 == 1)
        {
          filltype = FILL_TS;
          cdoPrint("Filling up stream2 >%s< by copying the first timestep.", cdoGetStreamName(1));
        }
      else if (ntsteps1 == 1 && ntsteps2 != 1)
        {
          filltype = FILL_TS;
          cdoPrint("Filling up stream1 >%s< by copying the first timestep.", cdoGetStreamName(0));

          std::swap(streamID1, streamID2);
          std::swap(vlistID1, vlistID2);
          std::swap(taxisID1, taxisID2);
          fieldx1 = &field2;
          fieldx2 = &field1;
        }

      if (filltype == FILL_TS)
        {
          cdoFillTs(vlistID2, vardata, varnmiss);
        }
    }

  const auto vlistID3 = vlistDuplicate(vlistID1);
  if (filltype == FILL_TS && vlistID1x != vlistID1)
    {
      const auto nvars = vlistNvars(vlistID1);
      for (varID = 0; varID < nvars; varID++) vlistDefVarMissval(vlistID3, varID, vlistInqVarMissval(vlistID1, varID));
    }

  if (lfill1x)
    {
      const auto zaxisID2 = vlistZaxis(vlistID2, 0);
      vlistChangeZaxisIndex(vlistID3, 0, zaxisID2);
    }

  const auto taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID3, taxisID3);

  const auto streamID3 = cdoOpenWrite(2);
  cdoDefVlist(streamID3, vlistID3);

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  int nrecs, nrecs2 = 0;
  int tsID = 0;
  int tsID2 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      if (tsID == 0 || filltype == FILL_NONE || filltype == FILL_FILE || filltype == FILL_VARTS)
        {
          nrecs2 = cdoStreamInqTimestep(streamID2, tsID2);
          if (nrecs2 == 0)
            {
              if (filltype == FILL_NONE && streamID2x == streamID2)
                {
                  filltype = FILL_FILE;
                  cdoPrint("Filling up stream2 >%s< by copying all timesteps.", cdoGetStreamName(1));
                }

              if (filltype == FILL_FILE)
                {
                  tsID2 = 0;
                  cdoStreamClose(streamID2);
                  streamID2 = cdoOpenRead(1);
                  streamID2x = streamID2;

                  vlistID2 = cdoStreamInqVlist(streamID2);

                  vlistCompare(vlistID1, vlistID2, CMP_DIM);

                  nrecs2 = cdoStreamInqTimestep(streamID2, tsID2);
                  if (nrecs2 == 0) cdoAbort("Empty input stream %s!", cdoGetStreamName(1));
                }
              else
                cdoAbort("Input streams have different number of timesteps!");
            }
        }

      taxisCopyTimestep(taxisID3, taxisID1);
      cdoDefTimestep(streamID3, tsID);

      const auto numrecs = lfill1x ? nrecs2 : nrecs;
      for (int recID = 0; recID < numrecs; recID++)
        {
          bool lread1 = true;
          if (lfill1x && recID > 0) lread1 = false;
          if (lread1)
            {
              cdoInqRecord(streamID1, &varID, &levelID);
              cdoReadRecord(streamID1, fieldx1->vec.data(), &fieldx1->nmiss);

              if (lfill1x)
                {
                  const auto gridsize = nwpv * varList1[varID].gridsize;
                  arrayCopy(gridsize, fieldx1->vec.data(), &vardata2[0]);
                  varnmiss2[0] = fieldx1->nmiss;
                }
            }

          if (lfill1x) levelID = recID;

          auto varID2 = varID;

          if (tsID == 0 || filltype == FILL_NONE || filltype == FILL_FILE || filltype == FILL_VARTS)
            {
              const bool lstatus = nlevels2 > 1 ? varID == 0 : recID == 0;
              if (lstatus || (filltype != FILL_VAR && filltype != FILL_VARTS))
                {
                  cdoInqRecord(streamID2, &varID2, &levelID2);
                  cdoReadRecord(streamID2, fieldx2->vec.data(), &fieldx2->nmiss);
                  if (varID != varID2) cdoAbort("Internal error, varIDs of input streams differ!");
                  if (lfill1x == false && levelID != levelID2) cdoAbort("Internal error, levelIDs of input streams differ!");
                }

              if (filltype == FILL_TS)
                {
                  const auto gridsize = nwpv * varList2[varID].gridsize;
                  const auto offset = gridsize * levelID;
                  arrayCopy(gridsize, fieldx2->vec.data(), &vardata[varID][offset]);
                  varnmiss[varID][levelID] = fieldx2->nmiss;
                }
              else if (lstatus && (filltype == FILL_VAR || filltype == FILL_VARTS))
                {
                  const auto gridsize = nwpv * varList2[0].gridsize;
                  const auto offset = gridsize * levelID2;
                  arrayCopy(gridsize, fieldx2->vec.data(), &vardata2[offset]);
                  varnmiss2[levelID2] = fieldx2->nmiss;
                }
            }
          else if (filltype == FILL_TS)
            {
              const auto gridsize = nwpv * varList2[varID2].gridsize;
              const auto offset = gridsize * levelID;
              arrayCopy(gridsize, &vardata[varID][offset], fieldx2->vec.data());
              fieldx2->nmiss = varnmiss[varID][levelID];
            }

          if (lfill1x)
            {
              const auto gridsize = nwpv * varList1[0].gridsize;
              arrayCopy(gridsize, &vardata2[0], fieldx1->vec.data());
              fieldx1->nmiss = varnmiss2[0];
            }

          fieldx1->grid = varList1[varID].gridID;
          fieldx1->missval = varList1[varID].missval;

          if (filltype == FILL_VAR || filltype == FILL_VARTS)
            {
              levelID2 = (nlevels2 > 1) ? levelID : 0;
              const auto gridsize = nwpv * varList2[0].gridsize;
              const auto offset = gridsize * levelID2;
              arrayCopy(gridsize, &vardata2[offset], fieldx2->vec.data());
              fieldx2->nmiss = varnmiss2[levelID2];
              fieldx2->grid = varList2[0].gridID;
              fieldx2->missval = varList2[0].missval;
            }
          else
            {
              fieldx2->grid = varList2[varID2].gridID;
              fieldx2->missval = varList2[varID2].missval;
            }

          if (nwpv == 2)
            vfarfuncplx(field1, field2, operfunc);
          else
            vfarfun(field1, field2, operfunc);

          cdoDefRecord(streamID3, varID, levelID);
          cdoWriteRecord(streamID3, field1.vec.data(), field1.nmiss);
        }

      tsID++;
      tsID2++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID3);

  cdoFinish();

  return nullptr;
}
