/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ydrunpctl    ydrunpctl         Multi-year daily running percentiles
*/

#include <cdi.h>

#include "cdo_options.h"
#include "cdo_vlist.h"
#include "datetime.h"
#include "functs.h"
#include "process_int.h"
#include "param_conversion.h"
#include "calendar.h"
#include "percentiles_hist.h"
#include "percentiles.h"


void *
Ydrunpctl(void *process)
{
  constexpr int MaxDays = 373;
  int varID;
  int nrecs;
  int levelID;
  int inp, its;
  size_t nmiss;
  int64_t vdates1[MaxDays], vdates2[MaxDays];
  int vtimes1[MaxDays] /*, vtimes2[MaxDays]*/;
  int nsets[MaxDays];
  FieldVector2D vars2[MaxDays];
  HistogramSet hsets[MaxDays];

  cdoInitialize(process);
  cdoOperatorAdd("ydrunpctl", func_pctl, 0, nullptr);

  operatorInputArg("percentile number, number of timesteps");
  operatorCheckArgc(2);
  const auto pn = parameter2double(cdoOperatorArgv(0));
  percentile_check_number(pn);
  const auto ndates = parameter2int(cdoOperatorArgv(1));

  for (int dayoy = 0; dayoy < MaxDays; dayoy++) nsets[dayoy] = 0;

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);
  const auto streamID3 = cdoOpenRead(2);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = cdoStreamInqVlist(streamID3);
  const auto vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = vlistInqTaxis(vlistID3);
  /* TODO - check that time axes 2 and 3 are equal */

  const auto taxisID4 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID4)) taxisDeleteBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  const auto dpy = calendar_dpy(taxisInqCalendar(taxisID1));

  const auto streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  std::vector<CdoDateTime> datetime(ndates + 1);

  FieldVector3D vars1(ndates + 1);
  for (its = 0; its < ndates; its++) fieldsFromVlist(vlistID1, vars1[its], FIELD_VEC);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      if (nrecs != cdoStreamInqTimestep(streamID3, tsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const auto vdate = taxisInqVdate(taxisID2);
      const auto vtime = taxisInqVtime(taxisID2);

      if (vdate != taxisInqVdate(taxisID3))
        cdoAbort("Verification dates at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      const auto dayoy = decodeDayOfYear(vdate);
      if (dayoy < 0 || dayoy >= MaxDays) cdoAbort("Day %d out of range!", dayoy);

      vdates2[dayoy] = vdate;
      // vtimes2[dayoy] = vtime;

      if (!vars2[dayoy].size())
        {
          fieldsFromVlist(vlistID2, vars2[dayoy], FIELD_VEC);
          hsets[dayoy].create(nvars);

          for (varID = 0; varID < nvars; varID++)
            {
              auto gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID));
              auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
              hsets[dayoy].createVarLevels(varID, nlevels, gridsize);
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars2[dayoy][varID][levelID].vec.data(), &nmiss);
          vars2[dayoy][varID][levelID].nmiss = nmiss;
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, field.vec.data(), &field.nmiss);
          field.grid = vars2[dayoy][varID][levelID].grid;
          field.missval = vars2[dayoy][varID][levelID].missval;

          hsets[dayoy].defVarLevelBounds(varID, levelID, vars2[dayoy][varID][levelID], field);
        }

      tsID++;
    }

  for (tsID = 0; tsID < ndates; tsID++)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) cdoAbort("File has less then %d timesteps!", ndates);

      datetime[tsID].date = taxisInqVdate(taxisID1);
      datetime[tsID].time = taxisInqVtime(taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          cdoReadRecord(streamID1, vars1[tsID][varID][levelID].vec.data(), &nmiss);
          vars1[tsID][varID][levelID].nmiss = nmiss;
        }
    }

  while (true)
    {
      datetime_avg(dpy, ndates, datetime.data());

      const auto vdate = datetime[ndates].date;
      const auto vtime = datetime[ndates].time;

      const auto dayoy = decodeDayOfYear(vdate);
      if (dayoy < 0 || dayoy >= MaxDays) cdoAbort("Day %d out of range!", dayoy);

      vdates1[dayoy] = vdate;
      vtimes1[dayoy] = vtime;

      if (!vars2[dayoy].size()) cdoAbort("No data for day %d in %s and %s", dayoy, cdoGetStreamName(1), cdoGetStreamName(2));

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

          const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
          for (levelID = 0; levelID < nlevels; levelID++)
            for (inp = 0; inp < ndates; inp++) hsets[dayoy].addVarLevelValues(varID, levelID, vars1[inp][varID][levelID]);
        }

      datetime[ndates] = datetime[0];
      vars1[ndates] = vars1[0];

      for (inp = 0; inp < ndates; inp++)
        {
          datetime[inp] = datetime[inp + 1];
          vars1[inp] = vars1[inp + 1];
        }

      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      datetime[ndates - 1].date = taxisInqVdate(taxisID1);
      datetime[ndates - 1].time = taxisInqVtime(taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          cdoReadRecord(streamID1, vars1[ndates - 1][varID][levelID].vec.data(), &nmiss);
          vars1[ndates - 1][varID][levelID].nmiss = nmiss;
        }

      nsets[dayoy] += ndates;
      tsID++;
    }
  /*
  int outyear = 1e9;
  for ( dayoy = 0; dayoy < MaxDays; dayoy++ )
    if ( nsets[dayoy] )
      {
        int year, month, day;
        cdiDecodeDate(vdates1[dayoy], &year, &month, &day);
        if ( year < outyear ) outyear = year;
      }

  for ( dayoy = 0; dayoy < MaxDays; dayoy++ )
    if ( nsets[dayoy] )
      {
        int year, month, day;
        cdiDecodeDate(vdates1[dayoy], &year, &month, &day);
        vdates1[dayoy] = cdiEncodeDate(outyear, month, day);
      }
  */
  int otsID = 0;
  for (int dayoy = 0; dayoy < MaxDays; dayoy++)
    if (nsets[dayoy])
      {
        if (decodeMonthAndDay(vdates1[dayoy]) != decodeMonthAndDay(vdates2[dayoy]))
          cdoAbort("Verification dates for day %d of %s, %s and %s are different!", dayoy, cdoGetStreamName(0),
                   cdoGetStreamName(1));

        for (varID = 0; varID < nvars; varID++)
          {
            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

            const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
            for (levelID = 0; levelID < nlevels; levelID++)
              hsets[dayoy].getVarLevelPercentiles(vars2[dayoy][varID][levelID], varID, levelID, pn);
          }

        taxisDefVdate(taxisID4, vdates1[dayoy]);
        taxisDefVtime(taxisID4, vtimes1[dayoy]);
        cdoDefTimestep(streamID4, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            cdoDefRecord(streamID4, varID, levelID);
            cdoWriteRecord(streamID4, vars2[dayoy][varID][levelID].vec.data(), vars2[dayoy][varID][levelID].nmiss);
          }

        otsID++;
      }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
