/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "process_int.h"
#include "printinfo.h"
#include "cdo_zaxis.h"


void *
Pinfo(void *process)
{
  int varID;
  int nrecs;
  int levelID;
  size_t nmiss, imiss = 0;
  int ivals = 0;
  char varname[CDI_MAX_NAME];
  double level;
  double arrmin, arrmax, arrmean;

  cdoInitialize(process);

  // clang-format off
  const auto PINFO  = cdoOperatorAdd("pinfo",  0, 0, nullptr);
  const auto PINFOV = cdoOperatorAdd("pinfov", 0, 0, nullptr);
  // clang-format on

  (void)(PINFO); //CDO_UNUSED

  const auto operatorID = cdoOperatorID();

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const auto gridsize = vlistGridsizeMax(vlistID1);

  Varray<double> array1(gridsize);
  Varray<double> array2(gridsize);

  int indg = 0;
  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);
      const auto vdateString = dateToString(vdate);
      const auto vtimeString = timeToString(vtime);

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          if (tsID == 0 && recID == 0)
            {
              if (operatorID == PINFOV)
                fprintf(stdout,
                        "   Rec :       Date  Time    Varname     Level    Size    Miss :     Minimum        Mean     Maximum\n");
              else
                fprintf(stdout, "   Rec :       Date  Time    Code  Level    Size    Miss :     Minimum        Mean     Maximum\n");
            }

          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, array1.data(), &nmiss);

          indg += 1;
          const auto code = vlistInqVarCode(vlistID1, varID);
          const auto gridID = vlistInqVarGrid(vlistID1, varID);
          const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
          auto gridsize = gridInqSize(gridID);
          const auto missval = vlistInqVarMissval(vlistID1, varID);

          if (operatorID == PINFOV) vlistInqVarName(vlistID1, varID, varname);

          if (operatorID == PINFOV)
            fprintf(stdout, "%6d :%s %s %-8s ", indg, vdateString.c_str(), vtimeString.c_str(), varname);
          else
            fprintf(stdout, "%6d :%s %s %3d", indg, vdateString.c_str(), vtimeString.c_str(), code);

          level = cdoZaxisInqLevel(zaxisID, levelID);
          fprintf(stdout, " %7g ", level);

          fprintf(stdout, "%7zu %7zu :", gridsize, nmiss);

          if (gridInqType(gridID) == GRID_SPECTRAL || (gridsize == 1 && nmiss == 0))
            {
              fprintf(stdout, "            %#12.5g\n", array1[0]);
            }
          else
            {
              if (nmiss)
                {
                  ivals = varrayMinMaxMeanMV(gridsize, array1, missval, arrmin, arrmax, arrmean);
                  imiss = gridsize - ivals;
                  gridsize = ivals;
                }
              else
                {
                  varrayMinMaxMean(gridsize, array1, arrmin, arrmax, arrmean);
                }

              if (gridsize)
                {
                  fprintf(stdout, "%#12.5g%#12.5g%#12.5g\n", arrmin, arrmean, arrmax);
                }
              else
                {
                  fprintf(stdout, "                     nan\n");
                }

              if (imiss != nmiss && nmiss) fprintf(stdout, "Found %zu of %zu missing values!\n", imiss, nmiss);
            }

          varrayCopy(gridsize, array1, array2);

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, array2.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
