/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "process_int.h"
#include "cdo_cdiWrapper.h"


void *
Importfv3grid(void *process)
{
  cdoInitialize(process);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto nvars = vlistNvars(vlistID1);
  if (nvars != 5) cdoAbort("Found %d variables, expected 5 variables!", nvars);

  std::vector<std::string> vars(nvars);
  vars[0] = "grid_lon";
  vars[1] = "grid_lat";
  vars[2] = "grid_lont";
  vars[3] = "grid_latt";
  vars[4] = "area";

  char varname[CDI_MAX_NAME];
  for (int varID = 0; varID < nvars; ++varID)
    {
      vlistInqVarName(vlistID1, varID, varname);
      if (strcmp(varname, vars[varID].c_str()) != 0)
        cdoAbort("Found variable %s, expected variable %s!", varname, vars[varID].c_str());
    }

  const auto ngrids = vlistNgrids(vlistID1);
  if (ngrids != 2) cdoAbort("Found %d grids, expected 2 grids!", nvars);

  const auto gridIDi1 = vlistGrid(vlistID1, 0);
  const auto gridIDi2 = vlistGrid(vlistID1, 1);

  const auto nx = gridInqXsize(gridIDi1);
  const auto ny = gridInqYsize(gridIDi1);
  const auto gridsize1 = gridInqSize(gridIDi1);
  const auto gridsize2 = gridInqSize(gridIDi2);

  cdoStreamInqTimestep(streamID1, 0);

  const auto gridIDo = gridCreate(GRID_UNSTRUCTURED, gridsize2);
  gridDefNvertex(gridIDo, 4);

  Varray<double> buffer(gridsize1);
  int varID, levelID;
  size_t nmiss;

  {
    Varray<double> grid_corner(4 * gridsize2);

    cdoInqRecord(streamID1, &varID, &levelID);  // grid_lon
    cdoReadRecord(streamID1, buffer.data(), &nmiss);

    for (size_t j = 1; j < ny; ++j)
      for (size_t i = 1; i < nx; ++i)
        {
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 0] = buffer[j * nx + i];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 1] = buffer[j * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 2] = buffer[(j - 1) * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 3] = buffer[(j - 1) * nx + i];
        }

    gridDefXbounds(gridIDo, grid_corner.data());

    cdoInqRecord(streamID1, &varID, &levelID);  // grid_lat
    cdoReadRecord(streamID1, buffer.data(), &nmiss);

    for (size_t j = 1; j < ny; ++j)
      for (size_t i = 1; i < nx; ++i)
        {
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 0] = buffer[j * nx + i];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 1] = buffer[j * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 2] = buffer[(j - 1) * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 3] = buffer[(j - 1) * nx + i];
        }

    gridDefYbounds(gridIDo, grid_corner.data());
  }

  cdoInqRecord(streamID1, &varID, &levelID);  // grid_lont
  cdoReadRecord(streamID1, buffer.data(), &nmiss);
  gridDefXvals(gridIDo, buffer.data());

  cdoInqRecord(streamID1, &varID, &levelID);  // grid_latt
  cdoReadRecord(streamID1, buffer.data(), &nmiss);
  gridDefYvals(gridIDo, buffer.data());

  cdoInqRecord(streamID1, &varID, &levelID);  // area
  cdoReadRecord(streamID1, buffer.data(), &nmiss);

  double sfclevel = 0;
  const auto surfaceID = zaxisCreate(ZAXIS_SURFACE, 1);
  zaxisDefLevels(surfaceID, &sfclevel);

  const auto vlistID2 = vlistCreate();
  varID = vlistDefVar(vlistID2, gridIDo, surfaceID, TIME_CONSTANT);
  vlistDefVarName(vlistID2, varID, "area");
  vlistDefVarStdname(vlistID2, varID, "area");
  vlistDefVarLongname(vlistID2, varID, "cell area");
  vlistDefVarUnits(vlistID2, varID, "m2");
  vlistDefVarDatatype(vlistID2, varID, CDI_DATATYPE_FLT32);

  const auto taxisID = cdoTaxisCreate(TAXIS_ABSOLUTE);
  vlistDefTaxis(vlistID2, taxisID);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);
  cdoDefTimestep(streamID2, 0);
  cdoDefRecord(streamID2, 0, 0);
  cdoWriteRecord(streamID2, buffer.data(), 0);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);
  gridDestroy(gridIDo);
  zaxisDestroy(surfaceID);

  cdoFinish();

  return nullptr;
}
