/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/
#ifndef SPECSPACE_H
#define SPECSPACE_H

#include <vector>
#include "transform.h"

class SPTRANS
{
 public:
  long nlon;
  long nlat;
  long ntr;
  long poldim;
  long ifax[10];
  std::vector<double> vtrig;
  std::vector<double> vpoli;
  std::vector<double> vpold;
  std::vector<double> vpol2;    // only for uv2dv
  std::vector<double> vpol3;    // only for uv2dv
  std::vector<double> vcoslat;  // only for scaluv with uv2dv
  std::vector<double> vrcoslat; // only for scaluv with uv2dv

 SPTRANS() {}

 void init(long _nlon, long _nlat, long _ntr, int _flag)
  {
    nlon = _nlon;
    nlat = _nlat;
    ntr = _ntr;

    const long nsp = (ntr + 1) * (ntr + 2);

    /*
      gp2sp:  wrong result with > N1500 (e.g. N1600)
      gp2spl: wrong result with > N1000 (e.g. N1280)
     */
    constexpr long max_nsp = 4002000L;
    if (nsp > max_nsp) cdoAbort("Too many spectral coefficients %ld (max=%ld)\n", nsp, max_nsp);

    poldim = nsp / 2 * nlat;

    vtrig.resize(nlon);
    fft_set(vtrig.data(), ifax, nlon);

    vpoli.resize(poldim);
    vpold.resize(poldim);

    if (_flag) vpol2.resize(poldim);
    if (_flag) vpol3.resize(poldim);

    vcoslat.resize(nlat);
    vrcoslat.resize(nlat);

    if (_flag)
      after_legini_full(ntr, nlat, vpoli.data(), vpold.data(), nullptr, vpol2.data(), vpol3.data(), vcoslat.data());
    else
      after_legini(ntr, nlat, vpoli.data(), vpold.data(), vcoslat.data());

    for (long jgl = 0; jgl < nlat; ++jgl) vrcoslat[jgl] = 1.0 / vcoslat[jgl];
  }
};

class DVTRANS
{
 public:
  long ntr;
  long fdim;
  std::vector<double> f1;
  std::vector<double> f2;

  DVTRANS() {}

 void init(long _ntr)
  {
    ntr = _ntr;

    const long dimsp = (ntr + 1) * (ntr + 2);
    fdim = dimsp / 2;

    f1.resize(fdim);
    f2.resize(fdim);

    geninx(ntr, f1.data(), f2.data());
  }
};

void dv2ps(const double *div, double *pot, long nlev, long ntr);

void trans_uv2dv(SPTRANS &sptrans, long nlev, int gridID1, double *gu, double *gv, int gridID2, double *sd, double *svo);

void trans_dv2uv(SPTRANS &sptrans, DVTRANS &dvtrans, long nlev, int gridID1, double *sd, double *svo, int gridID2, double *gu,
                 double *gv);

void grid2spec(SPTRANS &sptrans, int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);
void spec2grid(SPTRANS &sptrans, int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);
void four2spec(SPTRANS &sptrans, int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);
void spec2four(SPTRANS &sptrans, int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);
void four2grid(SPTRANS &sptrans, int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);
void grid2four(SPTRANS &sptrans, int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);

void spec2spec(int gridIDin, double *arrayIn, int gridIDout, double *arrayOut);
void speccut(int gridIDin, double *arrayIn, double *arrayOut, int *waves);

void spcut(double *arrayIn, double *arrayOut, long ntr, const int *waves);

#endif
