/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Yseaspctl  yseaspctl       Multi-year seasonal percentiles
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"

#include "percentiles_hist.h"
#include "percentiles.h"
#include "cdo_season.h"

#define NSEAS 4

struct date_time_t
{
  int64_t vdate;
  int vtime;
};

void set_date(int64_t vdate_new, int vtime_new, date_time_t *datetime);

int getmonthday(int64_t date);

void *
Yseaspctl(void *process)
{
  int varID;
  int gridID;
  int year, month, day, seas;
  int nrecs;
  int levelID;
  size_t nmiss;
  int nlevels;
  long nsets[NSEAS];
  date_time_t datetime1[NSEAS], datetime2[NSEAS];
  FieldVector2D vars1[NSEAS];
  HISTOGRAM_SET *hsets[NSEAS];

  cdoInitialize(process);
  cdoOperatorAdd("yseaspctl", func_pctl, 0, nullptr);

  operatorInputArg("percentile number");
  const double pn = parameter2double(operatorArgv()[0]);
  percentile_check_number(pn);

  for (seas = 0; seas < NSEAS; seas++)
    {
      hsets[seas] = nullptr;
      nsets[seas] = 0;
      datetime1[seas].vdate = 0;
      datetime1[seas].vtime = 0;
      datetime2[seas].vdate = 0;
      datetime2[seas].vtime = 0;
    }

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);
  CdoStreamID streamID3 = cdoOpenRead(2);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = cdoStreamInqVlist(streamID3);
  const int vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = vlistInqTaxis(vlistID2);
  const int taxisID3 = vlistInqTaxis(vlistID3);
  /* TODO - check that time axes 2 and 3 are equal */

  const int taxisID4 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID4)) taxisDeleteBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  CdoStreamID streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const int nvars = vlistNvars(vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      if (nrecs != cdoStreamInqTimestep(streamID3, tsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const int64_t vdate = taxisInqVdate(taxisID2);
      const int vtime = taxisInqVtime(taxisID2);

      if (vdate != taxisInqVdate(taxisID3))
        cdoAbort("Verification dates at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      cdiDecodeDate(vdate, &year, &month, &day);

      const int seas = month_to_season(month);

      set_date(vdate, vtime, &datetime2[seas]);

      if (!vars1[seas].size())
        {
          fieldsFromVlist(vlistID1, vars1[seas], FIELD_VEC);

          hsets[seas] = hsetCreate(nvars);

          for (varID = 0; varID < nvars; varID++)
            {
              gridID = vlistInqVarGrid(vlistID1, varID);
              nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

              hsetCreateVarLevels(hsets[seas], varID, nlevels, gridID);
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars1[seas][varID][levelID].vec.data(), &nmiss);
          vars1[seas][varID][levelID].nmiss = nmiss;
        }
      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, field.vec.data(), &field.nmiss);
          field.grid = vars1[seas][varID][levelID].grid;
          field.missval = vars1[seas][varID][levelID].missval;

          hsetDefVarLevelBounds(hsets[seas], varID, levelID, &vars1[seas][varID][levelID], &field);
        }

      tsID++;
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const int64_t vdate = taxisInqVdate(taxisID1);
      const int vtime = taxisInqVtime(taxisID1);

      cdiDecodeDate(vdate, &year, &month, &day);
      if (month < 0 || month > 16) cdoAbort("Month %d out of range!", month);

      const int seas = month_to_season(month);

      set_date(vdate, vtime, &datetime1[seas]);

      if (!vars1[seas].size()) cdoAbort("No data for season %d in %s and %s", seas, cdoGetStreamName(1), cdoGetStreamName(2));

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          cdoReadRecord(streamID1, vars1[seas][varID][levelID].vec.data(), &nmiss);
          vars1[seas][varID][levelID].nmiss = nmiss;

          hsetAddVarLevelValues(hsets[seas], varID, levelID, &vars1[seas][varID][levelID]);
        }

      nsets[seas]++;
      tsID++;
    }

  int otsID = 0;
  for (int seas = 0; seas < NSEAS; seas++)
    if (nsets[seas])
      {
        if (getmonthday(datetime1[seas].vdate) != getmonthday(datetime2[seas].vdate))
          cdoAbort("Verification dates for the season %d of %s and %s are different!", seas, cdoGetStreamName(0),
                   cdoGetStreamName(1));

        for (varID = 0; varID < nvars; varID++)
          {
            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

            const int nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
            for (levelID = 0; levelID < nlevels; levelID++)
              hsetGetVarLevelPercentiles(&vars1[seas][varID][levelID], hsets[seas], varID, levelID, pn);
          }

        taxisDefVdate(taxisID4, datetime1[seas].vdate);
        taxisDefVtime(taxisID4, datetime1[seas].vtime);
        cdoDefTimestep(streamID4, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const int varID = recList[recID].varID;
            const int levelID = recList[recID].levelID;
            cdoDefRecord(streamID4, varID, levelID);
            cdoWriteRecord(streamID4, vars1[seas][varID][levelID].vec.data(), vars1[seas][varID][levelID].nmiss);
          }

        otsID++;
      }

  for (int seas = 0; seas < NSEAS; seas++)
    {
      if (vars1[seas].size())
        {
          hsetDestroy(hsets[seas]);
        }
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
