/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Wind       uv2dv           U and V wind to divergence and vorticity
      Wind       dv2uv           Divergence and vorticity to U and V wind
      Wind       dv2ps           Divergence and vorticity to velocity potential and stream function
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"
#include <mpim_grid.h>
#include "griddes.h"
#include "specspace.h"
#include "listarray.h"
#include "util_string.h"

void *
Wind(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nlev = 0;
  int gridIDsp = -1, gridIDgp = -1;
  int gridID1 = -1, gridID2 = -1;
  size_t nmiss;
  long ntr = -1;
  int varID1 = -1, varID2 = -1;
  SPTRANS sptrans;
  DVTRANS dvtrans;
  char varname[CDI_MAX_NAME];

  cdoInitialize(process);

  const bool lcopy = unchangedRecord();

  // clang-format off
  const int UV2DV  = cdoOperatorAdd("uv2dv",  0, 0, nullptr);
  const int UV2DVL = cdoOperatorAdd("uv2dvl", 0, 0, nullptr);
  const int DV2UV  = cdoOperatorAdd("dv2uv",  0, 0, nullptr);
  const int DV2UVL = cdoOperatorAdd("dv2uvl", 0, 0, nullptr);
  const int DV2PS  = cdoOperatorAdd("dv2ps",  0, 0, nullptr);

  const int operatorID = cdoOperatorID();

  const bool luv2dv = (operatorID == UV2DV || operatorID == UV2DVL);
  const bool ldv2uv = (operatorID == DV2UV || operatorID == DV2UVL);
  const bool linear = (operatorID == UV2DVL || operatorID == DV2UVL);

  int (*nlat2ntr)(int) = linear ? nlat_to_ntr_linear : nlat_to_ntr;
  const char *ctype = linear ? "l" : "";

  if ((luv2dv || ldv2uv) && operatorArgc() == 1)
    {
      const char *type = parameter2word(operatorArgv()[0]);
      if      (cstrIsEqual(type, "linear"))    { nlat2ntr = nlat_to_ntr_linear; ctype = "l"; }
      else if (cstrIsEqual(type, "cubic"))     { nlat2ntr = nlat_to_ntr_cubic; ctype = "c"; }
      else if (cstrIsEqual(type, "quadratic")) { nlat2ntr = nlat_to_ntr; }
      else cdoAbort("Unsupported type: %s\n", type);
    }
  // clang-format on

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  // find variables
  const int nvars = vlistNvars(vlistID2);
  for (varID = 0; varID < nvars; varID++)
    {
      const int param = vlistInqVarParam(vlistID2, varID);
      int pnum, pcat, pdis;
      cdiDecodeParam(param, &pnum, &pcat, &pdis);
      int code = pnum;
      if (operatorID == UV2DV || operatorID == UV2DVL)
        {
          // search for u and v wind
          if (pdis != 255 || code <= 0)
            {
              vlistInqVarName(vlistID1, varID, varname);
              cstrToLowerCase(varname);
              if (cstrIsEqual(varname, "u")) code = 131;
              if (cstrIsEqual(varname, "v")) code = 132;
            }

          if (code == 131) varID1 = varID;
          if (code == 132) varID2 = varID;
        }
      else if (operatorID == DV2UV || operatorID == DV2UVL || operatorID == DV2PS)
        {
          // search for divergence and vorticity
          if (pdis != 255)  // GRIB2
            {
              vlistInqVarName(vlistID1, varID, varname);
              cstrToLowerCase(varname);
              if (cstrIsEqual(varname, "d")) code = 155;
              if (cstrIsEqual(varname, "vo")) code = 138;
            }
          else if (code <= 0)
            {
              vlistInqVarName(vlistID1, varID, varname);
              cstrToLowerCase(varname);
              if (cstrIsEqual(varname, "sd")) code = 155;
              if (cstrIsEqual(varname, "svo")) code = 138;
            }

          if (code == 155) varID1 = varID;
          if (code == 138) varID2 = varID;
        }
      else
        cdoAbort("Unexpected operatorID %d", operatorID);
    }

  const int ngrids = vlistNgrids(vlistID1);
  // find first spectral grid
  for (int index = 0; index < ngrids; index++)
    {
      const int gridID = vlistGrid(vlistID1, index);
      if (gridInqType(gridID) == GRID_SPECTRAL)
        {
          gridIDsp = gridID;
          break;
        }
    }
  // find first gaussian grid
  for (int index = 0; index < ngrids; index++)
    {
      const int gridID = vlistGrid(vlistID1, index);
      if (gridInqType(gridID) == GRID_GAUSSIAN)
        {
          gridIDgp = gridID;
          break;
        }
    }

  // define output grid
  if (luv2dv)
    {
      if (varID1 == -1) cdoWarning("U-wind not found!");
      if (varID2 == -1) cdoWarning("V-wind not found!");

      if (varID1 != -1 && varID2 != -1)
        {
          gridID1 = vlistInqVarGrid(vlistID1, varID1);

          if (gridInqType(gridID1) != GRID_GAUSSIAN) cdoAbort("U-wind is not on Gaussian grid!");

          if (gridID1 != vlistInqVarGrid(vlistID1, varID2)) cdoAbort("U and V wind must have the same grid represention!");

          if (gridID1 != -1)
            {
              const long ntr = nlat2ntr(gridInqYsize(gridID1));

              if (gridIDsp != -1)
                if (ntr != gridInqTrunc(gridIDsp)) gridIDsp = -1;

              if (gridIDsp == -1)
                {
                  gridIDsp = gridCreate(GRID_SPECTRAL, (ntr + 1) * (ntr + 2));
                  gridDefTrunc(gridIDsp, ntr);
                  gridDefComplexPacking(gridIDsp, 1);
                }
            }

          if (gridIDsp == -1 && gridInqType(vlistGrid(vlistID1, 0)) == GRID_GAUSSIAN_REDUCED)
            cdoAbort("Gaussian reduced grid found. Use option -R to convert it to a regular grid!");

          if (gridIDsp == -1) cdoAbort("No Gaussian grid data found!");

          gridID2 = gridIDsp;

          vlistChangeVarGrid(vlistID2, varID1, gridID2);
          vlistChangeVarGrid(vlistID2, varID2, gridID2);
          vlistDefVarParam(vlistID2, varID1, cdiEncodeParam(155, 128, 255));
          vlistDefVarParam(vlistID2, varID2, cdiEncodeParam(138, 128, 255));
          vlistDefVarName(vlistID2, varID1, "sd");
          vlistDefVarName(vlistID2, varID2, "svo");
          vlistDefVarLongname(vlistID2, varID1, "divergence");
          vlistDefVarLongname(vlistID2, varID2, "vorticity");
          vlistDefVarUnits(vlistID2, varID1, "1/s");
          vlistDefVarUnits(vlistID2, varID2, "1/s");

          const long nlon = gridInqXsize(gridID1);
          const long nlat = gridInqYsize(gridID1);
          ntr = gridInqTrunc(gridID2);

          sptrans.init(nlon, nlat, ntr, 1);
        }
    }
  else if (ldv2uv)
    {
      if (varID1 == -1) cdoWarning("Divergence not found!");
      if (varID2 == -1) cdoWarning("Vorticity not found!");

      if (varID1 != -1 && varID2 != -1)
        {
          gridID1 = vlistInqVarGrid(vlistID1, varID2);

          if (gridInqType(gridID1) != GRID_SPECTRAL) cdoAbort("Vorticity is not on spectral grid!");

          if (gridID1 != vlistInqVarGrid(vlistID1, varID1))
            cdoAbort("Divergence and vorticity must have the same grid represention!");

          if (gridIDgp != -1)
            {
              const long nlat = gridInqYsize(gridIDgp);
              const long ntr = nlat2ntr(nlat);
              if (gridInqTrunc(gridIDsp) != ntr) gridIDgp = -1;
            }

          if (gridIDgp == -1)
            {
              char gridname[20];
              snprintf(gridname, sizeof(gridname), "t%s%dgrid", ctype, gridInqTrunc(gridIDsp));
              gridIDgp = gridFromName(gridname);
            }

          gridID2 = gridIDgp;

          vlistChangeVarGrid(vlistID2, varID1, gridID2);
          vlistChangeVarGrid(vlistID2, varID2, gridID2);
          vlistDefVarParam(vlistID2, varID1, cdiEncodeParam(131, 128, 255));
          vlistDefVarParam(vlistID2, varID2, cdiEncodeParam(132, 128, 255));
          vlistDefVarName(vlistID2, varID1, "u");
          vlistDefVarName(vlistID2, varID2, "v");
          vlistDefVarLongname(vlistID2, varID1, "u-velocity");
          vlistDefVarLongname(vlistID2, varID2, "v-velocity");
          vlistDefVarUnits(vlistID2, varID1, "m/s");
          vlistDefVarUnits(vlistID2, varID2, "m/s");

          const long nlon = gridInqXsize(gridID2);
          const long nlat = gridInqYsize(gridID2);
          ntr = gridInqTrunc(gridID1);

          sptrans.init(nlon, nlat, ntr, 0);
          dvtrans.init(ntr);
        }
    }
  else if (operatorID == DV2PS)
    {
      if (varID1 == -1) cdoWarning("Divergence not found!");
      if (varID2 == -1) cdoWarning("Vorticity not found!");

      if (varID1 != -1 && varID2 != -1)
        {
          gridID1 = vlistInqVarGrid(vlistID1, varID2);

          if (gridInqType(gridID1) != GRID_SPECTRAL) cdoAbort("Vorticity is not on spectral grid!");

          if (gridID1 != vlistInqVarGrid(vlistID1, varID1))
            cdoAbort("Divergence and vorticity must have the same grid represention!");

          vlistDefVarParam(vlistID2, varID1, cdiEncodeParam(149, 128, 255));
          vlistDefVarParam(vlistID2, varID2, cdiEncodeParam(148, 128, 255));
          vlistDefVarName(vlistID2, varID1, "velopot");
          vlistDefVarName(vlistID2, varID2, "stream");
          vlistDefVarLongname(vlistID2, varID1, "velocity potential");
          vlistDefVarLongname(vlistID2, varID2, "streamfunction");
          vlistDefVarUnits(vlistID2, varID1, "m^2/s");
          vlistDefVarUnits(vlistID2, varID2, "m^2/s");

          ntr = gridInqTrunc(gridID1);
          gridID2 = gridID1;
        }
    }

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);
  std::vector<double> array1(gridsizemax);

  std::vector<double> ivar1, ivar2, ovar1, ovar2;
  if (varID1 != -1 && varID2 != -1)
    {
      nlev = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID1));

      size_t gridsize = gridInqSize(gridID1);
      ivar1.resize(nlev * gridsize);
      ivar2.resize(nlev * gridsize);

      gridsize = gridInqSize(gridID2);
      ovar1.resize(nlev * gridsize);
      ovar2.resize(nlev * gridsize);
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if ((varID1 != -1 && varID2 != -1) && (varID == varID1 || varID == varID2))
            {
              cdoReadRecord(streamID1, array1.data(), &nmiss);
              if (nmiss) cdoAbort("Missing values unsupported for spectral data!");

              const size_t gridsize = gridInqSize(gridID1);
              const size_t offset = gridsize * levelID;

              if (varID == varID1)
                arrayCopy(gridsize, array1.data(), &ivar1[offset]);
              else if (varID == varID2)
                arrayCopy(gridsize, array1.data(), &ivar2[offset]);
            }
          else
            {
              cdoDefRecord(streamID2, varID, levelID);
              if (lcopy)
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  cdoReadRecord(streamID1, array1.data(), &nmiss);
                  cdoWriteRecord(streamID2, array1.data(), nmiss);
                }
            }
        }

      if (varID1 != -1 && varID2 != -1)
        {
          if (luv2dv)
            trans_uv2dv(sptrans, nlev, gridID1, ivar1.data(), ivar2.data(), gridID2, ovar1.data(), ovar2.data());
          else if (ldv2uv)
            trans_dv2uv(sptrans, dvtrans, nlev, gridID1, ivar1.data(), ivar2.data(), gridID2, ovar1.data(), ovar2.data());
          else if (operatorID == DV2PS)
            {
              dv2ps(ivar1.data(), ovar1.data(), nlev, ntr);
              dv2ps(ivar2.data(), ovar2.data(), nlev, ntr);
            }

          const size_t gridsize = gridInqSize(gridID2);
          if (luv2dv || operatorID == DV2PS)
            {
              for (size_t levelID = 0; levelID < nlev; levelID++)
                {
                  const size_t offset = gridsize * levelID;
                  cdoDefRecord(streamID2, varID2, levelID);
                  cdoWriteRecord(streamID2, &ovar2[offset], 0);
                }
              for (size_t levelID = 0; levelID < nlev; levelID++)
                {
                  const size_t offset = gridsize * levelID;
                  cdoDefRecord(streamID2, varID1, levelID);
                  cdoWriteRecord(streamID2, &ovar1[offset], 0);
                }
            }
          else if (ldv2uv)
            {
              for (size_t levelID = 0; levelID < nlev; levelID++)
                {
                  const size_t offset = gridsize * levelID;
                  cdoDefRecord(streamID2, varID1, levelID);
                  cdoWriteRecord(streamID2, &ovar1[offset], 0);
                }
              for (size_t levelID = 0; levelID < nlev; levelID++)
                {
                  const size_t offset = gridsize * levelID;
                  cdoDefRecord(streamID2, varID2, levelID);
                  cdoWriteRecord(streamID2, &ovar2[offset], 0);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
