/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

     Deltat    deltat         Delta t
*/

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"
#include "datetime.h"


void *
Deltat(void *process)
{
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("deltat",     0,     0, nullptr);
  cdoOperatorAdd("dTdivdt",    0,     1, nullptr);
  // clang-format on

  const int operatorID = cdoOperatorID();
  const bool ldivdt = cdoOperatorF2(operatorID);

  const int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const int calendar = taxisInqCalendar(taxisID1);

  const int streamID2 = cdoStreamOpenWrite(cdoStreamName(1));
  pstreamDefVlist(streamID2, vlistID2);

  FieldVector2D vars;
  fieldsFromVlist(vlistID1, vars, FIELD_PTR);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);
  std::vector<double> array1(gridsizemax), array2(gridsizemax);

  int tsID = 0;
  int nrecs = cdoStreamInqTimestep(streamID1, tsID);

  JulianDate juldate0 = julianDateEncode(calendar, taxisInqVdate(taxisID1), taxisInqVtime(taxisID1));

  for (int recID = 0; recID < nrecs; ++recID)
    {
      pstreamInqRecord(streamID1, &varID, &levelID);
      pstreamReadRecord(streamID1, vars[varID][levelID].ptr, &nmiss);
      vars[varID][levelID].nmiss = nmiss;
    }

  tsID++;
  int tsID2 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const JulianDate juldate1 = julianDateEncode(calendar, taxisInqVdate(taxisID1), taxisInqVtime(taxisID1));
      const double idt_in_sec = ldivdt ? 1./julianDateToSeconds(julianDateSub(juldate1, juldate0)) : 1;
      juldate0 = juldate1;

      taxisCopyTimestep(taxisID2, taxisID1);
      pstreamDefTimestep(streamID2, tsID2);

      for (int recID = 0; recID < nrecs; ++recID)
        {
          pstreamInqRecord(streamID1, &varID, &levelID);
          pstreamReadRecord(streamID1, array1.data(), &nmiss);

          const double missval = vars[varID][levelID].missval;
          const size_t gridsize = vars[varID][levelID].size;
          double *array0 = vars[varID][levelID].ptr;
          if (nmiss || vars[varID][levelID].nmiss)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  if (DBL_IS_EQUAL(array0[i], missval) || DBL_IS_EQUAL(array1[i], missval))
                    array2[i] = missval;
                  else
                    array2[i] = (array1[i] - array0[i]) * idt_in_sec;
                }

              nmiss = arrayNumMV(gridsize, array2.data(), missval);
            }
          else
            {
              for (size_t i = 0; i < gridsize; ++i) array2[i] = (array1[i] - array0[i]) * idt_in_sec;
            }

          arrayCopy(gridsize, array1.data(), array0);

          pstreamDefRecord(streamID2, varID, levelID);
          pstreamWriteRecord(streamID2, array2.data(), nmiss);
        }

      tsID++;
      tsID2++;
    }

  fieldsFree(vlistID1, vars);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
