/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2018 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"
#include "listarray.h"

void *
Histogram(void *process)
{
  int nrecs, varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  // clang-format off
  int HISTCOUNT = cdoOperatorAdd("histcount", 0, 0, NULL);
  int HISTSUM   = cdoOperatorAdd("histsum",   0, 0, NULL);
  int HISTMEAN  = cdoOperatorAdd("histmean",  0, 0, NULL);
  int HISTFREQ  = cdoOperatorAdd("histfreq",  0, 0, NULL);
  // clang-format on

  UNUSED(HISTSUM);

  int operatorID = cdoOperatorID();

  operatorInputArg("bins");

  ListArray<double> listArrayFlt;
  int nbins = listArrayFlt.argvToFlt(operatorArgc(), operatorArgv()) - 1;
  if (nbins < 1) cdoAbort("Too few arguments!");
  double *fltarr = listArrayFlt.data();

  if (Options::cdoVerbose)
    {
      printf("nbins = %d\n", nbins);
      for (int i = 0; i < nbins; i++) printf("flt %d = %g\n", i + 1, fltarr[i]);
    }

  int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  int vlistID1 = cdoStreamInqVlist(streamID1);
  int taxisID1 = vlistInqTaxis(vlistID1);

  int vlistID2 = vlistDuplicate(vlistID1);

  /* create zaxis for output bins */
  int zaxisID2 = zaxisCreate(ZAXIS_GENERIC, nbins);

  {
    std::vector<double> bins(nbins);
    /* for ( int i = 0; i < nbins; i++ ) bins[i] = (fltarr[i]+fltarr[i+1])/2; */
    for (int i = 0; i < nbins; i++) bins[i] = fltarr[i];
    zaxisDefLevels(zaxisID2, bins.data());
  }

  zaxisDefLbounds(zaxisID2, fltarr);
  zaxisDefUbounds(zaxisID2, fltarr + 1);
  zaxisDefName(zaxisID2, "bin");
  zaxisDefLongname(zaxisID2, "histogram bins");
  zaxisDefUnits(zaxisID2, "level");

  /* check zaxis: only 2D fields allowed */
  int nzaxis = vlistNzaxis(vlistID1);
  for (int index = 0; index < nzaxis; index++)
    {
      int zaxisID = vlistZaxis(vlistID1, index);
      int nlevel = zaxisInqSize(zaxisID);
      if (nlevel > 1) cdoAbort("Found 3D field with %d levels. Only 2D fields allowed!", nlevel);
      vlistChangeZaxisIndex(vlistID2, index, zaxisID2);
    }

  int streamID2 = cdoStreamOpenWrite(cdoStreamName(1), cdoFiletype());

  int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  pstreamDefVlist(streamID2, vlistID2);

  int nvars = vlistNvars(vlistID2);
  std::vector<std::vector<double>> vardata(nvars);
  std::vector<std::vector<double>> varcount(nvars);
  std::vector<std::vector<double>> vartcount(nvars);
  for (varID = 0; varID < nvars; varID++)
    {
      size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID));
      vardata[varID].resize(nbins * gridsize, 0);
      varcount[varID].resize(nbins * gridsize, 0);
      vartcount[varID].resize(gridsize, 0);
    }

  size_t gridsizemax = vlistGridsizeMax(vlistID1);
  std::vector<double> array(gridsizemax);

  int tsID1 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID1)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          pstreamInqRecord(streamID1, &varID, &levelID);
          pstreamReadRecord(streamID1, array.data(), &nmiss);
          double missval = vlistInqVarMissval(vlistID1, varID);

          size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));

          nmiss = 0;
          for (size_t i = 0; i < gridsize; i++)
            {
              if (!DBL_IS_EQUAL(array[i], missval))
                {
                  vartcount[varID][i] += 1;
                  int index = 0;
                  while (index < nbins)
                    {
                      size_t offset = gridsize * index;
                      if (!DBL_IS_EQUAL(vardata[varID][offset + i], missval) && array[i] >= fltarr[index]
                          && array[i] < fltarr[index + 1])
                        {
                          vardata[varID][offset + i] += array[i];
                          varcount[varID][offset + i] += 1;
                          break;
                        }
                      index++;
                    }
                }
              else
                { /* missing value */
                  nmiss++;
                }
            }
        }
      tsID1++;
    }

  pstreamDefTimestep(streamID2, 0);

  for (varID = 0; varID < nvars; varID++)
    {
      double missval = vlistInqVarMissval(vlistID2, varID);
      size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID));

      /* fix mising values */

      for (int index = 0; index < nbins; index++)
        {
          nmiss = 0;
          size_t offset = gridsize * index;

          for (size_t i = 0; i < gridsize; i++)
            {
              if (vartcount[varID][i] > 0)
                {
                  if (operatorID == HISTMEAN || operatorID == HISTFREQ)
                    {
                      if (varcount[varID][offset + i] > 0)
                        {
                          if (operatorID == HISTMEAN)
                            vardata[varID][offset + i] /= varcount[varID][offset + i];
                          else
                            vardata[varID][offset + i] = varcount[varID][offset + i] / vartcount[varID][i];
                        }
                    }
                }
              else
                {
                  nmiss++;
                  varcount[varID][offset + i] = missval;
                  vardata[varID][offset + i] = missval;
                }
            }

          pstreamDefRecord(streamID2, varID, index);

          if (operatorID == HISTCOUNT)
            pstreamWriteRecord(streamID2, &varcount[varID][offset], nmiss);
          else
            pstreamWriteRecord(streamID2, &vardata[varID][offset], nmiss);
        }
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return 0;
}
