/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2018 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/
#include <stdio.h>
#include <string.h>

#include <cdi.h>

#include <cdo_int.h>
#include "dmemory.h"
#include "field.h"
#include "util.h"

void
fieldInit(Field &field)
{
  field.fpeRaised = 0;
  field.nwpv = -1;  // number of words per value; real:1  complex:2
  field.memtype = -1;
  field.grid = -1;
  field.zaxis = -1;
  field.size = 0;
  field.nsamp = 0;
  field.nmiss = 0;
  field.nmiss2 = 0;
  field.missval = 0;
  field.weight = NULL;
  field.ptr = NULL;
  field.ptrf = NULL;
  field.ptr2 = NULL;
  field.ptr2f = NULL;

}

void
fieldsFromVlistKernel(const int vlistID, FieldVector2D &field2D, int ptype, bool lfill, double fillValue)
{
  int nvars = vlistNvars(vlistID);
  field2D.resize(nvars);

  for (int varID = 0; varID < nvars; ++varID)
    {
      int nwpv = vlistInqNWPV(vlistID, varID);  // number of words per value; real:1  complex:2
      int gridID = vlistInqVarGrid(vlistID, varID);
      size_t gridsize = gridInqSize(gridID);
      int zaxisID = vlistInqVarZaxis(vlistID, varID);
      int nlevel = zaxisInqSize(zaxisID);
      double missval = vlistInqVarMissval(vlistID, varID);

      field2D[varID].resize(nlevel);

      for (int levelID = 0; levelID < nlevel; ++levelID)
        {
          fieldInit(field2D[varID][levelID]);

          field2D[varID][levelID].nwpv = nwpv;
          field2D[varID][levelID].grid = gridID;
          field2D[varID][levelID].size = gridsize;
          field2D[varID][levelID].nsamp = 0;
          field2D[varID][levelID].nmiss = 0;
          field2D[varID][levelID].nmiss2 = 0;
          if (ptype & FIELD_FLT) field2D[varID][levelID].memtype = MEMTYPE_FLOAT;
          field2D[varID][levelID].missval = missval;
          field2D[varID][levelID].ptr = NULL;
          field2D[varID][levelID].ptrf = NULL;
          field2D[varID][levelID].ptr2 = NULL;
          field2D[varID][levelID].ptr2f = NULL;
          field2D[varID][levelID].weight = NULL;

          if (ptype & FIELD_PTR)
            {
              if (ptype & FIELD_FLT)
                {
                  field2D[varID][levelID].ptrf = (float *) Malloc(nwpv * gridsize * sizeof(float));
                  if (lfill) arrayFill(nwpv * gridsize, field2D[varID][levelID].ptrf, (float)fillValue);
                }
              else
                {
                  field2D[varID][levelID].ptr = (double *) Malloc(nwpv * gridsize * sizeof(double));
                  if (lfill) arrayFill(nwpv * gridsize, field2D[varID][levelID].ptr, fillValue);
                }
            }

          if (ptype & FIELD_PTR2)
            {
              if (ptype & FIELD_FLT)
                {
                  field2D[varID][levelID].ptr2f = (float *) Malloc(nwpv * gridsize * sizeof(float));
                  if (lfill) arrayFill(nwpv * gridsize, (float *) field2D[varID][levelID].ptr2f, (float)fillValue);
                }
              else
                {
                  field2D[varID][levelID].ptr2 = (double *) Malloc(nwpv * gridsize * sizeof(double));
                  if (lfill) arrayFill(nwpv * gridsize, (double *) field2D[varID][levelID].ptr2, fillValue);
                }
            }

          if (ptype & FIELD_WGT)
            {
              field2D[varID][levelID].weight = (double *) Malloc(nwpv * gridsize * sizeof(double));
              if (lfill) arrayFill(nwpv * gridsize, field2D[varID][levelID].weight, fillValue);
            }
        }
    }
}

void
fieldsFromVlist(const int vlistID, FieldVector2D &field2D, int ptype)
{
  fieldsFromVlistKernel(vlistID, field2D, ptype, false, 0);
}

void
fieldsFromVlist(const int vlistID, FieldVector2D &field2D, int ptype, double fillValue)
{
  fieldsFromVlistKernel(vlistID, field2D, ptype, true, fillValue);
}

void
fieldsFree(int vlistID, FieldVector2D &field2D)
{
  int nvars = vlistNvars(vlistID);
  for (int varID = 0; varID < nvars; ++varID)
    {
      int nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID, varID));
      for (int levelID = 0; levelID < nlevel; ++levelID)
        {
          if (field2D[varID][levelID].ptr) Free(field2D[varID][levelID].ptr);
          if (field2D[varID][levelID].ptrf) Free(field2D[varID][levelID].ptrf);
          if (field2D[varID][levelID].ptr2) Free(field2D[varID][levelID].ptr2);
          if (field2D[varID][levelID].ptr2f) Free(field2D[varID][levelID].ptr2f);
          if (field2D[varID][levelID].weight) Free(field2D[varID][levelID].weight);
        }
    }
}
