;;; @file disk.s
;;; Extension for accessing a disk drive on an 8-bit Commodore
;;; @author Marko Mkel (msmakela@nic.funet.fi)

;;; The protocol for accessing the disk drive is as follows.
;;; C=			other computer
;;;			<- filler (00)
;;; status (0=ok) ->
;;; 
;;;			<- <cmd=write2/write15><length><chksum><data><filler>
;;; <status (resend=$ff)> ->
;;;			<- <cmd=read2/read15><filler>
;;; <status=0><length><chksum><data> ->
;;;			<- <cmd=resend($80..$ff)>
;;; <status=0><length><chksum><data> ->
;;;			<- <cmd=quit(00)>

;;; Note that this code is binary patched before it is loaded to the computer.
;;; In sequences a9 xx a2 xx a0 xx, the 2nd and 3nd "xx" are replaced
;;; with the device number and the secondary address.

#include "ext.s"

	jsr prepare		; prepare for using the link

	;; the patcher looks for lda#, ldx#, ldy# with identical constants
	lda #15
	ldx #15			; device number (self-modified)
	ldy #15			; secondary address (self-modified)
	SETLFS
	STDEV
	lda #iend-iname
	SETNAM(iname)
	clc
	jsr open
	READST
	bne abortst

	lda #2
	tay
	SETLFSD
	lda #$23		; '#'
	sta iname		; modify the file name
	jsr open
abort	jsr clrchn
	READST
abortst
	tax
	sei
	jsr receive		; get the filler byte
	txa
	jsr send_switch		; report the status
	;; main loop
loop	jsr receive_switch
	tax
	bne noexit
	;; zero command => terminate the extension
	lda #2
	jsr close
	lda #15
	jsr close
	jmp exit
noexit	bmi sameblock		; resend last block
	asl
	bmi write		; command=$42 or $4f => write to file
	;; read from file
	jsr chkin
	READST
	bne abortst
	ldx #0
	stx chksum		; clear checksum
	;; read up to 256 bytes from the file
rd	jsr chrin
	sta filebuf,x
	inx
	sec
chksum = *+1
	adc #0
	sta chksum
	READST
	beq noeof
	and #$bf		; mask off bit 6 ($40) that indicates EOF
	bne abortst
	beq eof
noeof	txa
	bne rd			; x ranges from 0 to 255
eof	stx blklen		; store the block length
	jsr clrchn
	sei
	jsr receive		; get the filler byte
	;; send the buffer contents to the other computer
sameblock
	lda #0
	jsr send_switch		; send zero status
	lda blklen
	jsr send		; send length
	lda chksum
	jsr send		; send checksum
	ldx #0

snd	lda filebuf,x
	jsr send
	inx
blklen = *+1
	cpx #0			; placeholder for self-modifying code
	bne snd
	beq loop		; branch always

write	lsr
	and #$bf		; mask off bit 6 ($40), the tag for commands
	sta chnum		; store the channel number
	jsr receive		; receive block length
	sta blklen2
	jsr receive
	sta chksum2		; receive block checksum
	ldx #0
	stx checksum		; clear the computed checksum
recv	jsr receive
	sta filebuf,x
	sec			; update checksum
checksum = *+1
	adc #0			; placeholder for self-modifying code
	sta checksum
	inx
blklen2 = *+1
	cpx #0			; placeholder for self-modifying code
	bne recv
chksum2 = *+1
	eor #0			; compare checksums (self-modifying code)
	beq writeit
	lda #$ff
abst	jmp abortst		; repeat block

	;; write max 256 bytes to the file
writeit:
chnum = *+1
	ldx #0			; placeholder for self-modifying code
	jsr chkout
	ldx #0
wr	READST
	bne abst
	lda filebuf,x
	jsr chrout
	inx
	cpx blklen2
	bne wr
	jmp abort

iname	.byte "I"
iend

filebuf			; checksum followed by 255 bytes of file data
