/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package cavesurvey.regression;

import static org.junit.Assert.*;

import java.text.ParseException;

import org.junit.Test;
import cavesurvey.regression.RunFileConverter;
import footleg.cavesurvey.cmdline.CaveConverter.CmdlineOpt;

/**
 * Regression test class which runs data file conversions and compares the output with reference
 * files. Tests pass if the output file is identical to the reference file. A file name prefix
 * is given as the argument for each test conversion. The test will look for an input file with
 * this prefix and the suffix '_in.txt' and compare the output with a reference file with
 * this prefix and the suffix '_ref.svx'.
 *  
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.08.19                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 */
public class PocketTopoToSurvexTest {

	@Test
	public void testPocketTopoToSurvexGourAven() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "GourAven", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing GourAven survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);
	}

	@Test
	public void testPocketTopoToSurvexHSC() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "HSC", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing HSC survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);
	}

	@Test
	public void testPocketTopoToSurvexStomps() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "Stomps", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing Stomps survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);	
	}

	@Test
	public void testPocketTopoToSurvexUzuGour() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "Uzu-Gour", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing Uzu-Gour survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);
	}

	@Test
	public void testPocketTopoToSurvexSwil20120909() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "Swil20120909", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing Swil20120909 survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);	
	}

	@Test
	public void testPocketTopoToSurvexSloppy2ZigZags() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "Sloppy2ZigZags", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing Sloppy2ZigZags survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);	
	}

	@Test
	public void testPocketTopoToSurvexTripComment() throws ParseException {
		//Convert PocketTopo text export file to Survex format data
		String fileCompare = RunFileConverter.convertPocketTopoToSurvex( "TripComment", CmdlineOpt.D, CmdlineOpt.T );
		assertEquals("Comparing TripComment survex file from PocketTopo with reference.", DiffFiles.FILES_IDENTICAL, fileCompare);	
	}
	
}
