/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package footleg.cavesurvey.data.writer;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

import footleg.cavesurvey.cmdline.CaveConverter;
import footleg.cavesurvey.data.model.CaveSurvey;
import footleg.cavesurvey.data.model.SeriesLink;
import footleg.cavesurvey.data.model.SurveyLeg;
import footleg.cavesurvey.data.model.SurveySeries;
import footleg.cavesurvey.data.model.SurveyStation;
import footleg.cavesurvey.tools.UtilityFunctions;

/**
 * Writer for Survex file format text data.
 * 
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.08.29                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 * 
 * @to.do
 * TODO Output entrance flags
 * TODO Output fix flags
 * TODO Output calibration comments fields
 * TODO Output team fields
 * TODO Output instrument fields
 * TODO Support option to output multiple files using includes and equates in master file and one series per file in child files
 */
public class SurvexWriter {

	/**
	 * Generates Survex format data from a cave survey
	 * 
	 * @param surveyData The cave survey model to generate Survex data for
	 * @param outputSplays Flag to switch on/off whether splay legs are included in the output
	 * @return Text lines of Survex format data
	 */
	public List<String> generateSurvexData( CaveSurvey surveyData, boolean outputSplays ) {
		List<String> outputData = new ArrayList<String>();
		
		CaveConverter.logMessage("Generating Survex format data...");
		
		//Create dummy parent series to pass in neutral calibration settings for top level series
		SurveySeries parentSeries = new SurveySeries("parent");
		
		//Loop through all series
		ListIterator<SurveySeries> seriesIterator = surveyData.listIterator();
		while ( seriesIterator.hasNext() ) {
			SurveySeries series = seriesIterator.next();

			outputData.addAll( generateSurvexDataSeries( series, parentSeries, outputSplays ) );
		}

		return outputData;
	}
	
	/**
	 * Generates Survex format data for a survey series
	 * 
	 * @param series Survey series to generate Survex format data from
	 * @param parentSeries Survey series which is the parent of the series being passed in (used for calibrations)
	 * @return Text lines for a Survex format series data block
	 */
	private List<String> generateSurvexDataSeries( SurveySeries series, SurveySeries parentSeries, boolean outputSplays ) {
		List<String> outputData = new ArrayList<String>();

		List<String> legsData = new ArrayList<String>();
		List<String> fixedStnsData = new ArrayList<String>();
		List<String> passageData = new ArrayList<String>();
		String lastToStation = "";
		double lastLegBearing = 0.0;
		boolean startNewPassageDataBlock = false;
		boolean duplicateFlagOn = false;
		boolean surfaceFlagOn = false;
		boolean splayFlagOn = false;
		
		//Start series block
		outputData.add( "*BEGIN " + series.getSeriesName() );
		
		//Write comment if present
		if ( series.getComment().length() > 0 ) {
			String[] dataItems = UtilityFunctions.parseTripComment( series.getComment() );
			for (String line : dataItems) {
				outputData.add( ";" + line );
			}
			outputData.add( "" );
		}
		
		//Write date if present
		if ( series.getSurveyDate() != null ) {
			outputData.add( "*DATE " + UtilityFunctions.dateToString( series.getSurveyDate(), UtilityFunctions.SURVEXDATE_FORMAT)  );
		}
		
		//Write calibration lines if different to parent series
		if ( series.getDeclination() != parentSeries.getDeclination() ) {
			outputData.add( "*CALIBRATE declination " + series.getDeclination() );
		}
		if ( series.getTapeCalibration() != parentSeries.getTapeCalibration() ) {
			outputData.add( "*CALIBRATE tape " + series.getTapeCalibration() );
		}
		if ( series.getCompassCalibration() != parentSeries.getCompassCalibration() ) {
			outputData.add( "*CALIBRATE compass " + series.getCompassCalibration() );
		}
		if ( series.getClinoCalibration() != parentSeries.getClinoCalibration() ) {
			outputData.add( "*CALIBRATE clino " + series.getClinoCalibration() );
		}
		outputData.add( "" );
		

		//Add equates to output
		ListIterator<SeriesLink> linksIterator = series.getLinks().listIterator();
		while ( linksIterator.hasNext() ) {
			SeriesLink link = linksIterator.next();
			String equate = "*EQUATE ";

			String series1Prefix = "";
			String series2Prefix = "";
			if ( link.getSeries1().length() > 0 ) {
				series1Prefix = link.getSeries1() + ".";
			}
			if ( link.getSeries2().length() > 0 ) {
				series2Prefix = link.getSeries2() + ".";
			}
			equate += series1Prefix + link.getStn1().getName() + " " +
					  series2Prefix + link.getStn2().getName();

			outputData.add( equate );
		}
		outputData.add( "");

		//Loop through the series legs writing details of each leg found
		for ( int legIdx = 0; legIdx < series.legCount(); legIdx++ ) {
			SurveyLeg leg = series.getLegRaw(legIdx);
			String fromStn = leg.getFromStn().getName();

			/* Check for valid leg
			 * Final 'leg' in a series can be just a 'from station' with LRUD data on it,
			 * so only write leg data if a 'to station' exists
			 */
			if ( leg.getToStn() != null ) {
				String toStn = leg.getToStn().getName();

				if ( leg.getLength() > 0 ) {
					//Determine if any flags need setting
					String flagsSetting = "";
					//Determine whether to set or clear duplicate flag
					if ( ( leg.isDuplicate() ) && duplicateFlagOn == false ) {
						flagsSetting += " DUPLICATE";
						duplicateFlagOn = true;
					}
					else if ( ( leg.isDuplicate() == false ) && duplicateFlagOn ) {
						flagsSetting += " NOT DUPLICATE";
						duplicateFlagOn = false;
					}
					//Determine whether to set or clear surface flag
					if ( ( leg.isSurface() ) && surfaceFlagOn == false ) {
						flagsSetting += " SURFACE";
						surfaceFlagOn = true;
					}
					else if ( ( leg.isSurface() == false ) && surfaceFlagOn ) {
						flagsSetting += " NOT SURFACE";
						surfaceFlagOn = false;
					}
					if ( outputSplays ) {
						//Determine whether to set or clear splays flag
						if ( ( leg.isSplay() ) && splayFlagOn == false ) {
							flagsSetting += " SPLAY";
							splayFlagOn = true;
						}
						else if ( ( leg.isSplay() == false ) && splayFlagOn ) {
							flagsSetting += " NOT SPLAY";
							splayFlagOn = false;
						}
					}
					//Write flags line if any flags changed
					if ( flagsSetting.length() > 0 ) {
						legsData.add( "*FLAGS" + flagsSetting);
					}
					if ( ( ( outputSplays == false ) && leg.isSplay() ) == false ) {
						//Write leg data
						String legLine = fromStn + "\t" + toStn + "\t" + 
								CaveConverter.padNumber( leg.getLength(), 2, 5 ) + "\t" + 
								CaveConverter.padNumber( leg.getCompass(), 2, 6 ) + "\t" + 
								CaveConverter.padNumber( leg.getClino(), 2, 6 );
						if ( leg.getComment().length() > 0 ) {
							legLine += "\t;" + leg.getComment();
						}
						legsData.add( legLine );
					}
				}
				else {
					//Zero length leg
					if ( leg.isSplay() == false ) {
						//Valid leg, so write as equate
						legsData.add( "*EQUATE " + fromStn + "\t" + toStn );
					}
				}

				//Add FIXed points to fixed stns block for series
				SurveyStation stn = leg.getFromStn();
				for (int i=0; i < 2; i++) {
					//Loop twice, to process from and to stations
					if ( stn.isFixed() == true ) {
						String fixedStnLine = "*FIX " + stn.getName();
						fixedStnLine += "\t" + stn.getEasting() + "\t" + stn.getNorthing() + "\t" + 
								stn.getAltitude();
						fixedStnsData.add( fixedStnLine );
					}
					stn = leg.getToStn();
				}
			}

			//Process LRUD data
			if ( leg.isSplay() == false ) {
				//Start new passage data block if leg does not continue from last leg
				if (leg.getFromStn().getName().compareTo( lastToStation ) != 0 ) {
					startNewPassageDataBlock = true;
				}
				//Add LRUD to passages block for series
				if ( leg.getLeft() + leg.getRight() + leg.getUp() + leg.getDown() > 0.0 ) {
					//Add passage data block first if a new one is needed
					if ( startNewPassageDataBlock ) {
						//Before starting a new passage data block, check whether the last toStn LRUD needs adding to end of last block
						if ( ( lastToStation.length() > 0 )
						&& ( leg.getFromStn().getName().compareTo( lastToStation ) != 0 ) ) {
							//Check for LRUD data on last toStn when used in another leg 
							//(the leg with the closest bearing to the leg where this was the toStn)
							SurveyLeg bestLeg = new SurveyLeg();
							double leastBearingDiff = 360.0;
							for ( int idx = 0; idx < series.legCount(); idx++ ) {
								SurveyLeg legChk = series.getLegRaw(idx);
								if ( ( legChk.getFromStn().getName().compareTo( lastToStation ) == 0 )
								&& ( legChk.getLeft() + legChk.getRight() + legChk.getUp() + legChk.getDown() > 0.0 ) ) {
									//Suitable leg to use for LRUD data, check if closest bearing yet
									double bearingDiff = 0.0;
									if ( lastLegBearing > legChk.getCompass() ) {
										bearingDiff = lastLegBearing - legChk.getCompass();
									} 
									else {
										bearingDiff = legChk.getCompass() - lastLegBearing;
									}
									if ( bearingDiff < leastBearingDiff ) {
										bestLeg = legChk;
										leastBearingDiff = bearingDiff;
									}
								}
							}
							if ( bestLeg.getFromStn() != null ) {
								passageData.add( bestLeg.getFromStn().getName() + "\t" + 
										CaveConverter.padNumber(bestLeg.getLeft(),2,5) + "\t" + 
										CaveConverter.padNumber(bestLeg.getRight(),2,5) + "\t" + 
										CaveConverter.padNumber(bestLeg.getUp(),2,5) + "\t" + 
										CaveConverter.padNumber(bestLeg.getDown(),2,5) + "\t" );
							}
						}
						passageData.add("*data passage station left right up down");
						startNewPassageDataBlock = false;
					}
					passageData.add( fromStn + "\t" + 
							CaveConverter.padNumber(leg.getLeft(),2,5) + "\t" + 
							CaveConverter.padNumber(leg.getRight(),2,5) + "\t" + 
							CaveConverter.padNumber(leg.getUp(),2,5) + "\t" + 
							CaveConverter.padNumber(leg.getDown(),2,5) + "\t" );
				}
			
				//Store new station name for last to station
				if ( leg.getToStn() == null ) {
					//Most likely a dummy leg to store LRUD for last station in a series
					lastToStation = "";
				}
				else {
					lastToStation = leg.getToStn().getName();
					lastLegBearing = leg.getCompass();
				}
			}
		}
		
		//Write fixes data block
		if ( fixedStnsData.size() > 0 ) {
			outputData.addAll( fixedStnsData );
		}
		
		//Write legs data block
		outputData.addAll( legsData );
		
		//Turn off splays flag if on
		if ( splayFlagOn ) {
			outputData.add( "*FLAGS NOT SPLAY");
			splayFlagOn = false;
		}

		//Write passage data block
		if ( passageData.size() > 0 ) {
			outputData.add( "");
			outputData.addAll( passageData );
		}

		//Loop through inner series
		ListIterator<SurveySeries> seriesIterator = series.getInnerSeriesList().listIterator();
		while ( seriesIterator.hasNext() ) {
			SurveySeries innerSeries = seriesIterator.next();

			outputData.addAll( generateSurvexDataSeries( innerSeries, series, outputSplays ) );
		}

		//Close the series
		outputData.add( "*END " + series.getSeriesName() );
		outputData.add( "" );

		return outputData;
	}
	
}
