// -*- indent-tabs-mode: nil -*-

#include <cppunit/extensions/HelperMacros.h>

#include <string>
#include <fstream>
#include <iostream>
#include <sstream>
#include <pthread.h>

#include <sys/types.h>
#include <errno.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <fcntl.h>
#include <arpa/inet.h>

#include "../canlxx.h"

struct thread_data{
  std::string host;
  int port;
  std::string message;
  AuthN::Context* context;
};

std::string capath_ = "trusted_certificates";
std::string certpath_ = "testCA/certs/cert.pem";
std::string keypath_ = "testCA/certs/key.pem";

pthread_t c_thread;
pthread_t s_thread;
pthread_mutex_t mutex;

struct thread_data client_network_thread_data;
struct thread_data server_network_thread_data;

struct thread_data client_socket_thread_data;
struct thread_data server_socket_thread_data;

void* do_client_network_thread(void* threadarg);
void* do_server_network_thread(void* threadarg);

void* do_client_socket_thread(void* threadarg);
void* do_server_socket_thread(void* threadarg);

class IOTest
  : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(IOTest);
  CPPUNIT_TEST(TestIONetworkConnect);
  CPPUNIT_TEST(TestIOSocketConnect);
  CPPUNIT_TEST(TestIONetworkPair);
  CPPUNIT_TEST(TestIOSocketPair);
  //CPPUNIT_TEST(TestIOSocketConnect);
  //CPPUNIT_TEST(TestIOSocketPair);
  CPPUNIT_TEST_SUITE_END();

public:
  IOTest() {};
  void setUp();
  void tearDown();
  void TestIONetworkConnect();
  void TestIONetworkPair();
  void TestIOSocketConnect();
  void TestIOSocketPair();

  static void do_client_network(const std::string& host, int port, const std::string& message, AuthN::Context& ctx);
  static void do_server_network(const std::string& interface, int port, AuthN::Context& ctx);

  static int do_client_socket(const std::string& host, int port);
  static int do_server_socket(const std::string& host, int port);
};

void IOTest::setUp() {
}

void IOTest::tearDown() {
}

void IOTest::TestIONetworkConnect() {
  AuthN::Context ctx(AuthN::Context::ClientFullContext);

  std::cout<<"====== TestIONetworkConnect ======"<<std::endl;

  do_client_network("download.nordugrid.org", 443, "GET / HTTP/1.0\r\n\r\n", ctx);
}

void IOTest::TestIONetworkPair() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  ctx.SetCredentials(certpath_, keypath_);
  ctx.SetCAPath(capath_);

  int rc;
  client_network_thread_data.host = "localhost";
  client_network_thread_data.port = 19999;
  client_network_thread_data.message = "Hello from client";
  client_network_thread_data.context = &ctx;
  server_network_thread_data.host = "";
  server_network_thread_data.port = 19999;
  server_network_thread_data.context = &ctx;

  std::cout<<"====== TestIONetworkPair ======"<<std::endl;

  //pthread_mutex_init(&mutex, NULL);

  rc = pthread_create(&s_thread, NULL, do_server_network_thread, (void *)&server_network_thread_data);
  CPPUNIT_ASSERT_EQUAL(rc, 0);
  //IOTest::do_server_network("", 19999, ctx);

  rc = pthread_create(&c_thread, NULL, do_client_network_thread, (void *)&client_network_thread_data);
  CPPUNIT_ASSERT_EQUAL(rc, 0);

  //pthread_mutex_destroy(&mutex);

  pthread_exit(NULL);

}

void IOTest::TestIOSocketConnect() {
  AuthN::Context ctx(AuthN::Context::ClientFullContext);
  std::string host = "download.nordugrid.org"; 
  int port = 443;
  int sock = -1;

  std::cout<<"====== TestIOSocketConnect ======"<<std::endl;

  sock = do_client_socket(host, port);
  CPPUNIT_ASSERT_MESSAGE("Failed to connect to server", sock!=-1);

  AuthN::Status st;
  AuthN::IOSocket iosock(ctx);
  st = iosock.Connect(sock);
  if(st) std::cout<<"Succeed to connect server with SSL enabled"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  st = iosock.Write("GET / HTTP/1.0\r\n\r\n");
  if(!st) std::cerr<<"Client: Write failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
  else std::cout<<"Client: Write succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  std::string str;
  st = iosock.Read(str);
  if(!st) std::cerr<<"Client: Read failed: "<<st.GetCode()<<std::endl;
  else std::cout<<"Client: Read succeeded: "<<str<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  iosock.Close();
}

void IOTest::TestIOSocketPair() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  ctx.SetCredentials(certpath_, keypath_);
  ctx.SetCAPath(capath_);

  int rc;
  client_socket_thread_data.host = "localhost";
  client_socket_thread_data.port = 20000;
  client_socket_thread_data.message = "Hello from client";
  client_socket_thread_data.context = &ctx;
  server_socket_thread_data.host = "";
  server_socket_thread_data.port = 20000;
  server_socket_thread_data.context = &ctx;

  std::cout<<"====== TestIOSocketPair ======"<<std::endl;

  rc = pthread_create(&s_thread, NULL, do_server_socket_thread, (void *)&server_socket_thread_data);
  CPPUNIT_ASSERT_EQUAL(rc, 0);
  rc = pthread_create(&c_thread, NULL, do_client_socket_thread, (void *)&client_socket_thread_data);
  CPPUNIT_ASSERT_EQUAL(rc, 0);

  pthread_exit(NULL);

}

void* do_client_network_thread(void* threadarg) {
  struct thread_data* my_data;
  my_data = (struct thread_data*) threadarg;
  std::string host = my_data->host;
  int port = my_data->port;
  std::string message = my_data->message;
  AuthN::Context* ctx = my_data->context;

  IOTest::do_client_network(host, port, message, *ctx);
  pthread_exit(NULL);
}

void IOTest::do_client_network(const std::string& host, int port, const std::string& message, AuthN::Context& ctx) {
  AuthN::IONetwork io(ctx);
  io.SetTimeout(10000);

  AuthN::Credentials cred(ctx);
  io.SetOwnCredentials(cred);

  AuthN::Status st = io.Connect(host, port);
  if(!st)  std::cerr<<"Connect failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
  else std::cout<<"Connect succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  st = io.Write(message);
  if(!st) std::cerr<<"Client: Write failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
  else std::cout<<"Client: Write succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  sleep(2);
  //pthread_mutex_lock(&mutex);

  std::string str;
  st = io.Read(str);
  if(!st) std::cerr<<"Client: Read failed: "<<st.GetCode()<<std::endl;
  else std::cout<<"Client: Read succeeded: "<<str<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  //pthread_mutex_unlock(&mutex);

  io.Close();
}

void* do_server_network_thread(void* threadarg) {
  struct thread_data* my_data;
  my_data = (struct thread_data*) threadarg;
  std::string interface = my_data->host;
  int port = my_data->port;
  AuthN::Context* ctx = my_data->context;

  IOTest::do_server_network(interface, port, *ctx);
  pthread_exit(NULL);
}

void IOTest::do_server_network(const std::string& interface, int port, AuthN::Context& ctx) {
  AuthN::IONetwork io(ctx);
  io.SetTimeout(10000);

  AuthN::Credentials cred(ctx);
  io.SetOwnCredentials(cred);

  AuthN::IO& connected_io = io.Accept(interface, port);
  if((bool)io) std::cout<<"Accept setup succeeded"<<std::endl;
  else std::cout<<"Accept setup failed"<<std::endl;
  CPPUNIT_ASSERT((bool)io);

  sleep(1);
  //pthread_mutex_lock (&mutex);

  AuthN::Status st;
  std::string str;
  st = connected_io.Read(str);

  if(!st) std::cerr<<"Server: Read failed: "<<st.GetCode()<<std::endl;
  else std::cout<<"Server: Read succeeded: "<<str<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  std::string res_str;
  res_str = "Echo from server: "; res_str += str;
  st = connected_io.Write(res_str);
  if(!st) std::cerr<<"Server: Write failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
  else std::cout<<"Server: Write succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  //pthread_mutex_unlock(&mutex);
 
  io.Close();
}

std::string tostring(int t) {
  std::stringstream ss;
  ss << t;
  return ss.str();
}

void* do_client_socket_thread(void* threadarg) {
  struct thread_data* my_data;
  my_data = (struct thread_data*) threadarg;
  std::string host = my_data->host;
  int port = my_data->port;
  std::string message = my_data->message;
  AuthN::Context* ctx = my_data->context;

  int sock = -1;
  sock = IOTest::do_client_socket(host, port);
  CPPUNIT_ASSERT_MESSAGE("Failed to connect to server", sock!=-1);

  AuthN::Status st;
  AuthN::IOSocket iosock(*ctx);
  st = iosock.Connect(sock);
  if(st) std::cout<<"Succeed to connect server with SSL enabled"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  st = iosock.Write(message);
  if(!st) std::cerr<<"Client: Write failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
  else std::cout<<"Client: Write succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  std::string str;
  st = iosock.Read(str);
  if(!st) std::cerr<<"Client: Read failed: "<<st.GetCode()<<std::endl;
  else std::cout<<"Client: Read succeeded: "<<str<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  iosock.Close();

  pthread_exit(NULL);
}

int IOTest::do_client_socket(const std::string& host, int port) {
  int sock = -1;
  fd_set fdset;
  struct timeval tv;

  struct addrinfo address;
  memset(&address, 0, sizeof(address));
  address.ai_family = AF_UNSPEC;
  address.ai_socktype = SOCK_STREAM;
  address.ai_protocol = IPPROTO_TCP;
  std::string port_str = tostring(port);
  struct addrinfo* info = NULL;
  int ret = getaddrinfo(host.c_str(), port_str.c_str(), &address, &info);
  if ((ret != 0) || (!info)) {
    std::string err_str = gai_strerror(ret);
    std::cout<<"Failed to resolve "<<host<<" : "<<err_str<<std::endl;
    return -1;
  }
  for(struct addrinfo *in = info; in; in=in->ai_next) {
    std::cout<<"Trying to connect "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<std::endl;
    sock = ::socket(in->ai_family, in->ai_socktype, in->ai_protocol);
    if(sock==-1) {
      std::cout<<"Failed to create socket for connecting to "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<" - "<<errno<<std::endl;
      continue;
    }
    int s_flags = ::fcntl(sock, F_GETFL, 0);
    if(s_flags != -1) {
      ::fcntl(sock, F_SETFL, s_flags | O_NONBLOCK);
    } else {
      std::cout<<"Failed to get TCP socket options for connection to "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<" - "<<errno<<std::endl;
    }
    if(::connect(sock, in->ai_addr, in->ai_addrlen) == -1) {
      if(errno != EINPROGRESS) {
        std::cout<<"Failed to connect to "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<" - "<<errno<<std::endl;
        close(sock); sock = -1;
        continue;
      }
      FD_ZERO(&fdset);
      FD_SET(sock, &fdset);
      tv.tv_sec = 5; 
      tv.tv_usec = 0;
      int r = select(sock + 1, NULL, &fdset, NULL, &tv);
      if(r==0) {
        std::cout<<"Time out"<<std::endl;
        close(sock); sock = -1;
        continue;
      }
      if(r != 1) {
        std::cout<<"Failed to wait for connection: "<<errno<<std::endl;
        close(sock); sock = -1;
        continue;
      }
      int so_error;
      socklen_t len = sizeof so_error;
      getsockopt(sock, SOL_SOCKET, SO_ERROR, &so_error, &len);
      if(so_error == 0) std::cout<<host<<":"<<port<<" is open"<<std::endl;
      else { close(sock); sock = -1; continue; }
    }
    break;
  }   
  freeaddrinfo(info);
  return sock;
}

void* do_server_socket_thread(void* threadarg) {
  struct thread_data* my_data;
  my_data = (struct thread_data*) threadarg;
  std::string host = my_data->host;
  int port = my_data->port;
  AuthN::Context* ctx = my_data->context;

  int sock = -1;
  sock = IOTest::do_server_socket(host, port);
  CPPUNIT_ASSERT_MESSAGE("Failed to accept connection", sock!=-1);

  AuthN::Status st;
  AuthN::IOSocket io(*ctx);
  st = io.Accept(sock);
  if(st) std::cout<<"Accept setup succeeded"<<std::endl;
  else std::cout<<"===Accept setup failed"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  std::string str;
  st = io.Read(str);

  if(!st) std::cerr<<"Server: Read failed: "<<st.GetCode()<<std::endl;
  else std::cout<<"Server: Read succeeded: "<<str<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  std::string res_str;
  res_str = "Echo from server: "; res_str += str;
  st = io.Write(res_str);
  if(!st) std::cerr<<"Server: Write failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
  else std::cout<<"Server: Write succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  io.Close();

  pthread_exit(NULL);
}

int IOTest::do_server_socket(const std::string& host, int port) {
  int sock = -1;
  int c_sock = -1;
  fd_set fdset;
  struct timeval tv;

  struct addrinfo address;
  memset(&address, 0, sizeof(address));
  address.ai_family = AF_INET;
  address.ai_socktype = SOCK_STREAM;
  address.ai_protocol = IPPROTO_TCP;
  address.ai_flags = AI_PASSIVE;

  std::string port_str = tostring(port);
  struct addrinfo* info = NULL;
  int ret = getaddrinfo(host.empty()?NULL:host.c_str(), port_str.c_str(), &address, &info);
  if ((ret != 0) || (!info)) {
    std::string err_str = gai_strerror(ret);
    std::cout<<"Failed to resolve local address for "<<host<<" : "<<port<<err_str<<std::endl;
    return -1;
  }

  bool bound = false;
  for(struct addrinfo *in = info; in; in=in->ai_next) {
    std::cout<<"Trying to creat socket on: "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<std::endl;
    sock = ::socket(in->ai_family, in->ai_socktype, in->ai_protocol);
    if(sock==-1) {
      std::cout<<"Failed to create socket for connecting to "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<" - "<<errno<<std::endl;
      continue;
    }

    if(::bind(sock,in->ai_addr,in->ai_addrlen) == -1) {
      std::cout<<"Failed to bind socket for "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<" - "<<errno<<std::endl;
      close(sock);
      bound = false;
      break;
    }

    if(::listen(sock,-1) == -1) {
      std::cout<<"Failed to listen at "<<host<<"("<<(in->ai_family==AF_INET6?"IPv6":"IPv4")<<"):"<<port<<" - "<<errno<<
std::endl;
      close(sock);
      continue;
    }
    bound = true;
  }
  freeaddrinfo(info);
  if(bound == false) return -1;
  std::cout<<"Succeeded to bind socket"<<std::endl;

  while(1) {
    fd_set readfds;
    FD_ZERO(&readfds);
    FD_SET(sock,&readfds);
    struct timeval tv; tv.tv_sec = 2; tv.tv_usec = 0;
    int n = select(sock+1,&readfds,NULL,NULL,&tv);
    if(n < 0) {
      if(errno != EINTR) {
        std::cout<< "Failed while waiting for connection request"<<std::endl;
        ::close(sock);
        return -1;
      }
      continue;
    } else if(n == 0) continue;

    if(FD_ISSET(sock, &readfds)) {
      struct sockaddr addr;
      socklen_t addrlen = sizeof(addr);
      c_sock = ::accept(sock,&addr,&addrlen);
      if(c_sock == -1) {
        std::cout<<"Failed to accept connection request: "<<errno<<std::endl;
        continue;
      }

      char buf[128];
      memset(buf,0,sizeof(buf));
      std::string c_port;
      std::string c_host;
      struct sockaddr_storage c_addr;
      socklen_t c_addrlen;
      c_addrlen=sizeof(c_addr);
      if(getpeername(c_sock, (struct sockaddr*)(&c_addr), &c_addrlen) == 0) {
        struct sockaddr_in *sin = (struct sockaddr_in *)&c_addr;
        const char* r = inet_ntop(AF_INET, &(sin->sin_addr), buf, sizeof(buf)-1);
        if(r!=NULL) {
          c_port = tostring(ntohs(sin->sin_port));
          buf[sizeof(r)-1] = 0;
          c_host = buf;
          std::cout<<"Connection is from: "<<c_host<<":"<<c_port<<std::endl;
        }
      }

      int s_flags = ::fcntl(c_sock, F_GETFL, 0);
      if(s_flags != -1) {
        ::fcntl(c_sock, F_SETFL, s_flags | O_NONBLOCK);
      } else {
        std::cout<<"Failed to get TCP socket options for connection to "<<c_host<<":"<<c_port<<" - "<<errno<<std::endl;
      }
/*
      while(1) {
        fd_set fs;
        struct timeval t;
        FD_ZERO(&fs);
        FD_SET(c_sock, &fs);
        t.tv_sec = 2;
        t.tv_usec = 0;
        int r = select(c_sock + 1, &fs, NULL, NULL, &t);
        if(r==0) {
          std::cout<<"Time out"<<std::endl;
          continue;
        }
        if(r != 1) { // More sophisticated processing needed
          std::cout<<"Failed to wait for incoming data: "<<errno<<std::endl;
          continue;
        }
        if(FD_ISSET(c_sock, &readfds)) break;
      }
*/
      break;
    } 
  }
  return c_sock;
}

CPPUNIT_TEST_SUITE_REGISTRATION(IOTest);
