/***************************************************************************
 * Copyright (C) 2015 by Pablo Daniel Pareja Obregon                       *
 *                                                                         *
 * This is free software; you can redistribute it and/or modify            *
 * it under the terms of the GNU General Public License as published by    *
 * the Free Software Foundation; either version 2, or (at your option)     *
 * any later version.                                                      *
 *                                                                         *
 * This software is distributed in the hope that it will be useful,        *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 * GNU General Public License for more details.                            *
 *                                                                         *
 * You should have received a copy of the GNU General Public License       *
 * along with this package; see the file COPYING.  If not, write to        *
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,   *
 * Boston, MA 02110-1301, USA.                                             *
 ***************************************************************************/

#include "fileexport.h"

#include "cgraphicsscene.h"
#include "component.h"
#include "idocument.h"
#include "library.h"
#include "portsymbol.h"

#include <QDebug>
#include <QFile>
#include <QFileInfo>
#include <QMessageBox>
#include <QRegularExpression>

namespace Caneda
{
    //! \brief Constructor.
    FormatSpice::FormatSpice(SchematicDocument *doc) :
        m_schematicDocument(doc)
    {
    }

    bool FormatSpice::save()
    {
        CGraphicsScene *scene = cGraphicsScene();
        if(!scene) {
            return false;
        }

        QFile file(fileName());
        if(!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
            QMessageBox::critical(0, QObject::tr("Error"),
                    QObject::tr("Cannot save document!"));
            return false;
        }

        QString text = generateNetlist();
        if(text.isEmpty()) {
            qDebug() << "Looks buggy! Null data to save! Was this expected?";
        }

        QTextStream stream(&file);
        stream << text;
        file.close();

        return true;
    }

    SchematicDocument *FormatSpice::schematicDocument() const
    {
        return m_schematicDocument;
    }

    CGraphicsScene *FormatSpice::cGraphicsScene() const
    {
        return m_schematicDocument ? m_schematicDocument->cGraphicsScene() : 0;
    }

    QString FormatSpice::fileName() const
    {
        if(m_schematicDocument) {
            QFileInfo info(m_schematicDocument->fileName());
            QString baseName = info.completeBaseName();
            QString path = info.path();

            return path + "/" + baseName + ".net";
        }

        return QString();
    }

    /*!
     *  \brief Generate netlist
     *
     *  Iterate over all components, saving to a string the schematic netlist
     *  according to the model provided as a set of rules. In order to do so,
     *  the netlist topology must also be created, that is the connections
     *  between the multiple components must be determined and numbered to be
     *  used for the spice netlist. The set of rules used for generating the
     *  netlist from the model is specified in \ref ModelsFormat.
     *
     *  \sa generateNetlistTopology(), \ref ModelsFormat
     */
    QString FormatSpice::generateNetlist()
    {
        LibraryManager *libraryManager = LibraryManager::instance();
        QList<QGraphicsItem*> items = cGraphicsScene()->items();
        QList<Component*> components = filterItems<Component>(items);
        PortsNetlist netlist = generateNetlistTopology();

        QStringList modelsList;
        QStringList subcircuitsList;
        QStringList directivesList;
        QStringList schematicsList;

        // Start the document and write the header
        QString retVal;
        retVal.append("* Spice automatic export. Generated by Caneda.\n");
        retVal.append("\n* Spice netlist.\n");

        // Copy all the elements and properties in the schematic by
        // iterating over all schematic components.
        // *Note*: the parsing order is important to allow, for example
        // cascadable commands and if control statements correct extraction.
        foreach(Component *c, components) {

            // Get the spice model (multiple models may be available)
            QString model = c->model("spice");

            // ************************************************************
            // Parse and replace the simple commands (e.g. label)
            // ************************************************************
            model.replace("%label", c->label());
            model.replace("%n", "\n");

            QString path = libraryManager->library(c->library())->libraryPath();
            model.replace("%librarypath", path);

            path = QFileInfo(m_schematicDocument->fileName()).absolutePath();
            model.replace("%filepath", path);

            // ************************************************************
            // Parse and replace the commands with parameters
            // ************************************************************
            QStringList commands;
            QRegularExpression re;
            QRegularExpressionMatchIterator it;

            // ************************************************************
            // First parse the cascadable commands (e.g. properties)
            // ************************************************************
            commands.clear();
            re.setPattern("(%\\w+\{([\\w+-]+)})");
            it = re.globalMatch(model);
            while (it.hasNext()) {
                QRegularExpressionMatch match = it.next();
                commands << match.captured(0);
            }

            // For each command replace the parameter with the correct value
            for(int i=0; i<commands.size(); i++){

                // Extract the parameters, removing the comand (including the
                // "{" and the last character "}")
                QString parameter = commands.at(i);
                parameter.remove(QRegularExpression("(%\\w+\{)")).chop(1);

                if(commands.at(i).startsWith("%port")){
                    foreach(Port *_port, c->ports()) {
                        if(_port->name() == parameter) {
                            // Found the port, now look for its netlist name
                            for(int j = 0; j < netlist.size(); ++j) {
                                if(netlist.at(j).first == _port) {
                                    model.replace(commands.at(i), netlist.at(j).second);
                                }
                            }
                        }
                    }
                }
                else if(commands.at(i).startsWith("%property")){
                    model.replace(commands.at(i), c->properties()->propertyValue(parameter));
                }
            }

            // ************************************************************
            // Parse if control statements
            // ************************************************************
            commands.clear();
            re.setPattern("(%\\w+\{([\\w =+-,]*)})");
            it = re.globalMatch(model);
            while (it.hasNext()) {
                QRegularExpressionMatch match = it.next();
                commands << match.captured(0);
            }

            // For each command replace the parameter with the correct value
            for(int i=0; i<commands.size(); i++){

                // Extract the parameters, removing the comand (including the
                // "{" and the last character "}")
                QString parameter = commands.at(i);
                parameter.remove(QRegularExpression("(%\\w+\{)")).chop(1);
                QStringList controlStrings = parameter.split(",");

                if(commands.at(i).startsWith("%if")){
                    if(!controlStrings.at(0).isEmpty() && controlStrings.size() > 1) {
                        model.replace(commands.at(i), controlStrings.at(1));
                    }
                    else {
                        model.remove(commands.at(i));
                    }
                }
            }

            // ************************************************************
            // Now parse the non-cascadable commands (e.g. models), which may
            // have a cascadable command as an argument (e.g. properties).
            // ************************************************************
            commands.clear();
            re.setPattern("(%\\w+\{([\\w =+-\\\\(\\\\)\\n\\*\{}]+)})");
            it = re.globalMatch(model);
            while (it.hasNext()) {
                QRegularExpressionMatch match = it.next();
                commands << match.captured(0);
            }

            // For each command replace the parameter with the correct value
            for(int i=0; i<commands.size(); i++){

                // Extract the parameters, removing the comand (including the
                // "{" and the last character "}")
                QString parameter = commands.at(i);
                parameter.remove(QRegularExpression("(%\\w+\{)")).chop(1);

                if(commands.at(i).startsWith("%model")){

                    // Models should be added to a temporal list to be included
                    // only once at the end of the spice file.
                    if(!modelsList.contains(parameter)) {
                        modelsList << parameter;
                    }

                    model.remove(commands.at(i));

                }
                else if(commands.at(i).startsWith("%subcircuit")){

                    // Subcircuits should be added to a temporal list to be included
                    // only once at the end of the spice file.
                    if(!subcircuitsList.contains(parameter)) {
                        subcircuitsList << parameter;
                    }

                    model.remove(commands.at(i));

                }
                else if(commands.at(i).startsWith("%directive")){

                    // Directives should be added to a temporal list to be included
                    // only once at the end of the spice file.
                    if(!directivesList.contains(parameter)) {
                        directivesList << parameter;
                    }

                    model.remove(commands.at(i));

                }
            }

            // ************************************************************
            // Now parse the generateNetlist command, which creates a
            // temporal list of schematics needed for recursive netlists
            // generation (for recursive simulations).
            // ************************************************************
            if(model.contains("%generateNetlist")){

                QFileInfo info(c->filename());
                QString baseName = info.completeBaseName();
                QString path = libraryManager->library(c->library())->libraryPath();
                QString schematic = path + "/" + baseName + ".xsch";

                if(!schematicsList.contains(schematic)) {
                    schematicsList << schematic;
                }

                model.remove("%generateNetlist");
            }

            // Add the model and a newline to the file
            retVal.append(model + "\n");
        }

        // ************************************************************
        // Write the QStringLists that should be in the end of the
        // file (e.g. device models).
        // ************************************************************
        // Append the spice models in modelsList
        if(!modelsList.isEmpty()) {
            retVal.append("\n* Device models.\n");
            for(int i=0; i<modelsList.size(); i++){
                retVal.append(".model " + modelsList.at(i) + "\n");
            }
        }

        // Append the spice subcircuits in subcircuitsList
        if(!subcircuitsList.isEmpty()) {
            retVal.append("\n* Subcircuits models.\n");
            for(int i=0; i<subcircuitsList.size(); i++){
                retVal.append(".subckt " + subcircuitsList.at(i) + "\n"
                              + ".ends" + "\n");
            }
        }

        // Append the spice directives in directivesList
        if(!directivesList.isEmpty()) {
            retVal.append("\n* Spice directives.\n");
            for(int i=0; i<directivesList.size(); i++){
                retVal.append(directivesList.at(i) + "\n");
            }
        }

        // ************************************************************
        // Create the needed recursive netlist documents
        // ************************************************************
        if(!schematicsList.isEmpty()) {
            for(int i=0; i<schematicsList.size(); i++){

                SchematicDocument *document = new SchematicDocument();
                document->setFileName(schematicsList.at(i));

                if(document->load()) {
                    // Export the schematic to a spice netlist
                    FormatSpice *format = new FormatSpice(document);
                    format->save();
                }

                delete document;
            }
        }

        // Remove multiple white spaces to clean up the file
        QRegularExpression re(" {2,}");
        retVal.replace(re, " ");

        return retVal;
    }

    /*!
     *  \brief Generate netlist net numbers
     *
     *  Iterate over all ports, to group all connected ports under
     *  the same name (name = equiId). This name or net number must
     *  be used afterwads by all component ports during netlist
     *  generation.
     *
     *  We use all connected ports (including those connected by wires),
     *  instead of connected wires during netlist generation. This allows
     *  to create a netlist node even on those places not connected by
     *  wires (for example when connecting two components together).
     *
     *  \sa saveComponents(), Port::getEquipotentialPorts()
     */
    PortsNetlist FormatSpice::generateNetlistTopology()
    {
        /*! \todo Investigate: If we use QList<CGraphicsItem*> canedaItems = filterItems<Ports>(items);
         *  some phantom ports appear, and seem to be uninitialized, generating an ugly crash. Hence
         *  we filter generic items and use an iteration over their ports as a workaround.
         */
        QList<QGraphicsItem*> items = cGraphicsScene()->items();
        QList<CGraphicsItem*> canedaItems = filterItems<CGraphicsItem>(items);
        QList<Port*> ports;
        foreach(CGraphicsItem *i, canedaItems) {
            ports << i->ports();
        }

        int equiId = 1;
        PortsNetlist netlist;
        QList<Port*> parsedPorts;

        foreach(Port *p, ports) {
            if(parsedPorts.contains(p)) {
                continue;
            }

            QList<Port*> equi;
            p->getEquipotentialPorts(equi);
            foreach(Port *_port, equi) {
                netlist.append(qMakePair(_port, QString::number(equiId)));
            }

            equiId++;
            parsedPorts += equi;
        }

        replacePortNames(&netlist);

        return netlist;
    }

    /*!
     * \brief Replace net names in the netlist by those specified by
     * portSymbols.
     *
     * Iterate over all nets in the netlist, replacing those names that
     * correspond to the ones selected by the user using PortSymbols.
     * Take special care of the ground nets, that must be named "0" to
     * be complatible with the spice netlist format.
     *
     * \param netlist Netlist which is to be used in PortSymbol names
     * replacement.
     *
     * \sa PortSymbol, generateNetlistTopology()
     */
    void FormatSpice::replacePortNames(PortsNetlist *netlist)
    {
        QList<QGraphicsItem*> items = cGraphicsScene()->items();
        QList<PortSymbol*> portSymbols = filterItems<PortSymbol>(items);

        // Iterate over all PortSymbols
        foreach(PortSymbol *p, portSymbols) {

            // Given the port, look for its netlist name
            QString netName;
            for(int i = 0; i < netlist->size(); ++i) {
                if(netlist->at(i).first == p->port()) {
                    netName = netlist->at(i).second;
                }
            }

            // Given the netlist name, rename all occurencies with the new name
            for(int i = 0; i < netlist->size(); ++i) {
                if(netlist->at(i).second == netName) {
                    if(p->label().toLower() == "ground" || p->label().toLower() == "gnd") {
                        netlist->replace(i, qMakePair(netlist->at(i).first, QString::number(0)));
                    }
                    else {
                        netlist->replace(i, qMakePair(netlist->at(i).first, p->label()));
                    }
                }
            }
        }
    }

} // namespace Caneda
