/***************************************************************************
 * Copyright (C) 2009-2013 by Pablo Daniel Pareja Obregon                  *
 *                                                                         *
 * This is free software; you can redistribute it and/or modify            *
 * it under the terms of the GNU General Public License as published by    *
 * the Free Software Foundation; either version 2, or (at your option)     *
 * any later version.                                                      *
 *                                                                         *
 * This software is distributed in the hope that it will be useful,        *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 * GNU General Public License for more details.                            *
 *                                                                         *
 * You should have received a copy of the GNU General Public License       *
 * along with this package; see the file COPYING.  If not, write to        *
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,   *
 * Boston, MA 02110-1301, USA.                                             *
 ***************************************************************************/

#ifndef SETTINGS_PAGES_H
#define SETTINGS_PAGES_H

#include "cgraphicsscene.h"

class QComboBox;
class QCheckBox;
class QDateEdit;
class QFrame;
class QLabel;
class QLineEdit;
class QListWidget;
class QRadioButton;
class QSpinBox;

namespace Caneda
{

    /*!
     * \brief This abstract class contains methods that all pages configuration
     * must implement.
     */
    class SettingsPage : public QWidget
    {
        Q_OBJECT

    public:
        SettingsPage(QWidget *parent);

        /// Applies the configuration input by the user in the configuration page
        virtual void applyConf() = 0;
        virtual QString title() const = 0;
        virtual QIcon icon() const = 0;

        void setBackgroundColor(QPushButton *, QColor);
        QColor getBackgroundColor(QPushButton *);
        void setForegroundColor(QPushButton *, QColor);
        QColor getForegroundColor(QPushButton *);
    };

    //! \brief This class represents the general configuration page.
    class GeneralConfigurationPage : public SettingsPage
    {
        Q_OBJECT

    public:
        GeneralConfigurationPage(QWidget *parent = 0);

    private Q_SLOTS:
        void slotColorButtonDialog(QPushButton *);
        void slotBackgroundColorDialog();
        void slotBackgroundSimulationColorDialog();
        void slotForegroundColorDialog();
        void slotLineColorDialog();
        void slotSelectionColorDialog();

    public:
        void applyConf();
        QString title() const;
        QIcon icon() const;

        QLabel *title_label_;
        QFrame *horiz_line_;

        QCheckBox *checkShowGrid;
        QPushButton *buttonForeground, *buttonBackground,
                    *buttonSimulationBackground, *buttonLine, *buttonSelection;
        QSpinBox *spinWidth;
    };

    //! \brief This class represents the libraries configuration page.
    class LibrariesConfigurationPage : public SettingsPage
    {
        Q_OBJECT

    public:
        LibrariesConfigurationPage(QWidget *parent = 0);

    private Q_SLOTS:
        void slotAddLibrary();
        void slotRemoveLibrary();
        void slotAddHdlLibrary();
        void slotRemoveHdlLibrary();
        void slotGetNewLibraries();

    public:
        void applyConf();
        QString title() const;
        QIcon icon() const;

        QLabel *title_label_;
        QFrame *horiz_line_;

        QListWidget *libraryList, *hdlLibraryList;
        QPushButton *addLibrary, *removeLibrary, *addHdlLibrary, *removeHdlLibrary, *getNewLibraries;
    };

    //! \brief This class represents the simulation configuration page.
    class SimulationConfigurationPage : public SettingsPage
    {
        Q_OBJECT

    public:
        SimulationConfigurationPage(QWidget *parent = 0);

    private Q_SLOTS:
        void slotSimulationEngineSelected();

    public:
        void applyConf();
        QString title() const;
        QIcon icon() const;

        QLabel *title_label_;
        QFrame *horiz_line_;

        QRadioButton *ngspiceMode, *customMode, *binaryMode, *asciiMode;
        QLineEdit *lineSimulationCommand;
    };

    //! \brief This class represents the hdl configuration page
    class HdlConfigurationPage : public SettingsPage
    {
        Q_OBJECT

    public:
        HdlConfigurationPage(QWidget *parent = 0);

    private Q_SLOTS:
        void slotColorKeyword();
        void slotColorType();
        void slotColorAttribute();
        void slotColorBlock();
        void slotColorClass();
        void slotColorData();
        void slotColorComment();
        void slotColorSystem();

    public:
        void applyConf();
        QString title() const;
        QIcon icon() const;

        QLabel *title_label_;
        QFrame *horiz_line_;

        QPushButton *keywordButton, *typeButton, *attributeButton,
                    *blockButton, *classButton, *dataButton,
                    *commentButton, *systemButton;
    };

    //! \brief This class represents the layout configuration page
    class LayoutConfigurationPage : public SettingsPage
    {
        Q_OBJECT

    public:
        LayoutConfigurationPage(QWidget *parent = 0);

    private Q_SLOTS:
        void slotColorMetal1();
        void slotColorMetal2();
        void slotColorPoly1();
        void slotColorPoly2();
        void slotColorActive();
        void slotColorContact();
        void slotColorNwell();
        void slotColorPwell();

    public:
        void applyConf();
        QString title() const;
        QIcon icon() const;

        QLabel *title_label_;
        QFrame *horiz_line_;

        QPushButton *metal1Button, *metal2Button, *poly1Button,
                    *poly2Button, *activeButton, *contactButton,
                    *nwellButton, *pwellButton;
    };

} // namespace Caneda

#endif //SETTINGS_PAGES_H
