
#include "ccvt.h"

#include "VideoCollector.h"
#include "VideoDeviceWin32.h"

/** 
  \param index The index of the video driver


*/
CVideoDeviceWin32::CVideoDeviceWin32(int index)
{
   char name[100], version[100];
   CAPDRIVERCAPS Capabilities;

qDebug(">> CVideoDeviceWin32::CVideoDeviceWin32(%d)", index);

   m_BMISize = 0;
   m_pBMIBuf = 0;
   m_pVideoInfo = 0;
   m_Palette = 0;

   m_CapIndex = index;
   capGetDriverDescription(m_CapIndex, name, sizeof(name), version, sizeof(version));
   name[99] = '\0';
   version[99] = '\0';
   m_NodeName = version;
   m_IntfName = name;

   // First create the capture window. Well, this is an invisible window
   // with no border, no attributes, no nuttin'.
   m_CapHandle = capCreateCaptureWindow((LPTSTR)TEXT("Capture Window"),
                                        0,
                                        0, 0, 160, 120,
                                        (HWND) 0, 0);

   if (!m_CapHandle) {
     qWarning("Could not create capturewindow handle.");
   }
   else {
     // Connect to capture device, get capabilities 
     if (!capDriverConnect(m_CapHandle, m_CapIndex)) {
       qWarning("Failed to connect to capDriver %d", m_CapIndex);
     } 
     else {
       if (!capDriverGetName(m_CapHandle, name, sizeof(name)))
         m_IntfName = "** error **";
       else
         m_IntfName = name;
       if (!capDriverGetVersion(m_CapHandle, version, sizeof(version)))
         m_NodeName = "** error **";
       else
         m_NodeName = version;

       if (!capDriverGetCaps(m_CapHandle, &Capabilities, sizeof(Capabilities)))
         qWarning("Could not get driver capabilities!");
       else {
         m_HasDisplayDialog = Capabilities.fHasDlgVideoDisplay;
         m_HasFormatDialog  = Capabilities.fHasDlgVideoFormat;
         m_HasSourceDialog  = Capabilities.fHasDlgVideoSource;
   
         qDebug("HasOverlay             %s", Capabilities.fHasOverlay              ? "yes" : "no");
         qDebug("HasDlgVideoSource      %s", Capabilities.fHasDlgVideoSource       ? "yes" : "no");
         qDebug("HasDlgVideoFormat      %s", Capabilities.fHasDlgVideoFormat       ? "yes" : "no");
         qDebug("HasDlgVideoDisplay     %s", Capabilities.fHasDlgVideoDisplay      ? "yes" : "no");
         qDebug("CaptureInitialized     %s", Capabilities.fCaptureInitialized      ? "yes" : "no");
         qDebug("DriverSuppliesPalettes %s", Capabilities.fDriverSuppliesPalettes  ? "yes" : "no");

         qDebug("VideoIn     = %d", Capabilities.hVideoIn);
         qDebug("VideoOut    = %d", Capabilities.hVideoOut);
         qDebug("VideoExtIn  = %d", Capabilities.hVideoExtIn);
         qDebug("VideoExtOut = %d", Capabilities.hVideoExtOut);
       }

       if (!capDriverDisconnect(m_CapHandle))
         qWarning("Could not disconnect from handle?");
       DestroyWindow(m_CapHandle);
       m_CapHandle = 0;
       m_Validated = true;
     }
   }
qDebug("<< CVideoDeviceWin32::CVideoDeviceWin32()");
}

CVideoDeviceWin32::~CVideoDeviceWin32()
{
   qDebug("CVideoDeviceWin32::~CVideoDeviceWin32(%d)", m_CapIndex);
}

// private

void CVideoDeviceWin32::CreateImagesRGB()
{
   unsigned int i;
qDebug("<> CVideoDeviceWin32::CreateImagesRGB()");
   m_RGB.resize(m_Buffers);
   for (i = 0; i < m_Buffers; i++) {
      m_RGB.insert(i, new QImage(m_ImageSize, 32));
   }
}

void CVideoDeviceWin32::DeleteImagesRGB()
{
qDebug("<> CVideoDeviceWin32::DeleteImagesRGB()");
   m_RGB.resize(0);
}

void CVideoDeviceWin32::CreateImagesYUV()
{
   QImage *img = 0;
   unsigned int i, j;
   QSize QuarterSize;

qDebug("<> CVideoDeviceWin32::CreateImagesYUV()");
   QuarterSize.setWidth(m_ImageSize.width() / 2);
   QuarterSize.setHeight(m_ImageSize.height() / 2);
   m_Y.resize(m_Buffers);
   m_U.resize(m_Buffers);
   m_V.resize(m_Buffers);
   for (i = 0; i < m_Buffers; i++) {
      img = new QImage(m_ImageSize, 8, 256);
      for (j = 0; j< 256; j++)
         img->setColor(j, m_GrayScale[j]);
      m_Y.insert(i, img);

      img = new QImage(QuarterSize, 8, 256);
      for (j = 0; j< 256; j++)
         img->setColor(j, m_GrayScale[j]);
      m_U.insert(i, img);

      img = new QImage(QuarterSize, 8, 256);
      for (j = 0; j< 256; j++)
         img->setColor(j, m_GrayScale[j]);
      m_V.insert(i, img);
   }
}

void CVideoDeviceWin32::DeleteImagesYUV()
{
qDebug("<> CVideoDeviceWin32::DeleteImagesYUV()");
   m_Y.resize(0);
   m_U.resize(0);
   m_V.resize(0);
}


bool CVideoDeviceWin32::TryPalette(Palettes pal)
{
   bool ret = false;

   switch(pal) {
     case palI420:
       m_pVideoInfo->bmiHeader.biSizeImage = (m_ImageSize.width() * m_ImageSize.height() * 3) / 2;
       m_pVideoInfo->bmiHeader.biPlanes = 1;
       m_pVideoInfo->bmiHeader.biBitCount = 12;
       m_pVideoInfo->bmiHeader.biCompression = palI420;
       ret = capSetVideoFormat(m_CapHandle, m_pVideoInfo, m_BMISize);
       break;
         
     case palRGB24:
       m_pVideoInfo->bmiHeader.biSizeImage = 0;
       m_pVideoInfo->bmiHeader.biPlanes = 1;
       m_pVideoInfo->bmiHeader.biBitCount = 24;
       m_pVideoInfo->bmiHeader.biCompression = BI_RGB;
       ret = capSetVideoFormat(m_CapHandle, m_pVideoInfo, m_BMISize);
       break;

     case palRGB32:
       m_pVideoInfo->bmiHeader.biSizeImage = 0;
       m_pVideoInfo->bmiHeader.biPlanes = 1;
       m_pVideoInfo->bmiHeader.biBitCount = 32;
       m_pVideoInfo->bmiHeader.biCompression = BI_RGB;
       ret = capSetVideoFormat(m_CapHandle, m_pVideoInfo, m_BMISize);
       break;
     
     default:
       qWarning("CVideoDeviceWin32::TryPalette() unkown palette %d requested.", pal);
       break;
   }
   if (ret)
     m_Palette = pal;
qDebug("TryPalette(%d) return %c", pal, ret ? 'T' : 'F');
   return ret;
}

void CVideoDeviceWin32::UpdateCaptureStatus()
{
   CAPSTATUS NewStatus;

   if (m_CapHandle == 0) {
     memset(&m_CaptureStatus, 0, sizeof(m_CaptureStatus));
     return;
   }
   if (!capGetStatus(m_CapHandle, &NewStatus, sizeof(NewStatus))) {
     qWarning("Failed to get CapStatus.");
     return;
   }
   // compare with old status, emit signals if parameters have changed

   m_CaptureStatus = NewStatus;
}


void CVideoDeviceWin32::DumpVideoFormat(const BITMAPINFO *info) const
{
   const BITMAPINFOHEADER *h0 = &info->bmiHeader; // <= NT 3.51 
   const BITMAPV4HEADER   *h4 = 0; // NT 4.0, Win95
#if (WINVER >= 0x500)
   const BITMAPV5HEADER   *h5 = 0; // Win98, >= NT 5.0
#endif
   qDebug("CVideoDeviceWin32::DumpVideoFormat()");
   qDebug("struct size         : %d", h0->biSize);
   qDebug("width               : %d", h0->biWidth);
   qDebug("height              : %d", h0->biHeight);
   qDebug("planes              : %d", h0->biPlanes);
   qDebug("bit count           : %d", h0->biBitCount);
   qDebug("compression         :x%x", h0->biCompression);
   qDebug("size image          : %d", h0->biSizeImage);
   qDebug("xpels per meter     : %d", h0->biXPelsPerMeter);
   qDebug("ypels per meter     : %d", h0->biYPelsPerMeter);
   qDebug("colors used         : %d", h0->biClrUsed);
   qDebug("colors important    : %d", h0->biClrImportant);
}

void CVideoDeviceWin32::DumpCaptureParms(const CAPTUREPARMS *cp) const
{
   qDebug("CVideoDeviceWin32::DumpCaptureParms()");
   qDebug("uSeconds per frame  : %ld" , cp->dwRequestMicroSecPerFrame);
   qDebug("User must hit OK?   : %c"  , cp->fMakeUserHitOKToCapture ? 'T' : 'F');
   qDebug("Percent drop error  : %d%%", cp->wPercentDropForError);
   qDebug("Yield?              : %c"  , cp->fYield ? 'T' : 'F');
   qDebug("Index size          : %ld" , cp->dwIndexSize);
   qDebug("Chunk Granularity   : %d"  , cp->wChunkGranularity);
   qDebug("Using DOS memory?   : %c"  , cp->fUsingDOSMemory ? 'T': 'F');
   qDebug("Buffers requested   : %d"  , cp->wNumVideoRequested);
   qDebug("Capture Audio?      : %c"  , cp->fCaptureAudio ? 'T' : 'F');
   qDebug("Audio buffers       : %d"  , cp->wNumAudioRequested);
   qDebug("Abort key           :x%x"  , cp->vKeyAbort);
   qDebug("Abort left mouse?   : %c"  , cp->fAbortLeftMouse ? 'T' : 'F');
   qDebug("Abort right mouse?  : %c"  , cp->fAbortRightMouse ? 'T' : 'F');
   qDebug("Limit enabled?      : %c"  , cp->fLimitEnabled ? 'T' : 'F');
   qDebug("Time limit          : %d"  , cp->wTimeLimit);
   qDebug("MCI Controlled?     : %c"  , cp->fMCIControl ? 'T' : 'F');
   qDebug("Step MCI device?    : %c"  , cp->fStepMCIDevice ? 'T' : 'F');
   qDebug("MCI start time      : %ld" , cp->dwMCIStartTime);
   qDebug("MCI stop time       : %ld" , cp->dwMCIStopTime);
   qDebug("Double resolution?  : %c"  , cp->fStepCaptureAt2x ? 'T' : 'F');
   qDebug("Average capture step: %d"  , cp->wStepCaptureAverageFrames);
   qDebug("Audio buffer size   : %ld" , cp->dwAudioBufferSize);
   qDebug("Disable write cache?: %c"  , cp->fDisableWriteCache ? 'T' : 'F');
   qDebug("Stream Master       : %d"  , cp->AVStreamMaster);
}

void CVideoDeviceWin32::DumpCaptureStatus(const CAPSTATUS *cs) const
{
   qDebug("CVideoDeviceWin32::DumpCaptureStatus(%p)", cs);
   if (cs == 0)
     cs = &m_CaptureStatus;
   qDebug("Image  width        : %d"  , cs->uiImageWidth);
   qDebug("Image height        : %d"  , cs->uiImageHeight);
   qDebug("Live window?        : %c"  , cs->fLiveWindow ? 'T' : 'F');
   qDebug("Overlay window?     : %c"  , cs->fOverlayWindow ? 'T' : 'F');
   qDebug("Scaled?             : %c"  , cs->fScale ? 'T' : 'F');
   qDebug("Scroll              : (%d,%d)", cs->ptScroll.x, cs->ptScroll.y);
   qDebug("Using def. palette? : %c"  , cs->fUsingDefaultPalette ? 'T' : 'F');
   qDebug("Audio hardware?     : %c"  , cs->fAudioHardware ? 'T' : 'F');
   qDebug("Capture file exists?: %c"  , cs->fCapFileExists ? 'T' : 'F');
   qDebug("Current video frame : %ld" , cs->dwCurrentVideoFrame);
   qDebug("Video frames dropped: %ld" , cs->dwCurrentVideoFramesDropped);
   qDebug("Current wave samples: %ld" , cs->dwCurrentWaveSamples);
   qDebug("Time elapsed (ms)   : %ld" , cs->dwCurrentTimeElapsedMS); 
   qDebug("Capturing now       : %c"  , cs->fCapturingNow ? 'T' : 'F');
   qDebug("# video buffers     : %d"  , cs->wNumVideoAllocated);
   qDebug("# audio buffers     : %d"  , cs->wNumAudioAllocated);
}

// private slots

// protected

bool CVideoDeviceWin32::Init()
{
   bool ret = false;

   qDebug(">> CVideoDeviceWin32::Init()");
   m_Palette = 0;
   // Open handle in an invisible window
   m_CapHandle = capCreateCaptureWindow((LPTSTR)TEXT("Capture Window"),
#if 1
                                        0,
                                        0, 0, 160, 120,
#else
                                        WS_BORDER | WS_VISIBLE,
                                        0, 0, 320, 240,
#endif
                                        (HWND) 0, 0);

   if (!m_CapHandle) {
     qWarning("Could not create capwindow handle.");
   }
   else {
     // Connect to capture device
     if (capDriverConnect(m_CapHandle, m_CapIndex)) {
       // Register; this will make sure we are being callbacked (if you will excuse the grammar...)
       CVideoCollector::Instance()->RegisterDevice(m_CapHandle, this);
       ret = true;

       // Get the size of the video format structure & allocate memory
       m_BMISize = capGetVideoFormatSize(m_CapHandle);
       ASSERT(m_BMISize > 0);
       m_pBMIBuf = new uchar[m_BMISize];
       m_pVideoInfo = (BITMAPINFO *)m_pBMIBuf;

       // Get the actual format structure; It contains width, height, and
       // 'compression' info
       if (capGetVideoFormat(m_CapHandle, m_pBMIBuf, m_BMISize)) {
         qDebug("Got VideoFormat, struct size = %d", m_BMISize);
         DumpVideoFormat(m_pVideoInfo);
       }
       else
         qDebug("capGetVideoFormat failed.");

       // Get initial capture status
       capGetStatus(m_CapHandle, &m_CaptureStatus, sizeof(m_CaptureStatus));
       m_ImageSize.setWidth(m_CaptureStatus.uiImageWidth);
       m_ImageSize.setHeight(m_CaptureStatus.uiImageHeight);
     }
     else {
       qWarning("Failed to connect to capDriver %d", m_CapIndex);
       DestroyWindow(m_CapHandle);
       m_CapHandle = 0;
     }
   }
   qDebug("<< CVideoDeviceWin32::Init()");
   return ret;
}

/** 
  \brief Make device inactive
  
  This function is called when the device is no longer in use;
  it unregisters the device (eliminating callback), and destroys the 
  capture handle
*/
void CVideoDeviceWin32::Exit()
{
   qDebug(">> CVideoDeviceWin32::Exit()");
   if (m_CapHandle != 0) {
     CVideoCollector::Instance()->UnregisterDevice(m_CapHandle);
     if (!capDriverDisconnect(m_CapHandle))
       qWarning("Could not disconnect from handle?");
     DestroyWindow(m_CapHandle);
   }
   m_CapHandle = 0;

   delete [] m_pBMIBuf;
   m_pBMIBuf = 0;
   m_pVideoInfo = 0;
   m_BMISize = 0;

   m_Palette = 0;
   m_ImageSize.setWidth(-1);
   m_ImageSize.setHeight(-1);
   qDebug("<< CVideoDeviceWin32::Exit()");
}



bool CVideoDeviceWin32::StartCapture()
{
   bool ret = true;
   CAPTUREPARMS CP;

   if (m_CapHandle == 0)
     return false;

   qDebug(">> CVideoDeviceWin32::StartCapture()");
   memset(&CP, 0, sizeof(CP));

#if 0
qDebug("enabling Preview");
   if (!capPreviewRate(m_CapHandle, 200)) //200 ms = 5 fps
     qWarning("Could nog set preview rate. Hmmm.");
   if (!capPreview(m_CapHandle, true)) {
     qWarning("Failed to setup Preview.");
     ret = false;
   }
#endif

   // set 'video' mode
qDebug("Setting video mode");
   m_pVideoInfo->bmiHeader.biWidth = m_ImageSize.width();
   m_pVideoInfo->bmiHeader.biHeight = m_ImageSize.height(); /* A negative height means the image is straight up. Somebody please shoot me... */

   /* When asked for both YUV and RGB images, we prefer YUV to RGB. */
   if (m_PalYUV > 0) {
     ret = TryPalette(palI420)  || 
           TryPalette(palRGB32) ||
           TryPalette(palRGB24);
   }
   else if (m_PalRGB > 0) {
     ret = TryPalette(palRGB32) ||
           TryPalette(palRGB24) ||
           TryPalette(palI420);
   }
   else {
     // should not happen...
     qWarning("StartCapture but no palette requested???");
     return false;
   }

   if (!ret) {
     qWarning("Could not set video format. Bugger.");
   }

   // Get capture setup
   if (!capCaptureGetSetup(m_CapHandle, &CP, sizeof(CP)))
     qWarning("capCaptureGetSetup() failed!");
DumpCaptureParms(&CP);

   m_Buffers = CP.wNumVideoRequested;
   if (m_RequestedBuffers > 0 && m_Buffers > m_RequestedBuffers)
     m_Buffers = m_RequestedBuffers;

   // Set capture parameters
#if 1
//   CP.dwRequestMicroSecPerFrame = 200000; // 200 ms = 5 fps
//   CP.fMakeUserHitOKToCapture = false;
//   CP.wPercentDropForError = 10; // 10% framedrop allowed
   CP.fYield = true; // when true, re-entrant. 
//   CP.dwIndexSize = 0; // use default
//   CP.wChunkGranularity = 0;
//   CP.wNumVideoRequested = m_RequestedBuffers; // 
   CP.fCaptureAudio = false; // for now
   CP.vKeyAbort = 0;
   CP.fAbortLeftMouse = false;
   CP.fAbortRightMouse = false;
   CP.fLimitEnabled = false; // endless streaming...
   CP.wTimeLimit = 0; // seconds
qDebug("calling capCaptureSetSetup");
   if (!capCaptureSetSetup(m_CapHandle, &CP, sizeof(CP)))
     qWarning("capCaptureSetSetup() failed!");
#endif

   /* now that we know what palette we are going to get from the cam,
      create the actual image buffers before we start capturing 
      (race conditions....)
    */
   if (m_PalRGB > 0)
     CreateImagesRGB();    
   if (m_PalYUV > 0)
     CreateImagesYUV();
   CreateVideoFrames();
#if 1
   // CaptureSequenceNoFile() will cause a video stream without writing it to a file.
qDebug("calling capCaptureSequenceNoFile");
   if (!capCaptureSequenceNoFile(m_CapHandle))
     qWarning("capCaptureSequenceNoFile() failed!");
#endif

   UpdateCaptureStatus();
DumpCaptureStatus();

   start();
   qDebug("<< CVideoDeviceWin32::StartCapture()");
   return ret;
}

void CVideoDeviceWin32::StopCapture()
{
   qDebug(">> CVideoDeviceWin32::StopCapture()");
   if (m_CapHandle != 0) {
     capPreview(m_CapHandle, false);
     capCaptureStop(m_CapHandle);

     /* Clean up image buffers */
     DeleteVideoFrames();
     if (m_PalRGB == 0)
       DeleteImagesRGB();
     if (m_PalYUV == 0)
       DeleteImagesYUV();

     UpdateCaptureStatus();
   }
   if (running())
     m_Capturing.wakeAll();
   while (running())
      {};
   qDebug("<< CVideoDeviceWin32::StopCapture()");
}


/* Does nothing, really */

void CVideoDeviceWin32::run()
{
   qDebug(">> CVideoDeviceWin32::run()");
   m_Capturing.wait();
   qDebug("<< CVideoDeviceWin32::run()");
}

// public

long CVideoDeviceWin32::GetDescriptor() const
{
   return (long)m_CapHandle;
}

void CVideoDeviceWin32::Mute(bool on) const
{
   /* do nothing */
}
  


/* Various callback functions */

bool CVideoDeviceWin32::CallbackControl(int id)
{
   switch(id) {
     case CONTROLCALLBACK_PREROLL:
       qDebug("CallbackControl: Get ready to roll!");
       break;
     case CONTROLCALLBACK_CAPTURING:
       //qDebug("CallbackControl: Capturing...");
       break;
     default:
       qDebug("CallbackControl: unknown code %d", id);
       break;
   }
//qDebug("CaptureCount = %d", GetCaptureCount());
   return (GetCaptureCount() > 0);
}

bool CVideoDeviceWin32::CallbackError(int id, const QString &text)
{
qDebug("CallbackError(%d, \"%s\")", id, (const char *)text);
   return true;
}

bool CVideoDeviceWin32::CallbackStatus(int id, const QString &text)
{
   switch(id)
   {
     case 0:             qDebug("CallbackStatus: Reset"); break;
     case 502:           qDebug("CallbackStatus: Capture prepare"); break;
     case IDS_CAP_BEGIN: qDebug("CallbackStatus: Capture starts"); ;
     case IDS_CAP_END:   qDebug("CallbackStatus: Capture ends"); break;
     default:            qDebug("CallbackStatus(%d, \"%s\")", id, (const char *)text); break;
   }
   UpdateCaptureStatus();
   DumpCaptureStatus();
   return true;
}

bool CVideoDeviceWin32::CallbackVideoStream(VIDEOHDR *vh)
{
   uchar *dst = 0;
   const QImage *img = 0;
   CVideoFrame *frame = 0;

   m_FrameCount++;
#if 0
   if ((m_FrameCount % 10) == 0)
   {
qDebug("CallbackVideoStream(buffer = %p, len = %d, used = %d, time = %d, user = %d, flags = %d",
       vh->lpData, 
       vh->dwBufferLength, vh->dwBytesUsed, vh->dwTimeCaptured,
       vh->dwUser, vh->dwFlags);
   }
#endif
   if (!m_CaptureStatus.fCapturingNow)
     return false;
   
   frame = GetFillFrame(); 
   if (frame == 0) {
     m_FrameDropped++;
     return true;
   }

   img = frame->GetRGB();
   if (img != 0) {
     dst = img->bits();
     if (dst != 0) {
       switch(m_Palette)
       {
         case palI420:
           ccvt_420p_bgr32(m_ImageSize.width(), m_ImageSize.height(), vh->lpData, dst);
           emit FrameReady();
           break;

         case palRGB24:
           /* Negative height means image is upside down. Somebody please 
              shoot the person who is responsible for this.... *sigh* 
            */
           ccvt_bgr24_bgr32(m_ImageSize.width(), -m_ImageSize.height(), vh->lpData, dst);
           break;
       }
     }
   }
   ReturnFillFrame();
   SendSignal(frame_ready);
   return true;
}

bool CVideoDeviceWin32::CallbackYield()
{
qDebug("CallbackYield()");
   return true;
}



// public slosts

void CVideoDeviceWin32::ShowDisplayDialog()
{
   if (m_CapHandle && m_HasDisplayDialog) {
     capDlgVideoDisplay(m_CapHandle);
     UpdateCaptureStatus();
   }
}

void CVideoDeviceWin32::ShowFormatDialog()
{
   if (m_CapHandle && m_HasFormatDialog) {
     capDlgVideoFormat(m_CapHandle);
     UpdateCaptureStatus();
   }
}

void CVideoDeviceWin32::ShowSourceDialog()
{
   if (m_CapHandle && m_HasSourceDialog) {
     capDlgVideoSource(m_CapHandle);
     UpdateCaptureStatus();
   }
}


