/*  CamStream: a collection of GUI webcam tools
    Copyright (C) 2002 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

/**
  \class CVideoDevice
  \brief Base class for a video capturing device

  This class is the (abstract) base of all platform dependant classes. It
  has the common functionality of video capturing:
  - opening/closing (with multiple readers)
  - getting information
  - start/stop capture stream
  - show user-dialogs for various settings (*)
  - selection of inputs, tuners and channels

  (*) This is mainly due to the Windows platform, because there it is
  the \em driver that supplies the various dialogs that control the
  parameters of the device like width & height, brightness, input,
  tuner frequency, etc. This in contrast to Linux, which has it all
  wrapped up in the Video4Linux API. The behaviour is simulated under
  Linux :-)

  The class supports output in both RGB and YUV (4:2:0) format; a caller has
  to specify which format he desires (maybe even both). CVideoDevice
  internally transforms one format to the other. If no format is specified,
  no image is captured!

  Support for TV tuners is also built in; the user can specify a list of
  presets through a dialog supplied by this class.
*/

/*
  The design of this class is as follows: You create the class; this will
  do a probe and some initial bookkeeping.  Then, you Open() the device; the
  device can be "openend" multiple times. Nothing happens yet, until you
  call EnableRGB/EnableYUV. Then the class will start streaming, and
  emitting signals whenever a new frame is in.

  To stop streaming, Close() the device or call DisableRGB/DisableYUV, as
  many times as you called EnableRGB resp. EnableYUV. The same is true for
  Open()/Close().
*/

#include <qevent.h>

#include "ChannelEditorDlg.h"
#include "VideoDevice.h"



// private class
class CVideoDeviceEvent: public QEvent
{
public:
   CVideoDeviceEvent(CVideoDevice::Signals signal_type);

   CVideoDevice::Signals m_Signal;
};

CVideoDeviceEvent::CVideoDeviceEvent(CVideoDevice::Signals signal_type)
        : QEvent(QEvent::User)
{
   m_Signal = signal_type;
}




CVideoDevice::CVideoDevice()
{
qDebug(">> CVideoDevice::CVideoDevice()");
   m_Validated = false;

   m_HasDisplayDialog = false;
   m_HasFormatDialog = false;
   m_HasSourceDialog = false;
   m_HasTunerDialog = false;

   m_pTunerDlg = 0;

   m_OpenCount = 0;
   m_CaptureCount = 0;
   m_RequestedBuffers = 0;
   m_FrameCount = 0;
   m_FrameDropped = 0;
   m_Buffers = 0;
   m_PalRGB = 0;
   m_PalYUV = 0;

   m_Mutable = false;

   m_RGB.setAutoDelete(true);
   m_Y.setAutoDelete(true);
   m_U.setAutoDelete(true);
   m_V.setAutoDelete(true);

   m_VideoFrames.setAutoDelete(true);

   m_pNullImage = new QImage();
   for (int i = 0; i < 256; i++)
      m_GrayScale[i] = qRgb(i, i, i);
   m_TVChannels.setAutoDelete(true);
qDebug("<< CVideoDevice::CVideoDevice()");
}

CVideoDevice::~CVideoDevice()
{
qDebug(">> CVideoDevice::~CVideoDevice()");
   while (m_OpenCount > 0)
      Close();
qDebug("<< CVideoDevice::~CVideoDevice()");
}

// private

void CVideoDevice::IncrementPalette(Palettes pal)
{
   int new_count = 0;

qDebug(">> CVideoDevice::IncrementPalette(%d)", pal);
   switch(pal) {
     case RGB: new_count = m_PalRGB; break;
     case YUV: new_count = m_PalYUV; break;
     default:
       qWarning("Unknown palette %d", pal);
       return;
   }

  /* At first use of a palette, stop and restart capture, so that the driver
     can take an intelligent decision on which palette to use.
   */

   if (new_count == 0 && m_CaptureCount > 0)
     StopCapture(); // we were busy...
   switch(pal) {
     case RGB: m_PalRGB++; break;
     case YUV: m_PalYUV++; break;
   }
   m_CaptureCount = m_PalRGB + m_PalYUV;
   if (new_count == 0)
     StartCapture();
qDebug("<< CVideoDevice::IncrementPalette()");
}

void CVideoDevice::DecrementPalette(Palettes pal)
{
   int new_count = 0;

qDebug(">> CVideoDevice::DecrementPalette(%d)", pal);
   if (m_CaptureCount == 0) {
     // Should not happen
     qFatal("CVideoDevice::DecrementPalette() and m_CaptureCount == 0 !");
   }

   switch(pal) {
     case RGB: new_count = --m_PalRGB; break;
     case YUV: new_count = --m_PalYUV; break;
     default:
       qWarning("Unknown palette %d", pal);
       return;
   }

   if (new_count == 0) { // Last usage of this palette
     StopCapture();

   m_CaptureCount = m_PalRGB + m_PalYUV;
   // In case we still need data, restart capture, but perhaps with a different palette
   if (m_CaptureCount > 0)
     StartCapture();
   }
qDebug("<< CVideoDevice::DecrementPalette()");
}


/**
  \brief Parse XML node for TV Channels
*/
void CVideoDevice::ParseTVChannels(const QDomNode &node)
{
  QDomNode v;
  QDomElement e, f;
  QString fs;
  TVChannel *tv = 0;

  m_TVChannels.clear();

  v = node.firstChild();
  while (!v.isNull()) {
    e = v.toElement();
    if (!e.isNull() && e.tagName() == "channel") {
      tv = new TVChannel();
      if (tv == 0) {
        qWarning("CChannelEditorDlg::ParseTVChannels() Failed to create TVChannel struct");
      }
      else {
        tv->Name = e.attribute("name");

        f = e.namedItem("frequency_system").toElement();
        if (!f.isNull()) {
          fs = f.text();
          tv->FrequencySystem = TVChannel::European; // default
          if (fs == "american")
            tv->FrequencySystem = TVChannel::American;
          if (fs == "japanese")
            tv->FrequencySystem = TVChannel::Japanese;
        }

        f = e.namedItem("channel").toElement();
        if (!f.isNull()) {
          tv->Channel = f.text().toInt();
        }

        f = e.namedItem("finetuning").toElement();
        if (!f.isNull()) {
          tv->Finetuning = f.text().toInt();
        }

        f = e.namedItem("colorsystem").toElement();
        if (!f.isNull()) {
          fs = f.text();

          tv->ColorSystem = TVChannel::PAL_BG; // default
          if (fs == "ntsc")       tv->ColorSystem = TVChannel::NTSC;
          if (fs == "secam")      tv->ColorSystem = TVChannel::SECAM;
          if (fs == "pal-nc")     tv->ColorSystem = TVChannel::PAL_NC;
          if (fs == "pal-m")      tv->ColorSystem = TVChannel::PAL_M;
          if (fs == "pal-n")      tv->ColorSystem = TVChannel::PAL_N;
          if (fs == "ntsc-japan") tv->ColorSystem = TVChannel::NTSC_JAPAN;
        }
        m_TVChannels.append(tv);
      }
    } // ..if "channel"
    v = v.nextSibling();
  }

  // Is there a 'current' setting?
  v = node.namedItem("current");
  if (!v.isNull()) {
    e = v.toElement();
    if (!e.isNull()) {
      fs = e.text();

      // search list
      for (m_TVChannels.first(); m_TVChannels.current() != 0; m_TVChannels.next())
      {
        if (m_TVChannels.current()->Name == fs)
        {
          break; // done!
        }
      }
    }
  }

  // Let dialog know new tv channels
  if (m_pTunerDlg != 0) {
    m_pTunerDlg->SetTVChannels(&m_TVChannels);
  }
}


/**
  \brief Create XML node with all TV channels
*/
void CVideoDevice::GetTVChannels(QDomNode &node) const
{
  QDomNode ChannelsNode, Root;
  QListIterator<TVChannel> it(m_TVChannels);
  const TVChannel *channel;

  ChannelsNode = node.namedItem("channels");
  Root = node.ownerDocument().createElement("channels");
  if (ChannelsNode.isNull()) {
    node.appendChild(Root);
  }
  else {
    node.replaceChild(Root, ChannelsNode);
  }
  for (; it.current(); ++it) {
     QDomElement e, v;
     QString s;

     channel = it.current();
     e = node.ownerDocument().createElement("channel");
     e.setAttribute("name", channel->Name);

     v = node.ownerDocument().createElement("frequency_system");
     switch (channel->FrequencySystem)
     {
       case TVChannel::European: s = "european"; break;
       case TVChannel::American: s = "american"; break;
       case TVChannel::Japanese: s = "japanese"; break;
     }
     v.appendChild(node.ownerDocument().createTextNode(s));
     e.appendChild(v);

     v = node.ownerDocument().createElement("channel");
     v.appendChild(node.ownerDocument().createTextNode(s.sprintf("%d", channel->Channel)));
     e.appendChild(v);

     v = node.ownerDocument().createElement("finetuning");
     v.appendChild(node.ownerDocument().createTextNode(s.sprintf("%d", channel->Finetuning)));
     e.appendChild(v);

     v = node.ownerDocument().createElement("colorsystem");
     switch (channel->ColorSystem)
     {
        case TVChannel::PAL_BG  : s = "pal-bg"; break;
        case TVChannel::NTSC    : s = "ntsc";   break;
        case TVChannel::SECAM   : s = "secam";  break;
        case TVChannel::PAL_NC  : s = "pal-nc"; break;
        case TVChannel::PAL_M   : s = "pal-m";  break;
        case TVChannel::PAL_N   : s = "pal-n";  break;
        case TVChannel::NTSC_JAPAN: s = "ntsc-japan"; break;
     }
     v.appendChild(node.ownerDocument().createTextNode(s));
     e.appendChild(v);

     Root.appendChild(e);
  }

  // Add the current channel (if any)
  if (m_TVChannels.current() != 0)
  {
    QDomElement v = node.ownerDocument().createElement("current");
    v.appendChild(node.ownerDocument().createTextNode(m_TVChannels.current()->Name));
    Root.appendChild(v);
  }
}



// private slots

void CVideoDevice::NewCurrentChannel()
{
   if (m_TVChannels.count() == 0)
     return;

   TVChannel *tvc = m_TVChannels.current();
   if (tvc == 0)
     return;
   NewFrequency(tvc->Frequency() * 1e6);
   emit TVChannelChanged();
}


// protected

int CVideoDevice::GetCaptureCount() const
{
   return m_CaptureCount;
}

/**
  \brief Create our VideoFrames list

  Call this function when you have created your RGB/YUV images. It will
  create m_Buffers CVideoFrames, mapped to these RGB/YUV images.

  Initially all frames are placed on the empty list.
*/
void CVideoDevice::CreateVideoFrames()
{
   uint i;
   CVideoFrame *frame;
   const QImage *rgb, *y, *u, *v;

   if (m_PalRGB > 0)
     CreateImagesRGB();
   if (m_PalYUV > 0)
     CreateImagesYUV();

   m_BufMutex.lock();
   m_VideoFrames.clear();
   m_EmptyFrames.clear();
   m_FullFrames.clear();
   m_pFillFrame = 0;
   for (i = 0; i < m_Buffers; i++) {
      rgb = 0;
      y = u = v = 0;
      if (i < m_RGB.count())
        rgb = m_RGB.at(i);
      if (i < m_Y.count())
        y = m_Y.at(i);
      if (i < m_U.count())
        u = m_U.at(i);
      if (i < m_V.count())
        v = m_V.at(i);
      frame = new CVideoFrame(i, rgb, y, u, v);
      m_VideoFrames.append(frame);
      m_EmptyFrames.append(frame);
   }
   m_BufMutex.unlock();
}

void CVideoDevice::DeleteVideoFrames()
{
   m_BufMutex.lock();
   m_VideoFrames.clear();
   m_EmptyFrames.clear();
   m_FullFrames.clear();
   m_pFillFrame = 0;
   m_BufMutex.unlock();
   DeleteImagesRGB();
   DeleteImagesYUV();
}



/**
  \brief Get a videoframe that can be filled
  \return A CVideoFrame, or NULL

  When a frame is ready from the device, call this function to get a free
  CVideoFrame that you can use to fill. This CVideoFrame is taken from the
  empty list first, then the full list if nobody is using the frame.

  This function will return NULL when no CVideoFrame is available.

  See also \ref ReturnFillFrame
*/
CVideoFrame *CVideoDevice::GetFillFrame()
{
   m_BufMutex.lock();
   if (m_pFillFrame != 0)
   {
     m_BufMutex.unlock();
     return 0;
   }

   // look at empty list
   if (m_EmptyFrames.count() > 0)
     m_pFillFrame = m_EmptyFrames.take(0);

   // If no entry found, use full list; look at oldest frame first
   if (m_pFillFrame == 0) {
     m_pFillFrame = m_FullFrames.last();
     while (m_pFillFrame != 0 && m_pFillFrame->GetRefCount() > 1)
        m_pFillFrame = m_FullFrames.prev();
     m_FullFrames.take(); // remove from list
   }

   m_BufMutex.unlock();
   return m_pFillFrame;
}

void CVideoDevice::ReturnFillFrame(bool filled)
{
   m_BufMutex.lock();
   if (m_pFillFrame != 0) {
     if (filled) {
       // it's the latest frame, so it gets to the head of the queue
       m_FullFrames.prepend(m_pFillFrame);
     }
     else {
       m_EmptyFrames.append(m_pFillFrame);
     }
     m_pFillFrame = 0;
   }
   m_BufMutex.unlock();
}

void CVideoDevice::SendSignal(Signals s)
{
   postEvent(this, new CVideoDeviceEvent(s));
}



// public

bool CVideoDevice::event(QEvent *e)
{
   CVideoDeviceEvent *ve = 0;
   if (e->type() < QEvent::User)
     return QObject::event(e);

   ve = dynamic_cast<CVideoDeviceEvent *>(e);
   if (ve == 0) {
     return false;
   }
   switch (ve->m_Signal) {
     case CVideoDevice::frame_ready: emit FrameReady(); break;
     default: return false; break;
   }
   return true;
}

/**
  \brief Sets the XML node which contains the settings for this device.

  The configuration of a device is stored in an XML subtree. This function
  sets the root node of this subtree. The format of the subtree is only
  defined for the TV channels/tuner part, and the slection of input
  and/or tuner. All other settings should be set by the specific video class.

  Note: the given node should be the parent of our real node (i.e.
  the one where you add the node from \ref GetConfigration to).
  Example:


  \attention Call this before \ref Open().
*/


void CVideoDevice::SetConfiguration(const QDomNode &node)
{
   QDomNode v;
   QDomNode Root = node.namedItem("settings");

   ParseTVChannels(Root.namedItem("channels"));
   v = Root.namedItem("input");
   if (!v.isNull())
   {
     int Input = v.toElement().text().toInt();
qDebug("Input set = %d", Input);
     if (Input >= 0) {
       SetInput(Input);
     }
   }
   v = Root.namedItem("tuner");
   if (!v.isNull())
   {
     int Tuner = v.toElement().text().toInt();
qDebug("Tuner set = %d", Tuner);
     if (Tuner >= 0) {
       SetTuner(Tuner);
     }
   }

   if (m_TVChannels.current() != 0) {
     NewFrequency(m_TVChannels.current()->Frequency() * 1e6);
   }
}


/**
  \brief Return information in XML subtree

  This function returns the common part of the video device as an XML
  node. Specifically, it contains the TV channel/tuner part.

  You should overload this function in your OS-specific subclass,
  and merge this node with the private settings from your device.
*/
void CVideoDevice::GetConfiguration(QDomNode &node) const
{
   // Add or replace in current XML tree
   QString s;
   QDomNode v, Root;

   Root = node.namedItem("settings");
   if (Root.isNull()) {
qDebug("creating <settings> node");
     Root = node.ownerDocument().createElement("settings");
     node.appendChild(Root);
   }
   GetTVChannels(Root);

   // add input/tuner to <settings>
   v = Root.namedItem("input");
   if (!v.isNull())
     Root.removeChild(v);
   v = node.ownerDocument().createElement("input");
   v.appendChild(node.ownerDocument().createTextNode(s.sprintf("%d", GetInput())));
   Root.appendChild(v);

   v = Root.namedItem("tuner");
   if (!v.isNull())
     Root.removeChild(v);
   v = node.ownerDocument().createElement("tuner");
   v.appendChild(node.ownerDocument().createTextNode(s.sprintf("%d", GetTuner())));
   Root.appendChild(v);
}


bool CVideoDevice::IsValid() const
{
   return m_Validated;
}


bool CVideoDevice::Open(uint buffers)
{
   if (m_OpenCount++ > 0) {
     return true;
   }

   m_FrameCount = 0;
   m_FrameDropped = 0;
   m_RequestedBuffers = buffers;
   if (!Init()) {
     return false;
   }
   if (m_TVChannels.current() != 0) {
     NewFrequency(m_TVChannels.current()->Frequency() * 1e6);
   }
   emit Opened();
   return true;
}

void CVideoDevice::Close()
{
   if (m_OpenCount == 0) {
     qWarning("CVideoDevice::Close() Count already at 0.");
     return;
   }

   m_OpenCount--;
   if (m_OpenCount == 0) {
     if (m_CaptureCount > 0) {
       StopCapture();
       m_CaptureCount = 0;
     }
     emit Closed();
     Exit();
   }
}


bool CVideoDevice::IsOpen() const
{
   return (m_OpenCount > 0);
}

/**
  \brief Return number of programmed TV channels
  \return a positive integer

  The user (and to some extent the program) has the ability to define
  channels for the Tuner through the \ref ShowTunerDialog(). This function
  returns the number of presets stored.

  It is not possible to define new channels directly in CVideoDevice; this is
  left to the TunerDialog. You can only query and select them.

  \note The returned of channels can change at any time! Especially if the
  TunerDialog is open...

  \note The channels are stored in the XML tree with \ref GetConfiguration().

  \note The correct tuner + input has to be selected first in order to get a picture
  from the tuner...
*/
unsigned int CVideoDevice::GetNumberOfTVChannels() const
{
   return m_TVChannels.count();
}

/**
  \brief Return info of one of the pre-programmed TV channels
  \param preset The channel number. Should be between 0 and GetNumberOfTVChannels()
  \return A TVChannel struct, which contains information about the desired channel

  This simply returns a TVChannel struct with the name, frequency system, etc. of
  the preset channel. If \b preset is out of bounds, an invalid structure is
  returned (Name is a null string).

*/
TVChannel CVideoDevice::QueryTVChannel(unsigned int preset)
{
   if (preset < 0 || preset >= m_TVChannels.count())
     return TVChannel();
   return *m_TVChannels.at(preset);

}


/**
  \brief Return info of the current TV channel
  \return A TVChannel struct, which contains information about the currently selected channel

  This simply returns a TVChannel struct with the name, frequency system, etc. of
  the current selected channel.

*/
TVChannel CVideoDevice::GetCurrentTVChannel() const
{
   if (m_TVChannels.at() < 0)
     return TVChannel();
   return *m_TVChannels.current();
}

/**
  \brief Select one of the pre-programmed TV channels
  \param preset The channel number. Should be between 0 and GetNumberOfTVChannels()
  \return A TVChannel struct, which contains information about the chosen channel

  This function will select one of the preset TV channels. If \b preset is out of
  bounds, nothing will happen; a TVChannel struct is still returned.

  If there <b>are</b> no TVChannels at all, an invalid struct is returns.

  \sa SelectNextTVChannel(), SelectPrevTVChannel()
*/
TVChannel CVideoDevice::SelectTVChannel(unsigned int preset)
{
  if (m_TVChannels.count() == 0)
    return TVChannel();

  if (preset >= 0 && preset < m_TVChannels.count()) {
    m_TVChannels.at(preset);
    NewCurrentChannel();
  }
  return *m_TVChannels.current();
}


/**
  \brief Select the next TV channel from the list of presets
  \param wrap When true, after the last channel the first channel is chosen. When false, nothing happens when the last channel is current.
  \return A TVChannel struct

  This function will select the channel after the current one. If that
  happened to be the last one of the list, and \b wrap is true, the
  first entry is chosen. Otherwise, nothing happens.

  If no channels are present, an invalid struct is returned.

  \sa SelectTVChannel(), SelectPrevTVChannel()
*/
TVChannel CVideoDevice::SelectNextTVChannel(bool wrap)
{
   TVChannel *tvc;

   if (m_TVChannels.count() == 0)
     return TVChannel();
   tvc = m_TVChannels.next();
   if (tvc == 0) {
     tvc = m_TVChannels.first();
   }
   NewCurrentChannel();
   return *tvc;
}


/**
  \brief Select the previous TV channel from the list of presets
  \param wrap When true, after the first channel the last channel is chosen. When false, nothing happens when the first channel is current.
  \return A TVChannel struct

  This function will select the channel before the current one. If that
  happened to be the first one of the list, and \b wrap is true, the
  last entry is chosen. Otherwise, nothing happens.

  If no channels are present, an invalid struct is returned.

  \sa SelectTVChannel(), SelectNextTVChannel()
*/
TVChannel CVideoDevice::SelectPrevTVChannel(bool wrap)
{
   TVChannel *tvc;

   if (m_TVChannels.count() == 0)
     return TVChannel();
   tvc = m_TVChannels.prev();
   if (tvc == 0) {
     tvc = m_TVChannels.last();
   }
   NewCurrentChannel();
   return *tvc;
}


/**
  \brief Returns device inode name

  This function returns the name of the device inode, like /dev/video0,
  /dev/video1, etc.
*/
QString CVideoDevice::GetNodeName() const
{
   return m_NodeName;
}

/**
  \brief Returns internal name of device.

  This function returns the name of the device through the V4L interface.
*/
QString CVideoDevice::GetIntfName() const
{
   return m_IntfName;
}

/**
  \brief Returns serial number of device, when available.
  \return A QSstring; is a null string when no serial number is available

  This function returns the serial number of the device; this is not
  always possible, but USB devices do in general have a serial number.
  If no number could be determined, a null string is returned.
*/
QString CVideoDevice::GetSerialNumber() const
{
   return m_SerialNumber;
}

int CVideoDevice::GetBuffers() const
{
   return m_Buffers;
}


void CVideoDevice::EnableRGB()
{
   IncrementPalette(RGB);
}

void CVideoDevice::DisableRGB()
{
   DecrementPalette(RGB);
}

void CVideoDevice::EnableYUV()
{
   IncrementPalette(YUV);
}


void CVideoDevice::DisableYUV()
{
   DecrementPalette(YUV);
}


/**
  \brief Return the latest video frame

  This will return a (shallow copy) of \ref CVideoFrame. Delete the object
  when you're done with it.
*/
CVideoFrame *CVideoDevice::GetLatestVideoFrame(int backlog)
{
   CVideoFrame *frame = 0;

   m_BufMutex.lock();
   frame = m_FullFrames.first();
   if (frame != 0)
     frame = new CVideoFrame(*frame);
   m_BufMutex.unlock();
   return frame;
}


bool CVideoDevice::HasDisplayDialog() const
{
   return m_HasDisplayDialog;
}

bool CVideoDevice::HasFormatDialog() const
{
   return m_HasFormatDialog;
}

bool CVideoDevice::HasSourceDialog() const
{
   return m_HasSourceDialog;
}

bool CVideoDevice::HasTunerDialog() const
{
  return m_HasTunerDialog;
}


QSize CVideoDevice::GetSize() const
{
   return m_ImageSize;
}


bool CVideoDevice::IsMutable() const
{
   return m_Mutable;
}




void CVideoDevice::ShowTunerDialog()
{
  if (m_pTunerDlg == 0) {
    m_pTunerDlg = new CChannelEditorDlg();
  }
  if (m_pTunerDlg != 0) {
    m_pTunerDlg->SetTVChannels(&m_TVChannels);
    connect(m_pTunerDlg, SIGNAL(NewCurrentChannel()), SLOT(NewCurrentChannel()));
    connect(m_pTunerDlg, SIGNAL(NewFrequency(float)), SLOT(NewFrequency(float)));
    m_pTunerDlg->show();
  }
}
