/**
  \class CCamStreamApp
  \brief The CamStream main application class.

  This class is the main object for the CamStream application. It does
  the following:
   - creates the canvas, loads widgets;
   - determines available image file formats;
   - stores and retrieves user settings;


  The configuration is stored as an XML document. The structure still
  has to be defined fully, and is partially defined by sub-elements.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>

#include <qdatetime.h>
#include <qdir.h>
#include <qfile.h>
#include <qimage.h>
#include <qmessagebox.h>
#include <qstring.h>
#include <qstrlist.h>
#include <qtextstream.h>

#include "CamStreamApp.h"
#include "VideoOptions.h"

#include "tracer.h"

/* This must follow FileTypeEnum! */
static struct {
  const char *extension;
  const char *format;
} FileTypes[] = {
  { "jpg", "JPEG" },
  { "png", "PNG" },
  { "ppm", "PPM" },
  { "bmp", "BMP" },
};

CCamStreamApp *g_pCamApp = NULL; /* should be the only one */

TR_MODULE("CamStreamApp");

/**
 \fn CCamStreamApp::CCamStreamApp(int argc, char *argv[])
 \brief Constructor

 Takes the \b argc and \b argv arguments from main().
 */

CCamStreamApp::CCamStreamApp(int argc, char *argv[])
	: QApplication(argc, argv)
{
   QDomDocument DD("CamStreamConfig");

   assert(g_pCamApp == NULL);
   g_pCamApp = this;

   qWarning("CamStream version %s starting.", PACKAGE_VERSION);

   Configuration.Document = DD; // Weird but true; if we just use an empty initialized DomDocument, it doesn't work.

#ifdef _OS_WIN32_
   /* FIXME must be read from registry, or something like that */
//   m_ShareDir = "c:/program files/camstream";
#else
//   m_ShareDir = SHARE_DIR;
#endif

   VisiblePanels.setAutoDelete(TRUE);

   /* Set program wide defaults */
   InitFileTypes();

   m_ConfigDir.setPath(QDir::homeDirPath() + "/.camstream");
   m_ConfigFile.setName(m_ConfigDir.path() + "/config.xml");

   /* Read user settings */
   ReadConfigFile();

   /* Create temp directory in user's homedir */
   m_TempDir.setPath(m_ConfigDir.absPath() + "/temp");
   if (!m_TempDir.exists()) {
     if (!m_ConfigDir.mkdir("temp")) {
       if (QMessageBox::warning(0,
                      tr("Initialization failure"),
                      tr("Failed to create temporary upload directory\n"\
                         "(%1). Check if ~/.camstream exists.\n").arg(m_ConfigDir.absPath() + "/temp"),
                      QMessageBox::Ok    | QMessageBox::Default,
                      QMessageBox::Abort | QMessageBox::Escape) == QMessageBox::Abort) {
          QApplication::exit(1);
       }
       else {
         m_TempDir.setPath("");
       }
     }
   }
   m_TempFileNumber = (int)time(NULL); // starting point for temp files
}

/**
 \fn CCamStreamApp::~CCamStreamApp()
 \brief Destructor

 End of program; saves user settings (image formats, etc)
*/
CCamStreamApp::~CCamStreamApp()
{
   SaveConfigFile();
}

// private

/** Determine available formats for saving files, set basic filename and text color */
void CCamStreamApp::InitFileTypes()
{
   /* A friend of mine once told me this story:

      At work they used to have a mainframe or other heavy Unix machine
      that had a switch, labeled "Magic" and "More magic" (a simple,
      dual flip switch like an on-off button). Whenever they put the
      switch in the "Magic" position, the machine would crash instantly.
      In the "More magic" position it worked fine. Now for the weird part:
      there was only a single wire leading from the switch to the mainboard,
      and there it was connected to Ground. In other words: this wire was
      electrically speaking just floating in the air, with a switch on the
      end that wasn't connected to anything. Yet, if it was not in the
      right position, it would crash the machine.

      (Needless to say, this reeks of an Urban Legend)
   */
   QStrList lit;
   QStrListIterator *it;
   char *s;
   int c;

   snap_bitmask = 0;
   lit = QImage::outputFormats();
   it = new QStrListIterator(lit);
   s = it->toFirst();
   while (s != NULL) {
      for (c = 0; c < file_MAX; c++) {
         if (!strcmp(s, FileTypes[c].format))
           snap_bitmask |= (1 << c);
      }
      ++*it;
      s = it->current();
   }
}


/** This function will silently update the configuration of CamStream
    We used to have a .camstream file, we are now going to have a
    .camstream/ directory with a config file, upload directories, etc.

*/
void CCamStreamApp::UpdateConfiguration()
{
   QString s = QDir::home().path() + QDir::separator() + ".camstream";
   QFileInfo OldConfigFile(s);

   if (OldConfigFile.isFile())
   {
     qWarning("Found old .camstream file, upgrading configuration (this should happen only once)");

     // rename .camstream to .camstream.bak
     if (!QDir::home().rename(".camstream", ".camstream.bak")) {
       if (QMessageBox::warning(0,
                      tr("Upgrade failed"),
                      tr("Configuration upgrade failed.\n"\
                         "Could not rename ~/.camstream to ~/.camstream.bak.\n"\
                         "Please check permissions of your home directory."),
                      QMessageBox::Ok, QMessageBox::Abort) == QMessageBox::Abort) {
         QApplication::exit(1);
         return;
       }
     }
     else {
       // renamed, create subdirectory
       if (!QDir::home().mkdir(".camstream")) {
         if (QMessageBox::warning(0,
                        tr("Upgrade failed"),
                        tr("Configuration upgrade failed.\n"\
                           "Could not create ~/.camstream directory.\n"),
                        QMessageBox::Ok, QMessageBox::Abort) == QMessageBox::Abort) {
           QApplication::exit(1);
           return;
         }
       }
       else {
         // subdirectory created, now move .bak file into it
         if (!QDir::home().rename(".camstream.bak", ".camstream/config.xml")) {
           if (QMessageBox::warning(0,
                          tr("Upgrade failed"),
                          tr("Configuration upgrade failed.\n"\
                             "Could not move ~/.camstream.bak to ~/.camstream/config.xml\n"\
                             "Check if the ~/.camstream directory has been created properly."),
                        QMessageBox::Ok, QMessageBox::Abort) == QMessageBox::Abort) {
             QApplication::exit(1);
             return;
           }
         }
       }
     } // ..rename
   }
   else {
     // check if it's a directory; if not, create one
     if (!OldConfigFile.isDir()) {
       if (!QDir::home().mkdir(".camstream")) {
         if (QMessageBox::warning(0,
                        tr("Upgrade failed"),
                        tr("Configuration failed.\n"\
                           "Could not create ~/.camstream directory.\n"),
                        QMessageBox::Ok, QMessageBox::Abort) == QMessageBox::Abort) {
           QApplication::exit(1);
           return;
         }
       }
     }
   } // ..OldConfigFile.exists
}


void CCamStreamApp::ReadConfigFile()
{
   QDomNode node;
   bool Okay = false;

   TR_ENTER();
   UpdateConfiguration();
   if (!m_ConfigFile.open(IO_ReadOnly))
     qWarning("Failed to open configuration file '%s' for reading.", (const char *)m_ConfigFile.name());
   else {
     if (Configuration.Document.setContent(&m_ConfigFile))
       Okay = true;
     else
       qWarning("Failed to parse configuration file (%s).", m_ConfigFile.name().ascii());
     m_ConfigFile.close();
   }
   if (Okay) {
     Configuration.Root = Configuration.Document.documentElement();
   }
   else {
     Configuration.Root = Configuration.Document.createElement("config");
     Configuration.Root.setAttribute("version", VERSION);
     Configuration.Document.appendChild(Configuration.Root);
   }

   // Check for toplevel nodes.
   node = Configuration.Root.namedItem("defaults");
   if (node.isNull() || !node.isElement()) {
     qDebug("creating <defaults> node");
     node = Configuration.Document.createElement("defaults");
     Configuration.Root.appendChild(node);
   }
   Configuration.Defaults = node.toElement();

   node = Configuration.Root.namedItem("videodevices");
   if (node.isNull() || !node.isElement())
   {
     qDebug("creating <videodevices> node");
     node = Configuration.Document.createElement("videodevices");
     Configuration.Root.appendChild(node);
   }
   Configuration.VideoDevices = node.toElement();

   node = Configuration.Root.namedItem("audiodevices");
   if (node.isNull() || !node.isElement())
   {
     qDebug("creating <audiodevices> node");
     node = Configuration.Document.createElement("audiodevices");
     Configuration.Root.appendChild(node);
   }
   Configuration.AudioDevices = node.toElement();

   TR_SET_CONFIG(Configuration.Root);
   TR_LEAVE();
}

void CCamStreamApp::SaveConfigFile()
{
   QTextStream *ts;
   QDomElement config;

   TR_ENTER();
   /* Get tracer info, add or replace in tree */
   TR_GET_CONFIG(Configuration.Root);

#if defined(Q_OS_UNIX) || defined(_OS_UNIX_)
   mode_t oldmask;

   /* Make sure our configuration cannot be read by others (FTP passwords!) */
   oldmask = umask(077);
#endif
   if (!m_ConfigFile.open(IO_WriteOnly)) {
     qWarning("Failed to open configuration file '%s' for writing.", (const char *)m_ConfigFile.name());
   }
   else {
     /* Save config. We assume a consistent tree has been built in ReadConfigFile. */
     ts = new QTextStream(&m_ConfigFile);
     Configuration.Document.save(*ts, 0);
     delete ts;
     m_ConfigFile.close();
   }
#if defined(Q_OS_UNIX) || defined(_OS_UNIX_)
   umask(oldmask);
#endif
   TR_LEAVE();
}



// public

QString CCamStreamApp::GetShareDir() const
{
   return m_ShareDir;
}

#ifdef OBSOLETE
const QDir &CCamStreamApp::GetConfigDir() const
{
   return m_ConfigDir;
}
#endif

const QString CCamStreamApp::GetTempFileName(const QString &extension)
{
   QString s;

   s.sprintf("cstmp%d", m_TempFileNumber);
   m_TempFileNumber++;
   return m_TempDir.path() + QDir::separator() + s + extension;
}

/**
  \brief Return number of file formats

  This function returns the number of programmed file formats, including
  the formats that are currently not supported by Qt. The supported formats
  are determined at run time, see \ref GetFileTypeMask.
*/
int CCamStreamApp::GetNumberOfFileTypes() const
{
   return file_MAX;
}

/**
  \brief Return bitmak of available file formats.

  This function returns a bitmask of available formats. Every enumerated
  fileformat is represented by its corresponding bit (use 1 << \e n to
  mask out a bit). The number of fileformats is thus limited to 32.

 */
int CCamStreamApp::GetFileTypeMask() const
{
   return snap_bitmask;
}

/**
  \brief Get string with the proper filename extension.
  \param n One of \ref FileTypeEnum

  Return the proper filename extension for the given file format. For example,
  for JPEG files, it will return "jpg".
*/
QString CCamStreamApp::GetFileTypeExtension(int n)
{
   if (n >= 0 && n < file_MAX) {
     return QString(FileTypes[n].extension);
   }
   return QString("xxx"); /* something we don't understand */
}

/**
  \brief Get string with format for saving files.

  Return proper handler string for current file format (which is different
  from the file extension). See \b QImage::save(...)
 */
QString CCamStreamApp::GetFileTypeFormatStr(int n)
{
   if (n >= 0 && n < file_MAX) {
     return QString(FileTypes[n].format);
   }
   return QString("");
}

/**
  \brief Find enumeration belonging to file format
  \return A positive integer on a match, -1 when nothing was found

  This function returns a number from \ref FileTypeEnum that matches
  the string given in \e format. See \ref GetFileTypeFormatStr
*/
int CCamStreamApp::FormatStrToEnum(const QString &format)
{
   int i;

   for (i = 0; i < file_MAX; i++)
      if (format == FileTypes[i].format)
        return i;
   return -1; // No match
}

QString CCamStreamApp::FormatStrToExtension(const QString &format)
{
   int i;

   for (i = 0; i < file_MAX; i++)
      if (format == FileTypes[i].format)
        return QString(FileTypes[i].extension);
   return QString(""); // No match
}


/**
  \brief Find the options of a videodevice, matching by name or device node
  \param name The name from the device, e.g. "CPiA webcam"
  \param node The device nodename, e.g. /dev/video1
  \return a QDomNode structure, or a null node if nothing was found

  This function tries to find a matching QDomNode for a device.
  It first searches the list looking for the 'name' (which is
  a symbolic name returned by the device). If there is more than one match,
  use the node also
*/

QDomNode CCamStreamApp::FindVideoDeviceConfig(const QString &name, const QString &node, bool create)
{
   QDomNode search, best;
   QDomElement elem;
   QString Name, Node;
   int found = 0;

   TR_ENTER();
   TR(1, "Trying to find video options for %s@%s", name.ascii(), node.ascii());

   for (search = Configuration.VideoDevices.firstChild(); !search.isNull(); search = search.nextSibling()) {
      elem = search.toElement();
      if (elem.isNull())
        continue;

      Name = elem.attribute("name");
      if (Name == name) {
        found++;
        best = search;
      }
   }
   if (found > 1) { // second run, also use node name
      for (search = Configuration.VideoDevices.firstChild(); !search.isNull(); search = search.nextSibling()) {
         elem = search.toElement();
         if (elem.isNull())
           continue;

         Name = elem.attribute("name");
         Node = elem.attribute("node");
         if (Name == name && Node == node)
           best = search;
         if (best.isNull() && Node == node)
           best = search;
      }
   }

   if (best.isNull() && create) {
     qDebug("Creating new node for %s@%s", name.ascii(),node.ascii());
     elem = Configuration.Document.createElement("videodevice");
     elem.setAttribute("name", name);
     elem.setAttribute("node", node);
     Configuration.VideoDevices.appendChild(elem);
     best = elem;
   }
   TR_RET(best);
}


QDomNode CCamStreamApp::FindAudioDeviceConfig(const QString &name, const QString &node, bool create)
{
   QDomNode search, best;
   QDomElement elem;
   QString Name, Node;
   int found = 0;

   TR_ENTER();
   TR(1, "Trying to find audio options for %s@%s", name.ascii(), node.ascii());
   if (name.isEmpty())
     TR_RET(best);
   for (search = Configuration.AudioDevices.firstChild(); !search.isNull(); search = search.nextSibling())
   {
      elem = search.toElement();
      if (elem.isNull())
        continue;

      Name = elem.attribute("name");
      if (Name == name)
      {
        found++;
        best = search;
      }
   }
   /* Found more than one, use node as well */
   if (found > 1)
   {
      for (search = Configuration.AudioDevices.firstChild(); !search.isNull(); search = search.nextSibling())
      {
         elem = search.toElement();
         if (elem.isNull())
           continue;

         Name = elem.attribute("name");
         Node = elem.attribute("node");
         if (Name == name && Node == node)
           best = search;
         if (best.isNull() && Node == node)
           best = search;
      }
   }

   if (best.isNull() && create) {
     qDebug("Creating new node for %s@%s", name.ascii(),node.ascii());
     elem = Configuration.Document.createElement("audiodevice");
     elem.setAttribute("name", name);
     elem.setAttribute("node", node);
     Configuration.AudioDevices.appendChild(elem);
     best = elem;
   }
   TR_RET(best);
}




void CCamStreamApp::DumpConfigFile()
{
   TR_ENTER();
   qDebug(Configuration.Document.toString());
   TR_LEAVE();
}

