/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

/**
  \struct SoundAttributes

  \brief A struct for storage of audio device or audio file parameters.


  \par Sample definition

  A sample is defined as one or more audio values that are measured at
  the same point in time. So, for a normal stereo track, a sample
  consists of two values, one for the Left and one for the Right channel.

  A number of successive samples is called a stream.

  Up to 32 channels can be present in an audio sample/stream; each channel
  can be associated with a position (Left, Right, Center, Subwoofer, etc),
  or, in case Position is < Mono, a channel is defined as a single
  unassociated stream (but can still be numbered from 0 to 31). There is no
  pre-defined order for the channel positions; a stereo sample could be
  stored as |Left|Right| or |Right|Left|.

  Each channel must have the same format (8 bit, 16 bit, signed,
  unsigned, etc.).

*/


#include "SoundAttributes.h"

SoundAttributes::SoundAttributes()
{
   Reset();
}

void SoundAttributes::Reset()
{
   SampleRate = 0;
   SampleFormat = Unknown;
   Channels = 0;
   for (int i = 0; i < 32; i++)
      ChannelPosition[i] = NotUsed;
}

void SoundAttributes::SetPreset(Preset p)
{
   switch(p)
   {
     case Speech:
       SampleRate = 8000;
       SampleFormat = Unsigned8;
       Channels = 1;
       ChannelPosition[0] = Mono;
       break;

     case Radio:
       SampleRate = 22025;
       SampleFormat = Signed16;
       Channels = 1;
       ChannelPosition[0] = Mono;
       break;

     case CD:
       SampleRate = 44100;
       SampleFormat = Signed16;
       Channels = 2;
       ChannelPosition[0] = Left;
       ChannelPosition[1] = Right;
       break;

     case DAT:
       SampleRate = 48000;
       SampleFormat = Signed16;
       Channels = 2;
       ChannelPosition[0] = Left;
       ChannelPosition[1] = Right;
       break;

     case Dolby51:
       SampleRate = 44100;
       SampleFormat = Signed16;
       Channels = 6;
       ChannelPosition[0] = Left;
       ChannelPosition[1] = Right;
       ChannelPosition[2] = Center;
       ChannelPosition[3] = LeftRear;
       ChannelPosition[4] = RightRear;
       ChannelPosition[5] = LFE;
       break;
   };
}

/** \brief Returns number of bits for current format.

   Note: all channels in a sample(stream) have the same format.
*/
unsigned int SoundAttributes::FormatWidth() const
{
   switch(SampleFormat)
   {
     case Signed8:
     case Unsigned8:
       return 8;
     case Signed16:
     case Unsigned16:
       return 16;
     case Signed24:
     case Unsigned24:
     case Signed32:
     case Unsigned32:
     case Float:
       return 32;
   }
   return 0;
}

/** \brief Returns number of bytes required for a single sample in current format. */
unsigned int SoundAttributes::BytesPerSample() const
{
   return (FormatWidth() * Channels) / 8;
}


/** Returns true if all parameters and channels are equal */
bool SoundAttributes::operator ==(const struct SoundAttributes &comp) const
{
   if (SampleRate   != comp.SampleRate ||
       SampleFormat != comp.SampleFormat ||
       Channels     != comp.Channels)
     return false;
   for (unsigned int i = 0; i < Channels; i++)
      if (ChannelPosition[i] != comp.ChannelPosition[i])
        return false;
   return true;
}

/** Returns false if any of the parameters or channels are unequal */
bool SoundAttributes::operator !=(const struct SoundAttributes &comp) const
{
   if (SampleRate   != comp.SampleRate ||
       SampleFormat != comp.SampleFormat ||
       Channels     != comp.Channels)
     return true;
   for (unsigned int i = 0; i < Channels; i++)
      if (ChannelPosition[i] != comp.ChannelPosition[i])
        return true;
   return false;
}

SoundAttributes SoundAttributes ::GetFormat(Preset fmt)
{
   SoundAttributes at;

   at.SetPreset(fmt);
   return at;
}
