/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/


/**
  \class CAudioDevice

  \brief Base class for audio capturing/playback from and to an audio card

  This class is meant as a basis for OS/hardware specific audio card 'drivers'.
  Examples are ALSA & OSS for Linux, or Window's support for audio devices.

  \par General design

  ... AudioReader/Writer ...
  ... image ...

  \par How to implement

  In order to make this class useful, you must subclass it and implement
  8 pure virtual functions. 6 Of them are directly related to the sound supprt;
  they are \ref Init(), \ref Exit(),
  \ref StartCapture(), \ref StopCapture(), \ref StartPlayback() and
  \ref StopPlayback().

  \par
  2 Other functions are \ref GetPlaybackPointer() and \ref ShowMixerControls().
  They are described below.

  \par
  Finally, since CAudioDevice is derived from QThread, the subclass must
  implement the run() function; this should be written as a tight loop
  where reading and writing to the actual device handle is done.


  ...timeline diagram


*/


/**

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "AudioDevice.h"

CAudioDevice::CAudioDevice()
	: m_CaptureBuffer(1000000), m_PlaybackBuffer(1000000)
{
   qDebug("CAudioDevice::CAudioDevice()");
   m_Validated = false;
   m_OpenCount = 0;
   m_CaptureCount = m_PlaybackCount = 0;
   m_CurrentSoundAttr.SetPreset(SoundAttributes::Speech);

   connect(&m_CaptureBuffer,  SIGNAL(ReaderAttached()), this, SLOT(EnableCapture()));
   connect(&m_CaptureBuffer,  SIGNAL(ReaderDetached()), this, SLOT(DisableCapture()));
   connect(&m_PlaybackBuffer, SIGNAL(WriterAttached()), this, SLOT(EnablePlayback()));
   connect(&m_PlaybackBuffer, SIGNAL(WriterDetached()), this, SLOT(DisablePlayback()));
}

CAudioDevice::~CAudioDevice()
{
   qDebug("CAudioDevice::~CAudioDevice()");
}



// public

QString CAudioDevice::GetName() const
{
   return m_ShortName;
}

QString CAudioDevice::GetLongName() const
{
   return m_LongName;
}

QString CAudioDevice::GetNodeName() const
{
   return m_NodeName;
}

bool CAudioDevice::IsValid() const
{
   return m_Validated;
}

bool CAudioDevice::Open(Mode m)
{
   if (m_OpenCount++ > 0)
     return true;
   m_Mode = m;
   if (!Init())
     return false;
   emit Opened();
   return true;
}

void CAudioDevice::Close()
{
   if (m_OpenCount == 0) {
     qWarning("CAudioDevice::Close() Count already at 0.");
     return;
   }

   m_OpenCount--;
   if (m_OpenCount == 0) {
     if (m_CaptureCount > 0) {
       StopCapture();
       m_CaptureCount = 0;
     }
     if (m_PlaybackCount > 0) {
       StopPlayback();
       m_PlaybackCount = 0;
     }
     emit Closed();
     Exit();
   }
}

bool CAudioDevice::IsOpen() const
{
   return (m_OpenCount > 0);
}


/**
  \brief Change parameters of capture/playback
  \param attr Struct with new settings
  \return A boolean indicating success or not

  This function will set the card to different hardware parameters for
  capture or playback (or both). It returns true when it was succesful
  in doing so.

  If the new sound attributes are the same as the current ones,
  the function immediately returns true and does not emit a signal.

  In case no stream is being captured or played back, nothing really
  happens; it just sets the initial parameters, emits the
  SoundAttributesChanged signal and returns true.

  Otherwise, the following happens:
  - capture and playback are stopped
  - we wait until all buffers of all readers are drained
  - the new attributes are set
  - capture or playback is restarted
  - if that fails, we reset the old attributes and try again
  - if that fails too, the driver is left in an idle state

  Changes to the sound attributes are emit()ted between stopping and
  starting the audio stream(s). It is possible multiple emits are
  generated with no audio data in between, for example when the (hardware) driver
  rejects the new attributes. Because of this, the receiver always knows the
  exact format of the audio stream when reading from its CAudioRingBufferReader.

*/
bool CAudioDevice::SetSoundAttributes(const SoundAttributes &attr)
{
   SoundAttributes old_attr = m_CurrentSoundAttr;
   bool OK = true;

   if (attr == m_CurrentSoundAttr)
     return true;

   if (m_CaptureCount > 0 || m_PlaybackCount > 0) {
     StopCapture();
     StopPlayback();
     m_CurrentSoundAttr = attr;
     // Emit now, even if we didn't succeed yet
     emit SoundAttributesChanged(m_CurrentSoundAttr);
     if (m_CaptureCount > 0)
       OK = OK && (StartCapture() >= 0);
     if (m_PlaybackCount > 0)
       OK = OK && (StartPlayback() >= 0);
     if (!OK) {
       bool Second = true;
       m_CurrentSoundAttr = old_attr; // restore attributes
       if (m_CaptureCount > 0 && StartCapture() < 0)  { // okay, now we are in serious trouble
         qWarning("CAudioDevice::SetSoundAttributes() Failed to restore old sound parameters!");
         Exit();
         m_CaptureCount = 0;
         Second = false;
       }
       if (m_PlaybackCount > 0 && StartPlayback() < 0) {
         qWarning("CAudioDevice::SetSoundAttributes() Failed to restore old sound parameters!");
         Exit();
         m_PlaybackCount = 0;
         Second = false;
       }
       if (Second)
       {
         emit SoundAttributesChanged(m_CurrentSoundAttr);
       }
     }
   }
   else
   {
     // Just store; we'll need it when capturing starts
     m_CurrentSoundAttr = attr;
     emit SoundAttributesChanged(m_CurrentSoundAttr);
   }
   return OK; // For now
}

SoundAttributes CAudioDevice::GetSoundAttributes() const
{
   return m_CurrentSoundAttr;
}

/*
  \brief Create thread-safe AudioReader buffer
  \return A CAudioRingBufferReader object

  This function creates and returns an CAudioRingBufferReader; this
  is the only way to gain access to the captured data.

  You can create as many readers as you like; each reader works
  independantly (but shares the same buffer). Creating a reader
  automatically enables capturing, and deleting the last reader
  automatically stops capturing.

  You must keep the returned pointer.

*/
CAudioRingBufferReader *CAudioDevice::CreateReader()
{
   return new CAudioRingBufferReader(&m_CaptureBuffer, &m_CurrentSoundAttr);
}

CRingBufferWriter *CAudioDevice::CreateWriter()
{
   return new CRingBufferWriter(&m_PlaybackBuffer);
}


// private slots

void CAudioDevice::EnableCapture()
{
   qDebug("<> CAudioDevice::EnableCapture()");
   if (m_CaptureCount++ == 0) // first enable
     StartCapture();
}

void CAudioDevice::DisableCapture()
{
   qDebug("<> CAudioDevice::DisableCapture()");
   if (m_CaptureCount > 0)
   {
     if (--m_CaptureCount == 0) // last disable
       StopCapture();
   }
}

void CAudioDevice::EnablePlayback()
{
   qDebug("<> CAudioDevice::EnablePlayback()");
   if (m_PlaybackCount++ == 0)
     StartPlayback();
}

void CAudioDevice::DisablePlayback()
{
   qDebug("<> CAudioDevice::DisablePlayback()");
   if (m_PlaybackCount > 0)
   {
     if (--m_PlaybackCount == 0)
       StopPlayback();
   }
}
