#!
#! @ingroup group_sdk_cmake
#!
#! macro camitk_library simplifies the declaration of a library inside CamiTK
#! and is to be used for all inner project libraries
#!
#! The name of the CEP library is deduced from the name of the directory, unless LIBNAME is provided
#!
#! If the library is compiled as shared, it adds the needed library properties
#! If it needs relocation flag fPIC (on some plateforms), this macro should determine
#! if this is the case and specify this flag
#!
#! usage:
#! \code
#! camitk_library(
#!     SHARED|STATIC
#!     [LIBNAME libname]
#!     [PUBLIC]
#!     SOURCES source1.cpp source1.h...
#!     [NEEDS_CEP_LIBRARIES lib1 lib2...]
#!     [NEEDS_ITK]
#!     [NEEDS_LIBXML2]
#!     [NEEDS_XERCESC]
#!     [NEEDS_XSD]
#!     [NEEDS_QT_MODULES]
#!     [DEFINES flag1 flag2 ...]
#!     [EXTERNAL_LIBRARIES lib1 lib2... ]
#!     [INCLUDE_DIRECTORIES dir1 dir2...]
#!     [LINK_DIRECTORIES dir1 dir2...]
#!     [HEADERS_TO_INSTALL]
#!     [CEP_NAME]
#!     [DESCRIPTION]
#!     [EXTRA_TRANSLATE_LANGUAGE]
#!
#! )
#! \endcode
#!
#! \param SHARED                        optional, if the library should be compiled as shared
#! \param SOURCES                       all sources to be included in the library
#! \param INCLUDE_DIRECTORIES           all needed include directories (${CMAKE_CURRENT_SOURCE_DIR} and ${CMAKE_CURRENT_BINARY_DIR} are automatically added)
#! \param EXTERNAL_LIBRARIES            all the libraries to be used during the link (shared or static)
#! \param NEEDS_CEP_LIBRARIES           all the libraries to be used during the link (shared or static)
#! \param NEEDS_ITK                     add this if your component needs ITK.
#! \param NEEDS_LIBXML2                 add this if the CEP library needs libxml2
#! \param NEEDS_XERCESC                 add this if your action / component needs XercesC library
#! \param NEEDS_XSD                     add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
#! \param NEEDS_QT_MODULES              add this if your library depends on Qt modules, such as QtCore, QtGui, QtMultimedia, QtNetwork, QtOpenGL, QtScript, QtScriptTools, QtSql, QtSvg, QtWebkit, QtXml, QtXmlPatterns, QtDeclarative
#! \param LIBNAME                       force the CEP library name to be different from the directory it is in
#! \param LINK_DIRECTORIES              additional directories to use in link_directories(...)
#! \param DEFINES                       list of define flags to add at compilation time
#! \param PUBLIC                        The library is a public library that has to be loaded directly by the operating system.
#!                                      It is generally only needed for SDK library. A public library should not be installed in the
#!                                      lib/${CAMITK_SHORT_VERSION_STRING} but directly in lib/ (on Unix/MacOS) or bin/ (on windows)
#! \param HEADERS_TO_INSTALL            list of header files to install. File would be copied at build and installation time
#!                                      in the ./include/libraries/library-name subdirectory.
#!                                      Note that, if no header files are provided, all header files (*.h) will be install.
#! \param CEP_NAME                      specify the CEP_NAME, which is used to categorized the extension for packaging purpose
#!                                      No CEP_NAME provided will result in default categorization (generic extension).
#! \param DESCRIPTION                   Simple description of the extension. Used for packaging presentation for instance.
#! \param EXTRA_TRANSLATE_LANGUAGE      Additionnal extra language to translate the application

macro(camitk_library)

    #########################################################################
    #                                                                       #
    #   ARGUMENTS PARSING                                                   #
    #                                                                       #
    #   * Use a macro to create the CMAKE variables according to the        #
    #     provided options as input.                                        #
    #                                                                       #
    #########################################################################

    get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} DEFAULT_LIBRARY_NAME)

    parse_arguments(${DEFAULT_LIBRARY_NAME_CMAKE}
        "SOURCES;NEEDS_CEP_LIBRARIES;EXTERNAL_LIBRARIES;INCLUDE_DIRECTORIES;LIBNAME;DEFINES;LINK_DIRECTORIES;HEADERS_TO_INSTALL;CEP_NAME;DESCRIPTION;EXTRA_TRANSLATE_LANGUAGE"  # possible lists
        "SHARED;STATIC;NEEDS_ITK;NEEDS_LIBXML2;NEEDS_XERCESC;NEEDS_XSD;NEEDS_QT_MODULES;PUBLIC" # possible options
        ${ARGN}
    )



    #########################################################################
    #                                                                       #
    #   CREATE CMAKE VARIABLES                                              #
    #                                                                       #
    #   * Create required and useful CMake variables for the macro         #
    #                                                                       #
    #########################################################################

    # TARGET NAME
    # The target name is composed of the following: library-name
    # * library is the suffix
    # * name is deduced from the input folder containing the calling CMakeLists.txt file of the extension.
    # "-" is replaced by "_" if configuring for packaging with NSIS, the program to create a Windows installer.
    if (PACKAGING_NSIS)
        if ("${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME}" STREQUAL "")
            set(LIBRARY_TARGET_NAME library_${DEFAULT_LIBRARY_NAME})
        else()
            set(LIBRARY_TARGET_NAME library_${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME})
        endif()
    else()
        if ("${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME}" STREQUAL "")
            set(LIBRARY_TARGET_NAME library-${DEFAULT_LIBRARY_NAME})
        else()
            set(LIBRARY_TARGET_NAME library-${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME})
        endif()
    endif()



    #########################################################################
    #                                                                       #
    #   INCLUDE DIRECTORIES                                                 #
    #                                                                       #
    #   * Include library directories where to look for header files        #
    #   * Include also additional user provided directories                 #
    #   * These directories are used for compilation step                   #
    #                                                                       #
    #########################################################################

    # BASIC DIRECTORIES
    include_directories(${CAMITK_INCLUDE_DIRECTORIES})
    include_directories(${CMAKE_CURRENT_SOURCE_DIR})
    include_directories(${CMAKE_CURRENT_BINARY_DIR})

    # USER PROVIDED ADDITIONAL DIRECTORIEs
    include_directories(${${DEFAULT_LIBRARY_NAME_CMAKE}_INCLUDE_DIRECTORIES})



    #########################################################################
    #                                                                       #
    #   ADDITIONAL KNOWN LIBRARY DEPENDENCIES                               #
    #                                                                       #
    #   * Look for specific library needed                                  #
    #   * Specific libraries are specified as option with the               #
    #     NEEDS_LIBRARY syntax (see macro syntax for more options)          #
    #   * Backward compatibility : Warn user if using old NEEDS_TOOL syntax #
    #                                                                       #
    #########################################################################

    # Looking for ITK
    set(ITK_LIBRARIES "")
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_ITK)
        find_package(ITK REQUIRED)
        if(ITK_FOUND)
            include(${ITK_USE_FILE})
            set(ITK_VERSION ${ITK_VERSION_MAJOR}.${ITK_VERSION_MINOR}.${ITK_VERSION_PATCH}) #ITK_VERSION is not always set
            message(STATUS "${${DEFAULT_LIBRARY_NAME_CMAKE}_TARGET_NAME}: Found ITK version ${ITK_VERSION}")

            # Now select the ITK libraries according to the ITK version found.
            # ITK 3.x.x (mostly 3.20.1)
            if ((${ITK_VERSION} VERSION_GREATER "3") AND (${ITK_VERSION} VERSION_LESS "4"))
                if (MSVC) # Visual Studio compilers
                    if(MSVC90 OR MSVC10) #ITK 3.x only works with VS 2008 / 2010
                        set(ITK_LIBRARIES
                                debug ITKAlgorithms${CAMITK_DEBUG_POSTFIX}
                                optimized ITKAlgorithms
                                debug ITKBasicFilters${CAMITK_DEBUG_POSTFIX}
                                optimized ITKBasicFilters
                                debug ITKCommon${CAMITK_DEBUG_POSTFIX}
                                optimized ITKCommon
                                debug ITKIO${CAMITK_DEBUG_POSTFIX}
                                optimized ITKIO
                                debug ITKNrrdIO${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcm${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkjpeg12${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkjpeg16${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkopenjpeg${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkpng${CAMITK_DEBUG_POSTFIX}.lib
                                debug itktiff${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkjpeg8${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKSpatialObject${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvnl_inst${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvnl_algo${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkv3p_netlib${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvnl${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvcl${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkv3p_lsqr${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKMetaIO${CAMITK_DEBUG_POSTFIX}.lib
                                debug itksys${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKDICOMParser${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKEXPAT${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKniftiio${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKznz${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkzlib${CAMITK_DEBUG_POSTFIX}.lib
                                debug snmpapi.lib
                                debug rpcrt4.lib
                                debug ws2_32.lib
                                debug comctl32.lib
                                debug wsock32.lib
                                debug opengl32.lib
                            )
                    elseif(MSVC11 OR MSVC12) # Visual Studion 2012/2013
                        message(FATAL_ERROR "It seems you have ITK version 3.x installed. This one is not compatible with Visual Studio 2012 or 2013. Please install ITK 4.3.x or 4.5.x instead")
                    endif()
                elseif(MINGW)
                    message(FATAL_ERROR "CamiTK does not support ITK 3.x libraries for MinGW. You may file a feature request: https://bugzilla-timc.imag.fr/")
                else() #APPLE OR UNIX
                    set(ITK_LIBRARIES ITKBasicFilters ITKCommon ITKIO ITKAlgorithms)
                endif()

            # ITK 4.3.x (mostly 4.3.2)
            elseif((${ITK_VERSION} VERSION_GREATER "4") AND (${ITK_VERSION} VERSION_LESS "4.4"))
                if (MSVC) # Visual Studio compilers
                    if(MSVC90 OR MSVC10) #ITK 3.x only works with VS 2008 / 2010
                        message(FATAL_ERROR "ITK version ${ITK_VERSION} is not compatible with Visual Studio 2008 or 2010. Please install ITK version 4.3.x or 4.5.x")
                    elseif(MSVC11 OR MSVC12) # Visual Studion 2012/2013
                        link_directories(${ITK_DIR}/../..)
                        set(ITK_LIBRARIES
                            # Debug libraries
                            debug ITKCommon-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKNrrdIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmCommon-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkjpeg-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkopenjpeg-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkpng-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itktiff-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKSpatialObjects-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkvnl_algo-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkv3p_netlib-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkvnl-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkvcl-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkv3p_lsqr-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKMetaIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itksys-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKDICOMParser-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKEXPAT-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKniftiio-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKznz-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkzlib-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug snmpapi.lib
                            debug rpcrt4.lib
                            debug ws2_32.lib
                            debug comctl32.lib
                            debug wsock32.lib
                            debug opengl32.lib
                            debug ITKIOGIPL-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIONIFTI-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIONRRD-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOPNG-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmDICT-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmDSED-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmIOD-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmjpeg8-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmjpeg12-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmjpeg16-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkgdcmMSFF-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKgiftiio-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkhdf5_cpp-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkhdf5-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKMetaIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOBioRad-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOStimulate-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOTIFF-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOVTK-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOPNG-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKBiasCorrection-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKBioCell-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKFEM-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOBMP-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOCSV-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOGDCM-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOGE-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOGIPL-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOHDF5-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOImageBase-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOIPL-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOJPEG-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOLSM-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOMesh-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOMeta-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOSiemens-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOSpatialObjects-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOTransformBase-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOTransformHDF5-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOTransformInsightLegacy-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOTransformMatlab-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKIOXML-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKKLMRegionGrowing-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKLabelMap-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKMesh-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKMetaIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkNetlibSlatec-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkopenjpeg-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKOptimizers-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKOptimizersv4-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKPath-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkpng-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKPolynomials-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKQuadEdgeMesh-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKReview-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKStatistics-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkv3p_lsqr-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkv3p_netlib-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug itkvcl-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKVideoCore-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKVideoIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKVTK-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKVNLInstantiation-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            debug ITKWatersheds-4.3${CAMITK_DEBUG_POSTFIX}.lib
                            # Release librariesdebug
                            optimized ITKCommon-4.3.lib
                            optimized ITKNrrdIO-4.3.lib
                            optimized itkgdcmCommon-4.3.lib
                            optimized itkjpeg-4.3.lib
                            optimized itkopenjpeg-4.3.lib
                            optimized itkpng-4.3.lib
                            optimized itktiff-4.3.lib
                            optimized ITKSpatialObjects-4.3.lib
                            optimized itkvnl_algo-4.3.lib
                            optimized itkv3p_netlib-4.3.lib
                            optimized itkvnl-4.3.lib
                            optimized itkvcl-4.3.lib
                            optimized itkv3p_lsqr-4.3.lib
                            optimized ITKMetaIO-4.3.lib
                            optimized itksys-4.3.lib
                            optimized ITKDICOMParser-4.3.lib
                            optimized ITKEXPAT-4.3.lib
                            optimized ITKniftiio-4.3.lib
                            optimized ITKznz-4.3.lib
                            optimized itkzlib-4.3.lib
                            optimized snmpapi.lib
                            optimized rpcrt4.lib
                            optimized ws2_32.lib
                            optimized comctl32.lib
                            optimized wsock32.lib
                            optimized opengl32.lib
                            optimized ITKIOGIPL-4.3.lib
                            optimized ITKIONIFTI-4.3.lib
                            optimized ITKIONRRD-4.3.lib
                            optimized ITKIOPNG-4.3.lib
                            optimized itkgdcmDICT-4.3.lib
                            optimized itkgdcmDSED-4.3.lib
                            optimized itkgdcmIOD-4.3.lib
                            optimized itkgdcmjpeg8-4.3.lib
                            optimized itkgdcmjpeg12-4.3.lib
                            optimized itkgdcmjpeg16-4.3.lib
                            optimized itkgdcmMSFF-4.3.lib
                            optimized ITKgiftiio-4.3.lib
                            optimized itkhdf5_cpp-4.3.lib
                            optimized itkhdf5-4.3.lib
                            optimized ITKMetaIO-4.3.lib
                            optimized ITKIOBioRad-4.3.lib
                            optimized ITKIOStimulate-4.3.lib
                            optimized ITKIOTIFF-4.3.lib
                            optimized ITKIOVTK-4.3.lib
                            optimized ITKIOPNG-4.3.lib
                            optimized ITKBiasCorrection-4.3.lib
                            optimized ITKBioCell-4.3.lib
                            optimized ITKFEM-4.3.lib
                            optimized ITKIOBMP-4.3.lib
                            optimized ITKIOCSV-4.3.lib
                            optimized ITKIOGDCM-4.3.lib
                            optimized ITKIOGE-4.3.lib
                            optimized ITKIOGIPL-4.3.lib
                            optimized ITKIOHDF5-4.3.lib
                            optimized ITKIOImageBase-4.3.lib
                            optimized ITKIOIPL-4.3.lib
                            optimized ITKIOJPEG-4.3.lib
                            optimized ITKIOLSM-4.3.lib
                            optimized ITKIOMesh-4.3.lib
                            optimized ITKIOMeta-4.3.lib
                            optimized ITKIOSiemens-4.3.lib
                            optimized ITKIOSpatialObjects-4.3.lib
                            optimized ITKIOTransformBase-4.3.lib
                            optimized ITKIOTransformHDF5-4.3.lib
                            optimized ITKIOTransformInsightLegacy-4.3.lib
                            optimized ITKIOTransformMatlab-4.3.lib
                            optimized ITKIOXML-4.3.lib
                            optimized ITKKLMRegionGrowing-4.3.lib
                            optimized ITKLabelMap-4.3.lib
                            optimized ITKMesh-4.3.lib
                            optimized ITKMetaIO-4.3.lib
                            optimized itkNetlibSlatec-4.3.lib
                            optimized itkopenjpeg-4.3.lib
                            optimized ITKOptimizers-4.3.lib
                            optimized ITKOptimizersv4-4.3.lib
                            optimized ITKPath-4.3.lib
                            optimized itkpng-4.3.lib
                            optimized ITKPolynomials-4.3.lib
                            optimized ITKQuadEdgeMesh-4.3.lib
                            optimized ITKReview-4.3.lib
                            optimized ITKStatistics-4.3.lib
                            optimized itkv3p_lsqr-4.3.lib
                            optimized itkv3p_netlib-4.3.lib
                            optimized itkvcl-4.3.lib
                            optimized ITKVideoCore-4.3.lib
                            optimized ITKVideoIO-4.3.lib
                            optimized ITKVTK-4.3.lib
                            optimized ITKVNLInstantiation-4.3.lib
                            optimized ITKWatersheds-4.3.lib
                        )
                    endif()
                elseif(MINGW OR APPLE)
                    link_directories(${ITK_DIR}/../..)
                    set(ITK_LIBRARIES
                        ITKCommon-4.3
                        ITKNrrdIO-4.3
                        itkgdcmCommon-4.3
                        itkjpeg-4.3
                        itkopenjpeg-4.3
                        itkpng-4.3
                        itktiff-4.3
                        ITKSpatialObjects-4.3
                        itkvnl_algo-4.3
                        itkv3p_netlib-4.3
                        itkvnl-4.3
                        itkvcl-4.3
                        itkv3p_lsqr-4.3
                        ITKMetaIO-4.3
                        itksys-4.3
                        ITKDICOMParser-4.3
                        ITKEXPAT-4.3
                        ITKniftiio-4.3
                        ITKznz-4.3
                        itkzlib-4.3
                        snmpapi
                        rpcrt4
                        ws2_32
                        comctl32
                        wsock32
                        opengl32
                        ITKIOGIPL-4.3
                        ITKIONIFTI-4.3
                        ITKIONRRD-4.3
                        ITKIOPNG-4.3
                        itkgdcmDICT-4.3
                        itkgdcmDSED-4.3
                        itkgdcmIOD-4.3
                        itkgdcmjpeg8-4.3
                        itkgdcmjpeg12-4.3
                        itkgdcmjpeg16-4.3
                        itkgdcmMSFF-4.3
                        ITKgiftiio-4.3
                        itkhdf5_cpp-4.3
                        itkhdf5-4.3
                        ITKMetaIO-4.3
                        ITKIOBioRad-4.3
                        ITKIOStimulate-4.3
                        ITKIOTIFF-4.3
                        ITKIOVTK-4.3
                        ITKIOPNG-4.3
                        ITKBiasCorrection-4.3
                        ITKBioCell-4.3
                        ITKFEM-4.3
                        ITKIOBMP-4.3
                        ITKIOCSV-4.3
                        ITKIOGDCM-4.3
                        ITKIOGE-4.3
                        ITKIOGIPL-4.3
                        ITKIOHDF5-4.3
                        ITKIOImageBase-4.3
                        ITKIOIPL-4.3
                        ITKIOJPEG-4.3
                        ITKIOLSM-4.3
                        ITKIOMesh-4.3
                        ITKIOMeta-4.3
                        ITKIOSiemens-4.3
                        ITKIOSpatialObjects-4.3
                        ITKIOTransformBase-4.3
                        ITKIOTransformHDF5-4.3
                        ITKIOTransformInsightLegacy-4.3
                        ITKIOTransformMatlab-4.3
                        ITKIOXML-4.3
                        ITKKLMRegionGrowing-4.3
                        ITKLabelMap-4.3
                        ITKMesh-4.3
                        ITKMetaIO-4.3
                        itkNetlibSlatec-4.3
                        itkopenjpeg-4.3
                        ITKOptimizers-4.3
                        ITKOptimizersv4-4.3
                        ITKPath-4.3
                        itkpng-4.3
                        ITKPolynomials-4.3
                        ITKQuadEdgeMesh-4.3
                        ITKReview-4.3
                        ITKStatistics-4.3
                        itkv3p_lsqr-4.3
                        itkv3p_netlib-4.3
                        itkvcl-4.3
                        ITKVideoCore-4.3
                        ITKVideoIO-4.3
                        ITKVTK-4.3
                        ITKVNLInstantiation-4.3
                        ITKWatersheds-4.3
                    )
                else() #Linux
                    set(ITK_LIBRARIES ITKCommon-4.3)
                endif()

            # ITK 4.5.x (mostly 4.5.1)
            elseif((${ITK_VERSION} VERSION_GREATER "4.4") AND (${ITK_VERSION} VERSION_LESS "4.7"))
                if(MSVC)
                    message(FATAL_ERROR "ITK version 4.5.x is not supported yet on Visual Studio compiler. You may file a feature request: https://bugzilla-timc.imag.fr/")
                elseif(MINGW)
                    message(FATAL_ERROR "ITK version 4.5.x is not supported yet for MinGW compiler. You may file a feature request: https://bugzilla-timc.imag.fr/")
                elseif(APPLE)
                    set(ITK_LIBRARIES
                            ITKBiasCorrection
                            ITKBioCell
                            ITKCommon
                            ITKDICOMParser
                            ITKEXPAT
                            ITKFEM
                            ITKGPUAnisotropicSmoothing
                            ITKGPUCommon
                            ITKGPUFiniteDifference
                            ITKGPUImageFilterBase
                            ITKGPUPDEDeformable
                            ITKGPUSmoothing
                            ITKGPUThresholding
                            ITKIOBMP
                            ITKIOBioRad
                            ITKIOCSV
                            ITKIOGDCM
                            ITKIOGE
                            ITKIOGIPL
                            ITKIOHDF5
                            ITKIOIPL
                            ITKIOImageBase
                            ITKIOJPEG
                            ITKIOLSM
                            ITKIOMesh
                            ITKIOMeta
                            ITKIONIFTI
                            ITKIONRRD
                            ITKIOPNG
                            ITKIOSiemens
                            ITKIOSpatialObjects
                            ITKIOStimulate
                            ITKIOTIFF
                            ITKIOTransformBase
                            ITKIOTransformHDF5
                            ITKIOTransformInsightLegacy
                            ITKIOTransformMatlab
                            ITKIOVTK
                            ITKIOXML
                            ITKKLMRegionGrowing
                            ITKLabelMap
                            ITKMesh
                            ITKMetaIO
                            ITKNrrdIO
                            ITKOptimizers
                            ITKOptimizersv4
                            ITKPath
                            ITKPolynomials
                            ITKQuadEdgeMesh
                            ITKReview
                            ITKSpatialObjects
                            ITKStatistics
                            ITKVNLInstantiation
                            ITKVTK
                            ITKVideoCore
                            ITKVideoIO
                            ITKWatersheds
                            ITKgiftiio
                            ITKniftiio
                            ITKznz
                            itkNetlibSlatec
                            itkdouble-conversion
                            itkhdf5
                            itkhdf5_cpp
                            itkjpeg
                            itkopenjpeg
                            itkpng
                            itksys
                            itktiff
                            itkv3p_lsqr
                            itkv3p_netlib
                            itkvcl
                            itkvnl
                            itkvnl_algo
                            itkzlib
                        )
                else() #Linux
                    message(FATAL_ERROR "ITK version 4.5.x is not supported yet for Linux platform. You may file a feature request: https://bugzilla-timc.imag.fr/")
                endif()

            # ITK version > 4.6 (mostly ITK 5.x)
            else()
                message(FATAL_ERROR "CamiTK does not support yet ITK version 5.x. You may file a feature request: https://bugzilla-timc.imag.fr/")
            endif()
        endif()
    endif() #NEEDS ITK

    # LIBXML2
    set(LIBXML2_LIBRARY "")
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_LIBXML2)
        # LibXml2 is required
        find_package(Xml2)
        if (LIBXML2_FOUND)
            add_definitions(${LIBXML2_DEFINITIONS})
            include_directories(${LIBXML2_INCLUDE_DIR})
            set(LIBXML2_LIBRARY ${LIBXML2_LIBRARIES})
        else()
            # most probably win32 or crosscompiling
            message(WARNING "${LIBRARY_TARGET_NAME}: libxml2 required")
        endif()
    endif()

    # XERCES-C
    set(XERCESC_LIBRARIES)
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_XERCESC)
        # XercesC is required
        find_package(XercesC REQUIRED)
        if (XERCESC_FOUND)
            include_directories(${XERCESC_INCLUDE_DIR})
            set(XERCESC_LIBRARIES ${XERCESC_LIBRARY})
        else()
            # most probably win32 or crosscompiling
            message(FATAL_ERROR "${DEFAULT_LIBRARY_NAME}: xerces-c required. Please provide Xerces-C path.")
        endif()
    endif()

    # XSD
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_XSD)
        # XercesC is required
        find_package(XercesC REQUIRED)
        if (XERCESC_FOUND)
            include_directories(${XERCESC_INCLUDE_DIR})
            set(XERCESC_LIBRARIES ${XERCESC_LIBRARY})
            find_package(XSD REQUIRED)
            include_directories(${XSD_INCLUDE_DIR})
        else()
            # most probably win32 or crosscompiling
            message(FATAL_ERROR "${LIBRARY_TARGET_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
        endif()
    endif()

    # ADDITIONAL QT MODULES
    set(QT_MODULES_LIBRARIES)
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_QT_MODULES)
        find_package(Qt4 "4.7" COMPONENTS QtCore QtGui QtXml QtXmlPatterns QtWebkit QtOpenGL QtScript QtSQL QtNetwork REQUIRED)
        if(NOT QT4_FOUND)
            message(SEND_ERROR "${DEFAULT_LIBRARY_NAME} : Failed to find Qt 4.7 or greater. This is needed by CamiTK.")
        endif()

        # CMake will add the Qt modules specific include directories
        include(${QT_USE_FILE})

        # Qt additional modules are required for linking
        set(QT_MODULES_LIBRARIES ${QT_LIBRARIES})
    endif()

    # EXTERNAL DEPENDENCIES
    set(LINKER_EXTERNAL_LIBRARIES)
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_EXTERNAL_LIBRARIES)
        foreach(EXTERNAL_LIBRARY ${${DEFAULT_LIBRARY_NAME_CMAKE}_EXTERNAL_LIBRARIES})
            if (MSVC)
                set(LINKER_EXTERNAL_LIBRARIES ${LINKER_EXTERNAL_LIBRARIES}
                                       debug ${EXTERNAL_LIBRARY}${CAMITK_DEBUG_POSTFIX}
                                       optimized ${EXTERNAL_LIBRARY}
                )
                message(STATUS "LINKER_EXTERNAL_LIBRARIES = ${LINKER_EXTERNAL_LIBRARIES}")
            else()
                set(LINKER_EXTERNAL_LIBRARIES ${LINKER_EXTERNAL_LIBRARIES} ${EXTERNAL_LIBRARY})
            endif()
        endforeach()
    endif()



    #########################################################################
    #                                                                       #
    #   LINK DIRECTORIES                                                    #
    #                                                                       #
    #   * Link directories are used to indicate the compiler where          #
    #     to look for folder containing libraries to link with.             #
    #   * Additional link directories provided by the user                  #
    #                                                                       #
    #########################################################################
    # CAMITK BASIC LIB DIRECTORIES
    link_directories(${CAMITK_LINK_DIRECTORIES})

    # ADDITIONAL LINK DIRECTORIES
    if (NOT "${${DEFAULT_LIBRARY_NAME_CMAKE}_LINK_DIRECTORIES}" STREQUAL "")
        link_directories(${${DEFAULT_LIBRARY_NAME_CMAKE}_LINK_DIRECTORIES})
    endif()



    #########################################################################
    #                                                                       #
    #   TARGET COMPILATION  DEFINITION                                      #
    #                                                                       #
    #   * Additional sources files to consider at compilation (.cpp)        #
    #   * CMake project target definition  depending on library type        #
    #     public / private                                                  #
    #                                                                       #
    #########################################################################
    # CMAKE TARGET DEFINITION DEPENDENDING ON THE LIBRARY TYPE (SHARED or STATIC)
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_SHARED) # shared library
        message(STATUS "Adding shared library: ${LIBRARY_TARGET_NAME}")
        add_library(${LIBRARY_TARGET_NAME} SHARED ${${DEFAULT_LIBRARY_NAME_CMAKE}_SOURCES})
        # prepare the library specific info (SONAME...)
        set(${LIBRARY_TARGET_NAME}_LIBRARY_PROPERTIES ${${LIBRARY_TARGET_NAME}_LIBRARY_PROPERTIES}
            VERSION   "${CAMITK_VERSION_MAJOR}.${CAMITK_VERSION_MINOR}.${CAMITK_VERSION_PATCH}"
            SOVERSION "${CAMITK_VERSION_MAJOR}"
        )
        # set the library specific info (SONAME...)
        set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ${${LIBRARY_TARGET_NAME}_LIBRARY_PROPERTIES} LINK_INTERFACE_LIBRARIES "")
    elseif (${DEFAULT_LIBRARY_NAME_CMAKE}_STATIC) # static library
        message(STATUS "Adding static library: ${LIBRARY_TARGET_NAME}")
        add_library(${LIBRARY_TARGET_NAME} STATIC ${${DEFAULT_LIBRARY_NAME_CMAKE}_SOURCES})
    else()
        message(FATAL_ERROR "In adding static library ${LIBRARY_TARGET_NAME}.\n   Please specify the library type: SHARED or STATIC")
    endif()



    #########################################################################
    #                                                                       #
    #   CAMITK LIBRARIES DEPENDENCIES                                       #
    #                                                                       #
    #   * Add in this section library dependencies to other camitk libraies #
    #     to keep a correct build order (NEEDS_CEP_LIBRARIES).              #
    #                                                                       #
    #########################################################################

    # CEP LIBRARIES DEPENDENCIES
    set(CEP_LIBRARIES)
    set(LIBRARIES_DEPENDENCY_LIST "") #use for generating the project.xml file
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_CEP_LIBRARIES)
        foreach(CEP_LIBRARY_NEEDED ${${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_CEP_LIBRARIES})
             # include directories from build, camitk (local or global install).
            include_directories(${CAMITK_BUILD_INCLUDE_DIR}/libraries/${CEP_LIBRARY_NEEDED})
            include_directories(${CAMITK_INCLUDE_DIR}/libraries/${CEP_LIBRARY_NEEDED})
            # file dependency
            if (MSVC)
                    set(CEP_LIBRARIES ${CEP_LIBRARIES} debug  ${CEP_LIBRARY_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                       optimized ${CEP_LIBRARY_NEEDED}
                    )
            else()
                if(PACKAGING_NSIS)
                    set(CEP_LIBRARIES ${CEP_LIBRARIES} ${CEP_LIBRARY_NEEDED})
                else()
                    set(CEP_LIBRARIES ${CEP_LIBRARIES} ${CEP_LIBRARY_NEEDED})
                endif()
            endif()
            # CMake / CDash dependencies
            if(PACKAGING_NSIS)
                add_dependencies(${LIBRARY_TARGET_NAME} library_${CEP_LIBRARY_NEEDED})
            else()
                add_dependencies(${LIBRARY_TARGET_NAME} library-${CEP_LIBRARY_NEEDED})
                set(LIBRARIES_DEPENDENCY_LIST ${LIBRARIES_DEPENDENCY_LIST} library-${CEP_LIBRARY_NEEDED})
            endif()
        endforeach()
    endif()



    #########################################################################
    #                                                                       #
    #   LINKING                                                             #
    #                                                                       #
    #   * Linking is the last stage of compilation                          #
    #   * Indicate what libraries to use for linking the target             #
    #                                                                       #
    #########################################################################
    # LINKING LIBRARIES
    target_link_libraries(${LIBRARY_TARGET_NAME} ${CEP_LIBRARIES} ${LINKER_EXTERNAL_LIBRARIES} ${LIBXML2_LIBRARY} ${ITK_LIBRARIES} ${QT_MODULES_LIBRARIES} ${XERCESC_LIBRARIES})



    #########################################################################
    #                                                                       #
    #   OUTPUT                                                              #
    #                                                                       #
    #   * Define the output directory (location and name)                   #
    #   * Define the output name of the library                             #
    #   * Add ${CAMITK_DEBUG_POSTFIX} suffix to Debug MSVC built libraries  #
    #                                                                       #
    #########################################################################

    # OUTPUT DIRECTORY LOCATION and NAME depending on the type of the library (PUBLIC or PRIVATE)
    # DEBGUG POSTFIX FOR MSVC
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_PUBLIC) # Public library => build in the bin folder
        if (MSVC)
            # With Visual Studio, public libraries are built in build\bin directory, else it's in build\lib
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                                    LIBRARY_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                                    LIBRARY_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                                    RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                                    RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                                    ARCHIVE_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                                    ARCHIVE_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX})
        else()
            # for xcode generation, the <CONFIG> postfix should also be used, but "a la" UNIX (lib in /lib etc...)
            # no need to check the generator with if(CMAKE_GENERATOR STREQUAL Xcode), as the <CONFIG> postfix should
            # not affect the other unix generators
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_PUBLIC_LIB_DIR}
                                                                    LIBRARY_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PUBLIC_LIB_DIR}
                                                                    LIBRARY_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PUBLIC_LIB_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_PUBLIC_LIB_DIR}
                                                                    RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PUBLIC_LIB_DIR}
                                                                    RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PUBLIC_LIB_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_PUBLIC_LIB_DIR}
                                                                    ARCHIVE_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PUBLIC_LIB_DIR}
                                                                    ARCHIVE_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PUBLIC_LIB_DIR}
            )
        endif()
    else() # Private library => build in lib folder
        if (MSVC)
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX})
        endif()
        set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                                LIBRARY_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                                LIBRARY_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}
        )
        set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                                RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                                RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}
        )
        set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                                ARCHIVE_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                                ARCHIVE_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}
        )
    endif()

    # OUTUPUT LIBRARY NAME (without the prefix "library-").
    string(REGEX REPLACE "^library_|^library-" "" LIBRARY_NAME ${LIBRARY_TARGET_NAME})
    set_target_properties(${LIBRARY_TARGET_NAME}
                           PROPERTIES OUTPUT_NAME ${LIBRARY_NAME}
    )



    #########################################################################
    #                                                                       #
    #   COMPILATION FLAG                                                    #
    #                                                                       #
    #   * Flags are options to give to the compiler                         #
    #   * Add user input flags                                              #
    #   * Add platform specific flags                                       #
    #                                                                       #
    #########################################################################

    # USER INPUT COMPILER FLAG
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_DEFINES)
      foreach (FLAG ${${DEFAULT_LIBRARY_NAME_CMAKE}_DEFINES})
        add_definitions(-D${FLAG})
      endforeach()
    endif()

    # PLATFORM SPECIFIC COMPILER FLAG
    # 64bits and other platform with relocation needs -fPIC
    include(TestCXXAcceptsFlag)
    check_cxx_accepts_flag(-fPIC FPIC_FLAG_ACCEPTED)
    # no need to add -fPIC on mingw, otherwise it generates a warning: -fPIC ignored for target (all code is position independent) [enabled by default]
    # msvc is also accepting the flag, but then produce warning D9002 : ignoring unknown option '-fPIC'   cl
    if(FPIC_FLAG_ACCEPTED AND NOT WIN32)
        set_property(TARGET ${LIBRARY_TARGET_NAME} APPEND PROPERTY COMPILE_FLAGS -fPIC)
    endif()



    #########################################################################
    #                                                                       #
    #   INSTALLATION                                                        #
    #                                                                       #
    #   * When installing the project, header files (.h) and test data are  #
    #     copied into an installation folder to determine.                  #
    #   * Indicate in this section, where to install your project and which #
    #     files to copy into that folder (during local/global installation) #
    #                                                                       #
    #########################################################################

    # FOLDER INSTALLATION
    # Indicate where to install the library, dependending on its property (public / private)
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_PUBLIC) # public library
        if(WIN32)
            # public library -> install in bin folder
            install(TARGETS ${LIBRARY_TARGET_NAME}
                    RUNTIME DESTINATION bin
                    LIBRARY DESTINATION lib
                    ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                    COMPONENT ${LIBRARY_NAME_INSTALL}
                    )
        else()
            # other public libraries -> install in lib folder
            install(TARGETS ${LIBRARY_TARGET_NAME}
                    RUNTIME DESTINATION lib
                    LIBRARY DESTINATION lib
                    ARCHIVE DESTINATION lib
                    COMPONENT ${LIBRARY_NAME_INSTALL}
                    )
        endif()
    else()
         # private library -> install in lib/camitk-version folder
            install(TARGETS ${LIBRARY_TARGET_NAME}
                RUNTIME DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                LIBRARY DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                COMPONENT ${LIBRARY_NAME_INSTALL}
                )
    endif()

    # HEADER FILES (.h) INSTALLATION
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_HEADERS_TO_INSTALL)
        # If headers information provided for installation => install them
        export_headers(${${DEFAULT_LIBRARY_NAME_CMAKE}_HEADERS_TO_INSTALL} COMPONENT ${LIBRARY_TARGET_NAME} GROUP libraries)
    else()
        # if not, install all headers (*.h) by default in the build/include/libraries/library-name directory
        file(GLOB_RECURSE ${DEFAULT_LIBRARY_NAME_CMAKE}_HEADERS ${CMAKE_CURRENT_SOURCE_DIR}/*.h)
        export_headers(${${DEFAULT_LIBRARY_NAME_CMAKE}_HEADERS} COMPONENT ${LIBRARY_TARGET_NAME} GROUP libraries)
    endif()



    #########################################################################
    #                                                                       #
    #   CDASH SUBPROJECT DESCRIPTION                                        #
    #                                                                       #
    #   * Update the XML descriton of the subprojects dependencies          #
    #     for CDash.                                                        #
    #                                                                       #
    #########################################################################
    # CDASH XML SUBPROJECTS DESCRIPTION UPDATE
    if(LIBRARY_TARGET_NAME STREQUAL "library-qtpropertybrowser")
        # library-qtpropertybrowser is a corelib as library-camitkcore depends on it
        camitk_sub_project_add(CORELIB ${LIBRARY_TARGET_NAME} DEPENDENCIES ${LIBRARIES_DEPENDENCY_LIST})
    else()
        camitk_sub_project_add(CEP_LIBRARY ${LIBRARY_TARGET_NAME} DEPENDENCIES ${LIBRARIES_DEPENDENCY_LIST})
    endif()



	#########################################################################
	#                                                                       #
	#   TRANSLATION				                                            #
	#                                                                       #
	#   * CAMITK_TRANSLATIONS contains the list of language to translate 	#
	#	the QString to.														#
	#                                                                       #
	#   * Create the translate.pro file which contains 4 sections:			#
	#		- HEADERS: 		list of .h/.hpp files to look for tr("") QString#		
	#		- SOURCES: 		list of .cpp files to look for tr("") QString	#
	#		- FORMS: 		list of .ui files to look for tr("") QString	#			
	#		- TRANSLATIONS: list of .ts files which use CAMITK_TRANSLATIONS #
	#			to define each .ts file 									#
	#  																		#
	#	* Execute lupdate program to update the .ts files with new QString	#
	#  		found.															#
	#                                                                       #
	#   * Execute lrelease program to create .qm files (binary equivalent of#
	#       .ts files														#
	#                                                                       #
	#   * Create translate.qrc which contains the list of .qm files.		#
	#   * Create the flags.qrc file which contains the list of .png flags	#
	# 		images															#
	#                                                                       #
	#########################################################################
	if(${DEFAULT_LIBRARY_NAME_CMAKE}_EXTRA_TRANSLATE_LANGUAGE)
		camitk_translate(EXTRA_LANGUAGE ${${DEFAULT_LIBRARY_NAME_CMAKE}_EXTRA_TRANSLATE_LANGUAGE})
	else()
		camitk_translate()
	endif()
	
	
	
    #########################################################################
    #                                                                       #
    #   PACKAGING CATEGORIZATION                                            #
    #                                                                       #
    #   * On Windows, when building a package (win32 installer), the        #
    #     install shield wizard proposes you to select which component      #
    #     to install.                                                       #
    #   * Each component to install has a short description following its   #
    #     name to understand its role.                                      #
    #   * This section deals with the categorization and the description    #
    #     of the component in this installer.                               #
    #                                                                       #
    #########################################################################

    # WINDOWS INSTALLER CATEGORIZATION
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_CEP_NAME) # This input variable describes the category
        if (${DEFAULT_LIBRARY_NAME_CMAKE}_CEP_NAME MATCHES "SDK")
            # The default SDK extensions are categorized as "required" and are not "unselectable" by the user at installation time
            cpack_add_component(${LIBRARY_TARGET_NAME}
                                DISPLAY_NAME ${LIBRARY_TARGET_NAME}
                                DESCRIPTION ${${DEFAULT_LIBRARY_NAME_CMAKE}_DESCRIPTION}
                                REQUIRED
                                GROUP SDK
                                )

        else()
            # Extension is selectable for installation in the wizard of the installer
            cpack_add_component(${LIBRARY_TARGET_NAME}
                                DISPLAY_NAME ${LIBRARY_TARGET_NAME}
                                DESCRIPTION ${${DEFAULT_LIBRARY_NAME_CMAKE}_DESCRIPTION}
                                GROUP ${${DEFAULT_LIBRARY_NAME_CMAKE}_CEP_NAME}
                                )
        endif()
    else()
        # Extension if not categorized for packaging presentation
        cpack_add_component(${LIBRARY_TARGET_NAME}
                            DISPLAY_NAME ${LIBRARY_TARGET_NAME}
                            DESCRIPTION ${${DEFAULT_LIBRARY_NAME_CMAKE}_DESCRIPTION}
                            )

    endif()

endmacro()

# Old macro to configure a tool
# Use camitk_library macro instead
# TODO CAMITK_DEPRECATED. This macro is marked as deprecated. It is to be removed in CamiTK 4.0
macro(camitk_tool)
        message(WARNING "Warning: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.txt: camitk_tool macro is deprecated, please usecamitk_library(...) instead (use the same parameters).")
#         string(REPLACE "NEEDS_TOOL" "NEEDS_CEP_LIBRARIES" OBSOLETE_ARGN ${ARGN})
#         message(STATUS "**************************************** ${ARGN} ===>>> ${OBSOLETE_ARGN} ")
        camitk_library(${OBSOLETE_ARGN})
endmacro()
