/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
// CamiTK includes
#include "DicomParser.h"

// GDCM includes
#include <gdcmScanner.h>
#include <gdcmIPPSorter.h>


// --------------- parseDirectory -------------------
QList<DicomSerie*> DicomParser::parseDirectory(const QString & directory) {

    QList<DicomSerie*> dicomSeries;

    // Recursively get all files in the input directory
    gdcm::Directory dir;
    dir.Load(directory.toStdString().c_str(), true);
    const std::vector< std::string > & fileNames = dir.GetFilenames();

    // filter files per image and STUDY
    gdcm::Tag studyUIDTag = gdcm::Tag(0x0020,0x000d);
    // Note : we can put an observer for load bar
    gdcm::Scanner studyScanner;
    studyScanner.AddTag(studyUIDTag);
    studyScanner.Scan(fileNames);

    // retrieve all the studies results
    const std::set< std::string > studyValues = studyScanner.GetValues();

    // for each value, build a new DicomStudyComponent given the associated study filenames
    foreach(std::string studyName, studyValues) {
        // get study associated filenames
        std::vector< std::string > studyFileNames = studyScanner.GetAllFilenamesFromTagToValue(studyUIDTag, studyName.c_str());

        // scan files for series
        gdcm::Scanner seriesScanner;
        gdcm::Tag serieUIDTag = gdcm::Tag(0x0020, 0x000e);
        seriesScanner.AddTag(serieUIDTag);
        seriesScanner.Scan(studyFileNames);
        const std::set< std::string > serieValues = seriesScanner.GetValues();
        foreach(std::string serieName, serieValues) {
            // get file associated with this serie
            std::vector< std::string > serieFileNames = seriesScanner.GetAllFilenamesFromTagToValue(serieUIDTag, serieName.c_str());

            // This filter aims at removing volumic images containing only a few images, for doctors TOPO for instance
            // if ippsorter algorithm returns 0, it means it hasn't been able to sort the files according to their Z spacing
            // It could come from a bad z spacing tolerance or the image files have no z spacing
            gdcm::IPPSorter ippsorter;
            ippsorter.SetComputeZSpacing(true);
            ippsorter.SetZSpacingTolerance(0.01);
            ippsorter.Sort(serieFileNames);
            serieFileNames = ippsorter.GetFilenames();

            // build a DicomSerie object corresponding to what we have found
            if(serieFileNames.size() > 0) {
                DicomSerie* crtSerie = new DicomSerie();
                crtSerie->setAcquisitionDate(DicomParser::getAcquisitionDate(serieFileNames));
                crtSerie->setAcquisitionTime(DicomParser::getAcquisitionTime(serieFileNames));
                crtSerie->setStudyName(DicomParser::getStudyName(serieFileNames));
                crtSerie->setSerieName(DicomParser::getSerieName(serieFileNames));
                crtSerie->setSerieDescription(DicomParser::getSerieDescription(serieFileNames));
                crtSerie->setPatientName(DicomParser::getPatientName(serieFileNames));
                crtSerie->setStdFileNames(serieFileNames);
                crtSerie->setFileNames(stdListOfStringToQt(serieFileNames));

                dicomSeries.append(crtSerie);
            }
        }


    }

    return dicomSeries;

}

// --------------- getAcquisitionDate -------------------
QDate DicomParser::getAcquisitionDate(const std::vector<std::string> & serieFileNames) {
    gdcm::Scanner scanner;
    gdcm::Tag acquisitionDateUIDTag = gdcm::Tag(0x0008, 0x0022);
    scanner.AddTag(acquisitionDateUIDTag);
    scanner.Scan(serieFileNames);

    std::string file = serieFileNames.at(0);
    const char* acquisitionDate = scanner.GetValue(file.c_str(), acquisitionDateUIDTag);

    return QDate::fromString(QString(acquisitionDate), "yyyyMMdd");
}

// --------------- getAcquisitionTime -------------------
QTime DicomParser::getAcquisitionTime(const std::vector<std::string> & serieFileNames) {
    gdcm::Scanner scanner;
    gdcm::Tag acquisitionTimeUIDTag = gdcm::Tag(0x0008, 0x0032);
    scanner.AddTag(acquisitionTimeUIDTag);
    scanner.Scan(serieFileNames);

    std::string file = serieFileNames.at(0);
    const char* acquisitionTime = scanner.GetValue(file.c_str(), acquisitionTimeUIDTag);

    return QTime::fromString(QString(acquisitionTime), "hhmmss");
}

// --------------- getPatientName -------------------
QString DicomParser::getPatientName(const std::vector<std::string> & serieFileNames) {
    gdcm::Scanner scanner;
    gdcm::Tag patientNameUIDTag = gdcm::Tag(0x0010, 0x0010);
    scanner.AddTag(patientNameUIDTag);
    scanner.Scan(serieFileNames);

    std::string file = serieFileNames.at(0);
    const char* patientName = scanner.GetValue(file.c_str(), patientNameUIDTag);
    return QString(patientName);
}


// --------------- getSerieName -------------------
QString DicomParser::getSerieName(const std::vector<std::string> & serieFileNames) {
    gdcm::Scanner scanner;
    gdcm::Tag serieUIDTag = gdcm::Tag(0x0020, 0x000e);
    scanner.AddTag(serieUIDTag);
    scanner.Scan(serieFileNames);

    std::string file = serieFileNames.at(0);
    const char* serieName = scanner.GetValue(file.c_str(), serieUIDTag);
    return QString(serieName);
}

// --------------- getSerieDescription -------------------
QString DicomParser::getSerieDescription(const std::vector<std::string> & serieFileNames) {
    gdcm::Scanner scanner;
    gdcm::Tag serieDescriptionTag = gdcm::Tag(0x0008, 0x103e);
    scanner.AddTag(serieDescriptionTag);
    scanner.Scan(serieFileNames);

    std::string file = serieFileNames.at(0);
    const char* serieDescription = scanner.GetValue(file.c_str(), serieDescriptionTag);
    return QString(serieDescription);
}

// --------------- getStudyName -------------------
QString DicomParser::getStudyName(const std::vector<std::string> & serieFileNames) {
    gdcm::Scanner scanner;
    gdcm::Tag serieUIDTag = gdcm::Tag(0x0020, 0x000e);
    scanner.AddTag(serieUIDTag);
    scanner.Scan(serieFileNames);

    std::string file = serieFileNames.at(0);
    const char* studyName = scanner.GetValue(file.c_str(), serieUIDTag);
    return QString(studyName);
}

// --------------- stdListOfStringToQt -------------------
QList<QString> DicomParser::stdListOfStringToQt(const std::vector<std::string> & inputList) {
    QList<QString> outputFileNames;
    foreach(std::string stdFile, inputList) {
        outputFileNames.append(QString(stdFile.c_str()));
    }

    return outputFileNames;
}

// --------------- qtListOfStringToStd -------------------
std::vector< std::string > DicomParser::qtListOfStringToStd(const QList<QString> & inputList) {
    std::vector< std::string > outputFileNames;
    foreach(QString qtFile, inputList) {
        outputFileNames.push_back(qtFile.toStdString());
    }

    return outputFileNames;
}
















