/*****************************************************************************
* $CAMITK_LICENCE_BEGIN$
*
* CamiTK - Computer Assisted Medical Intervention ToolKit
* (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
*
* Visit http://camitk.imag.fr for more information
*
* This file is part of CamiTK.
*
* CamiTK is free software: you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License version 3
* only, as published by the Free Software Foundation.
*
* CamiTK is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Lesser General Public License version 3 for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
*
* $CAMITK_LICENCE_END$
****************************************************************************/


#include "MeshComponent.h"
#include "InteractiveViewer.h"
#include "Property.h"

//-- Qt stuff
#include <QMessageBox>
#include <QTextStream>
#include <QFileInfo>

//-- vtk stuff
#include <vtkPointSet.h>
#include <vtkUnstructuredGrid.h>
#include <vtkTetra.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkPyramid.h>
#include <vtkCellArray.h>
#include <vtkGenericCell.h>

namespace camitk {
// -------------------- constructor --------------------
MeshComponent::MeshComponent(const QString & file) throw(AbortException) : Component(file, "Mesh", Component::GEOMETRY) {
    pickedCellId = -1;
    pickedPointId = -1;
}

MeshComponent::MeshComponent(vtkSmartPointer<vtkPointSet> aPointSet, const QString &name) : Component("", name, Component::GEOMETRY) {
    initRepresentation(aPointSet);
    setModified();
}

MeshComponent::MeshComponent(Component *parentComponent, vtkSmartPointer<vtkPointSet> aPointSet, const QString &name) : Component(parentComponent, name, Component::GEOMETRY) {
    initRepresentation(aPointSet);
}

// -------------------- destructor --------------------
MeshComponent::~MeshComponent() {
}

// -------------------- initRepresentation --------------------
void MeshComponent::initRepresentation(vtkSmartPointer<vtkPointSet> originalPointSet) {
    pickedCellId = -1;
    pickedPointId = -1;
    // if there is no point set yet, just do nothing
    if (originalPointSet == NULL)
        return;
    // else replace the point set
    if (myGeometry) {
        myGeometry->setPointSet(originalPointSet);
    }
    else {
        myGeometry = new Geometry(this->getName(), originalPointSet);
    }
    // add it in the InteractiveViewer (automatically)
    setVisibility(InteractiveViewer::get3DViewer(), true);

    // initialize the dynamic properties
    initDynamicProperties();
}

// -------------------- cellPicked --------------------
void MeshComponent::cellPicked(vtkIdType cellId, bool ) {
    pickedCellId = cellId;
}

// -------------------- pointPicked --------------------
void MeshComponent::pointPicked(vtkIdType pointId, bool ) {
    pickedPointId = pointId;
}

// -------------------- getPickedCellId --------------------
vtkIdType MeshComponent::getPickedCellId() {
    return pickedCellId;
}

// -------------------- getPickedPointId --------------------
vtkIdType MeshComponent::getPickedPointId() {
    return pickedPointId;
}

// -------------------- initDynamicProperties --------------------
void MeshComponent::initDynamicProperties() {
    vtkIdType count = 0;

    if (getPointSet()!=NULL)
        count = getPointSet()->GetNumberOfPoints();
    Property *nbPoints = new Property("Number Of Points", QVariant(QString("%1").arg(count)), "Number of 3D Points composing the geometry", "");
    nbPoints->setReadOnly(true);
    addProperty(nbPoints);

    if (getPointSet()!=NULL)
        count = getPointSet()->GetNumberOfCells();
    Property *nbCells = new Property("Number Of Cells", QVariant(QString("%1").arg(count)), "Number of Cells composing the geometry", "");
    nbCells->setReadOnly(true);
    addProperty(nbCells);

    if (getPointSet()!=NULL) {
        // add a dynamic property to manage the surface color
        // setProperty("position point #1", QVector3D(1.0,0.0,0.0));
        vtkSmartPointer<vtkGenericCell> cell = vtkGenericCell::New();
        std::map<unsigned char, int> elementsMap;
        std::map<unsigned char, int>::iterator elementsMapIt;

        for (int i=0; i<getPointSet()->GetNumberOfCells(); i++) {
            getPointSet()->GetCell( i, cell );

            if (!elementsMap.count( cell->GetCellType()))
                elementsMap[ cell->GetCellType()] = 0;
            elementsMap[ cell->GetCellType() ]++;

        }

        // the list of all possible cell types is defined in VTKCellType enum of the VTKCellType class
        for (elementsMapIt = elementsMap.begin(); elementsMapIt != elementsMap.end(); elementsMapIt++) {
            Property *cellProp;
            switch(elementsMapIt->first) {
            case VTK_EMPTY_CELL:
                cellProp = new Property("Empty Cells", elementsMapIt->second, "Number Of Empty Cells", "");
                break;
            case VTK_VERTEX:
                cellProp = new Property("Vertex", elementsMapIt->second, "Number Of Vertex Cells", "");
                break;
            case VTK_POLY_VERTEX:
                cellProp = new Property("Edges", elementsMapIt->second, "Number Of Edge Cells", "");
                break;
            case VTK_LINE :
                cellProp = new Property("Lines", elementsMapIt->second, "Number Of Line Cells", "");
                break;
            case VTK_POLY_LINE:
                cellProp = new Property("Polylines", elementsMapIt->second, "Number Of Polylines Cells", "");
                break;
            case VTK_TRIANGLE :
                cellProp = new Property("Triangles", elementsMapIt->second, "Number Of Triangle Cells", "");
                break;
            case VTK_TRIANGLE_STRIP:
                cellProp = new Property("Triangle Strips", elementsMapIt->second, "Number Of Triangle Strip Cells", "");
                break;
            case VTK_POLYGON:
                cellProp = new Property("Polygons", elementsMapIt->second, "Number Of Polygon Cells", "");
                break;
            case VTK_PIXEL:
                cellProp = new Property("Pixels", elementsMapIt->second, "Number Of Pixel Cells", "");
                break;
            case VTK_QUAD:
                cellProp = new Property("Quads", elementsMapIt->second, "Number Of Quad Cells", "");
                break;
            case VTK_TETRA :
                cellProp = new Property("Tetrahedra", elementsMapIt->second, "Number Of Tetrahedral Cells", "");
                break;
            case VTK_VOXEL:
                cellProp = new Property("Voxels", elementsMapIt->second, "Number Of Voxel Cells", "");
                break;
            case VTK_HEXAHEDRON :
                cellProp = new Property("Hexahedra", elementsMapIt->second, "Number Of Hexahedral Cells", "");
                break;
            case VTK_WEDGE :
                cellProp = new Property("Wedges", elementsMapIt->second, "Number Of Wedge Cells", "");
                break;
            case VTK_PYRAMID :
                cellProp = new Property("Pyramids", elementsMapIt->second, "Number Of Pyramid Cells", "");
                break;
            case VTK_PENTAGONAL_PRISM:
                cellProp = new Property("Pentagonal Prisms", elementsMapIt->second, "Number Of Pentagonal Prism Cells", "");
                break;
            case VTK_HEXAGONAL_PRISM:
                cellProp = new Property("Hexagonal Prisms", elementsMapIt->second, "Number Of Hexagonal Prism Cells", "");
                break;
            default :
                cellProp = new Property("Others", elementsMapIt->second, "Number Of <i>Other Type Of Cells</i>. <br/>It can be quadratic isoparametric cells, Cubic isoparametric cells, <br/>convex group of points, higher order cells in parametric form, <br/>higher order cells (see VTKCellType enum for more information)", "");
                break;
            }
            cellProp->setReadOnly(true);
            addProperty(cellProp);
        }
    }

    unsigned long memUsage = 0;
    if (getPointSet()!=NULL)
        memUsage = getPointSet()->GetActualMemorySize();
    Property *memoryUsage = new Property("Size In Memory", QVariant(QString("%1").arg(memUsage)), "Actual size of the data in kilobytes. <br/>This number is valid only after the pipeline has updated. <br/>The memory size returned is guaranteed to be greater than or <br/>equal to the memory required to represent the data<br/> (e.g., extra space in arrays, etc. are not included in the return value).", "Kb");
    memoryUsage->setReadOnly(true);
    addProperty(memoryUsage);
}


}
