/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core image component stuff classes
#include "ImageReconstructionAction.h"
#include "ImageComponent.h"
#include "MeshComponent.h"

// -- Core stuff classes
#include "Application.h"

// -- QT stuff
#include <QLineEdit>
#include <QFileDialog>
#include <QFileInfo>
#include <QMessageBox>

// -- stl stuff
#include <sstream>

// -- vtk filters stuff
#include <vtkImageResample.h>
#include <vtkContourFilter.h>
#include <vtkPolyDataNormals.h>
#include <vtkPolyDataConnectivityFilter.h>
#include <vtkPolyDataWriter.h>
#include <vtkMarchingCubes.h>
#include <vtkPointSet.h>
#include <vtkCallbackCommand.h>

#include <ImageComponent.h>
using namespace camitk;




// -------------------- constructor --------------------
ImageReconstructionAction::ImageReconstructionAction(ActionExtension* extension) : Action(extension) {
    this->setName("Reconstruction");
    this->setDescription("Action use to apply a reconstruction image component");
    this->setComponent("ImageComponent");
    this->setFamily("Reconstruction");
    this->addTag("Reconstruction");

    //-- properties managed by the specific UI
    setProperty("Threshold Value", 0.0); // Grey levels corresponding to the isovalue the isosurface represents

    // Do I keep the largest component ?
    setProperty("Keep Only Largest Connected Component", true); // If true, keep only the largest component

    // Subsample original image ?
    setProperty("Subsample Original Image", true); // If true, the image is subsampled by 1/2^3 to simplify the output object
    setProperty("X Size", 64); // subsampling to apply in the X direction
    setProperty("Y Size", 64); // subsampling to apply in the Y direction
    setProperty("Z Size", 64); // subsampling to apply in the Z direction
}

// -------------------- destructor --------------------
ImageReconstructionAction::~ImageReconstructionAction() {
    if (actionWidget)
        delete actionWidget;
}

// -------------------- getWidget --------------------
QWidget * ImageReconstructionAction::getWidget() {
    if (actionWidget)
        delete actionWidget;

    // rebuild widget from scratch
    actionWidget = new QWidget();
    myComponent = dynamic_cast<ImageComponent *>(getTargets().last());
    init();

    return actionWidget;
}

// -------------------- apply --------------------
Action::ApplyStatus ImageReconstructionAction::apply() {
    build3DModel();
    return SUCCESS;
}

// ---------------------- init ----------------------------
void ImageReconstructionAction::init() {
    ui.setupUi(actionWidget);

    // init thresholds and component name
    ui.lineImagesName->setText(myComponent->getName());
    double medianColor = (myComponent->getMaxColor() - myComponent->getMinColor()) / 2.0;
    setProperty("Threshold Value", medianColor);
    ui.lineThresholdValue->setText(property("Threshold Value").toString());

    connect(ui.PushButton5, SIGNAL(clicked()), this, SLOT(build3DModel()));
    connect(ui.keepComponentcheckBox, SIGNAL(toggled( bool )), this, SLOT(keepComponentcheckBox_toggled(bool)));
    connect(ui.subSamplecheckBox, SIGNAL(toggled( bool )), this, SLOT(subSamplecheckBox_toggled(bool)));

    // init subsample dimensions
    ui.lineXDimension->setText(property("X Size").toString());
    ui.lineYDimension->setText(property("Y Size").toString());
    ui.lineZDimension->setText(property("Z Size").toString());
}

// ---------------------- build3DModel  ----------------------------
void ImageReconstructionAction::build3DModel() {
    // computes the model
    setProperty("Threshold Value", ui.lineThresholdValue->text().toDouble());
    setProperty("X Size", ui.lineXDimension->text().toInt());
    setProperty("Y Size", ui.lineYDimension->text().toInt());
    setProperty("Z Size", ui.lineZDimension->text().toInt());

    vtkSmartPointer<vtkPointSet> resultPointSet = this->getMarchingCubesReconstruction();

    new MeshComponent(resultPointSet, myComponent->getName() + "_mesh");
    Application::refresh();

}

// -------------------- keepComponentcheckBox_toggled --------------------
void ImageReconstructionAction::keepComponentcheckBox_toggled(bool toggled) {
    setProperty("Keep Only Largest Connected Component", toggled);
}

// -------------------- subSamplecheckBox_toggled --------------------
void ImageReconstructionAction::subSamplecheckBox_toggled(bool toggled) {
    setProperty("Subsample Original Image", toggled);
    ui.labelImageSize->setEnabled(toggled);
    ui.label_X_Size->setEnabled(toggled);
    ui.lineXDimension->setEnabled(toggled);
    ui.label_Y_Size->setEnabled(toggled);
    ui.lineYDimension->setEnabled(toggled);
    ui.label_Z_Size->setEnabled(toggled);
    ui.lineZDimension->setEnabled(toggled);
}

// -------------------- getMarchingCubesReconstruction --------------------
vtkSmartPointer<vtkPointSet> ImageReconstructionAction::getMarchingCubesReconstruction() {
    ImageComponent * image = myComponent;
    double isoValue = property("Threshold Value").toDouble();
    bool keepLargestConnectedComponent = property("Keep Only Largest Connected Component").toBool();
    bool subsample = property("Subsample Original Image").toBool();
    int subSampledDimX = property("X Size").toInt();
    int subSampledDimY = property("Y Size").toInt();
    int subSampledDimZ = property("Z Size").toInt();

    Application::showStatusBarMessage("Computing Marching Cube Reconstruction");
    Application::resetProgressBar();
    vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
    progressCallback->SetCallback(&Application::vtkProgressFunction);

    vtkSmartPointer<vtkImageData> originalImageData = image->getImageData();
    vtkSmartPointer<vtkImageData> currentData = originalImageData;

    // For medical volumes, marching cubes generates a large number of triangles.
    // To be practical, we'll use a reduced resolution dataset.
    // For example, we take original 256^3 data and reduce it to 64^2 slices
    // by averaging neighboring pixels twice in the slice plane.
    // We call the resulting dataset quarter since it has 1/4 the resolution of the original data.
    vtkSmartPointer<vtkImageResample> resampleFilter = NULL;

    if (subsample) {
        resampleFilter = vtkSmartPointer<vtkImageResample>::New();
        int * previousDimensions = originalImageData->GetDimensions();

        double x_magnification = 0.0;
        double y_magnification = 0.0;
        double z_magnification = 0.0;
        x_magnification = subSampledDimX / (double) previousDimensions[0];
        y_magnification = subSampledDimY / (double) previousDimensions[1];

        resampleFilter->SetAxisMagnificationFactor(0, x_magnification);
        resampleFilter->SetAxisMagnificationFactor(1, y_magnification);

        if (originalImageData->GetDataDimension() == 3) {
            z_magnification = subSampledDimZ / (double) previousDimensions[2];
            resampleFilter->SetAxisMagnificationFactor(2, z_magnification);
        }

        resampleFilter->SetInput(originalImageData);
        resampleFilter->AddObserver(vtkCommand::ProgressEvent, progressCallback);

        currentData = resampleFilter->GetOutput();

        currentData->Update();
    }

    // The filter we have chosen to use to compute 3D isosurface of the volume image
    // is vtkMarchingcubes. We could also use vtkContourFilter since it will automatically
    // create an instance of vtkMarchingCubes as it delegates to the fastest subclass
    // for a particular dataset type.
    vtkSmartPointer<vtkMarchingCubes> mcSurface = vtkSmartPointer<vtkMarchingCubes>::New();
    mcSurface->SetInput(currentData);
    mcSurface->AddObserver(vtkCommand::ProgressEvent, progressCallback);

    mcSurface->SetValue(0, isoValue);

    // The class vtkPolyDataNormals is used to generate
    // nice surface normals for the data. vtkMarchingCubes can also generate normals,
    // but sometimes better results are achieved when the normals are directly from the
    // surface (vtkPolyDataNormals) versus from the data (vtkMarchingCubes).
    vtkSmartPointer<vtkPolyDataNormals> normalsFilter = vtkSmartPointer<vtkPolyDataNormals>::New();

    //Specify the angle that defines a sharp edge. If the difference in angle
    // across neighboring polygons is greater than this value,
    // the shared edge is considered "sharp".
    normalsFilter->SetFeatureAngle(60.0);

    // Keep largest connected component
    vtkSmartPointer<vtkPolyDataConnectivityFilter> connectivityFilter = NULL;

    if (keepLargestConnectedComponent) {
        connectivityFilter = vtkPolyDataConnectivityFilter::New();
        connectivityFilter->SetInput(mcSurface->GetOutput());
        connectivityFilter->SetExtractionModeToLargestRegion();
        normalsFilter->SetInput(connectivityFilter->GetOutput());
    }
    else
        normalsFilter->SetInput(mcSurface->GetOutput());

    normalsFilter->AddObserver(vtkCommand::ProgressEvent, progressCallback);

    vtkSmartPointer<vtkPointSet> resultPointSet = normalsFilter->GetOutput();
    resultPointSet->Update();

    // One can make efficient data visualization using vtkTriangleFilter
    // and vtkStripper. But first, let us try without...

    // Cleaning...
    originalImageData = NULL;
    currentData = NULL;

    Application::showStatusBarMessage("");
    Application::resetProgressBar();

    return resultPointSet;
}

