/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "ObjComponent.h"

//#include <Geometry.h>
#include <InteractiveViewer.h>
#include <Log.h>

#include <QFileInfo>

#include <vtkOBJReader.h>
#include <vtkProperty.h>

using namespace camitk;

// -------------------- default constructor  --------------------
ObjComponent::ObjComponent(const QString & file) throw(AbortException) : MeshComponent(file) {

    // use the file name without extension as component name
    setName(QFileInfo(file).baseName());

    //-- build the vtkPoinSet from the file
    // create an obj importer to read the OBJ file
    vtkSmartPointer<vtkOBJReader> importer = vtkSmartPointer<vtkOBJReader>::New();
    importer->SetFileName(getFileName().toUtf8().constData());
    importer->Update();

    // instanciate the Geometry
    initRepresentation(importer->GetOutput());

    // initialize the dynamic properties
    initDynamicProperties();
}

// -------------------- initDynamicProperties --------------------
void ObjComponent::initDynamicProperties() {
    // get the default color
    QColor defaultColor = Qt::darkRed;
    setActorColor(Surface, defaultColor.redF(), defaultColor.greenF(), defaultColor.blueF());

    // add a dynamic property to manage the surface color
    setProperty("Surface Color", QColor(defaultColor));
    setProperty("Show Glyph", true);
    setProperty("Show Label", false);

    // a dynamic property can be change at any time
    // the following line does not create a new property, it modifies the existing "Show Glyph" property
    setProperty("Show Glyph", QVariant(false));
    // at anytime, you can also access the dynamic property current value:
    // bool propValue = property("Show Glyph").toBool();
    // you can also convert your value to QString. But beware that for some types of
    // QVariant the conversion is not automatic (e.g. QColor to QString).
    // see the QVariant canConvert<T> and value<T> methods documentation
    CAMITK_INFO("ObjComponent", "initDynamicProperties", "Value of \"Show Glyph\": " << property("Show Glyph").toString().toStdString())
}

// -------------------- getNbPoints --------------------
int ObjComponent::getNbPoints() {
    return  getPointSet()->GetNumberOfPoints();
}

// -------------------- getNbCells --------------------
int ObjComponent::getNbCells() {
    return getPointSet()->GetNumberOfCells();
}

// -------------------- getMemoryUsage --------------------
int ObjComponent::getMemoryUsage() {
    return getPointSet()->GetActualMemorySize();
}

// -------------------- updateProperty  --------------------
void ObjComponent::updateProperty(QString name, QVariant value) {
    // check the modified property and act accordingly
    if (name == "Surface Color") {
        // change the surface color
        QColor newColor = value.value<QColor>();
        setActorColor(Surface, newColor.redF(), newColor.greenF(), newColor.blueF());
    }
    else
        // show glyph or not
        if (name == "Show Glyph") {
            getProp("glyph")->SetVisibility(value.toBool());
        }
        else
            if (name == "Show Label") {
                getProp("label")->SetVisibility(value.toBool());
            }
            else
            // just in case there is an inherited property to manage
            MeshComponent::updateProperty(name, value);

    // refresh the viewers
    refresh();
}

