#!
#! \addtogroup CamiTKMacros
#!
#! macro camitk_library simplifies the declaration of a library inside CamiTK
#! and is to be used for all inner project libraries
#!
#! The name of the CEP library is deduced from the name of the directory, unless LIBNAME is provided
#!
#! If the library is compiled as shared, it adds the needed library properties
#! If it needs relocation flag fPIC (on some plateforms), this macro should determine
#! if this is the case and specify this flag
#!
#! usage:
#! \code
#! camitk_library(
#!     SHARED|STATIC
#!     SOURCES source1.cpp source1.h...
#!     TARGET_LINK_LIBRARIES lib1 lib2...
#!     [INCLUDE_DIRECTORIES dir1 dir2...]
#!     [NEEDS_LIBXML2]
#!     [NEEDS_XERCESC]
#!     [NEEDS_XSD]
#!     [LINK_DIRECTORIES dir1 dir2...]
#!     [LIBNAME libname]
#!     [PUBLIC]
#!     [HEADERS_TO_INSTALL]
#!     [CEP_NAME]
#!     [DESCRIPTION]
#!     
#! )
#! \endcode
#!
#! \param SHARED                        optional, if the library should be compiled as shared
#! \param SOURCES                       all sources to be included in the library
#! \param INCLUDE_DIRECTORIES           all needed include directories (${CMAKE_CURRENT_SOURCE_DIR} and ${CMAKE_CURRENT_BINARY_DIR} are automatically added)
#! \param TARGET_LINK_LIBRARIES         all the libraries to be used during the link
#! \param NEEDS_LIBXML2                 add this if the CEP library needs libxml2
#! \param NEEDS_XERCESC                 add this if your action / component needs XercesC library
#! \param NEEDS_XSD                     add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
#! \param LIBNAME                       force the CEP library name to be different from the directory it is in
#! \param LINK_DIRECTORIES              additional directories to use in link_directories(...)
#! \param PUBLIC                        The library is a public library that has to be loaded directly by the operating system.
#!                                      It is generally only needed for SDK library. A public library should not be installed in the
#!                                      lib/${CAMITK_SHORT_VERSION_STRING} but directly in lib/ (on Unix/MacOS) or bin/ (on windows)
#! \param HEADERS_TO_INSTALL            list of headers to install, if present this will automatically
#!                                      create an "install-COMPONENT_NAMEcomponent" target, that can be used
#!                                      anywhere else to manage dependencies to this component.
#!                                      The headers are installed ${CAMITK_BUILD_INCLUDE_DIR}/COMPONENT_NAME
#!                                      when the target "install-COMPONENT_NAMEcomponent" is called.
#! \param CEP_NAME                      specify the CEP_NAME, which is used to categorized the extension for packaging purpose
#!                                      No CEP_NAME provided will result in default categorization (generic extension).
#! \param DESCRIPTION                   Simple description of the extension. Used for packaging presentation for instance.

macro(camitk_library)
    get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} DEFAULT_LIBRARY_NAME)

    parse_arguments(${DEFAULT_LIBRARY_NAME_CMAKE}
        "SOURCES;TARGET_LINK_LIBRARIES;INCLUDE_DIRECTORIES;LIBNAME;LINK_DIRECTORIES;HEADERS_TO_INSTALL;CEP_NAME;DESCRIPTION"  # possible lists
        "SHARED;STATIC;NEEDS_LIBXML2;NEEDS_XERCESC;NEEDS_XSD;PUBLIC" # possible options
        ${ARGN}
    )

    # if the libname is forced to be different (for string comparison problem, and
    # why there is a double quote here, see http://www.cmake.org/pipermail/cmake/2011-October/046939.html)
    # NSIS requires that cpack component names do not feature space or "-" characters
    if (PACKAGING_NSIS)
        if ("${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME}" STREQUAL "")
            set(LIBRARY_TARGET_NAME library_${DEFAULT_LIBRARY_NAME})
        else()
            set(LIBRARY_TARGET_NAME library_${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME})
        endif()
    else()
        if ("${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME}" STREQUAL "")
            set(LIBRARY_TARGET_NAME library-${DEFAULT_LIBRARY_NAME})
        else()
            set(LIBRARY_TARGET_NAME library-${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME})
        endif()
    endif()
    
    # when installing, remove the prefix 'library' 
    string(REGEX REPLACE "^library_|^library-" "" LIBRARY_NAME ${LIBRARY_TARGET_NAME})

    # includes
    include_directories ( ${${DEFAULT_LIBRARY_NAME_CMAKE}_INCLUDE_DIRECTORIES}
        ${CMAKE_CURRENT_SOURCE_DIR}
        ${CMAKE_CURRENT_BINARY_DIR}
    )

    # Looking for LibXML2.0
    set(LIBXML2_LIBRARIES)
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_LIBXML2)
        # LibXml2 is required
        find_package(Xml2)
        if (LIBXML2_FOUND)
            add_definitions(${LIBXML2_DEFINITIONS})
            include_directories(${LIBXML2_INCLUDE_DIR})
        else()
            # most probably win32 or crosscompiling
            message(WARNING "${LIBRARY_TARGET_NAME}: libxml2 required")
        endif()
    endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_XERCESC)
        # XercesC is required
        find_package(XercesC REQUIRED)
        if (XERCESC_FOUND)
            include_directories(${XERCESC_INCLUDE_DIR})
        else()
            # most probably win32 or crosscompiling
            message(STATUS "${LIBRARY_TARGET_NAME}: xerces-c required")
        endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_XSD)
        # XercesC is required
        find_package(XercesC REQUIRED)
        if (XERCESC_FOUND)
            include_directories(${XERCESC_INCLUDE_DIR})
            find_package(XSD REQUIRED)
            include_directories(${XSD_INCLUDE_DIR})
        else()
            # most probably win32 or crosscompiling
            message(STATUS "${LIBRARY_TARGET_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
        endif()
    endif()

  # some additional link directories are needed
  if (NOT "${${DEFAULT_LIBRARY_NAME_CMAKE}_LINK_DIRECTORIES}" STREQUAL "")
    link_directories(${${DEFAULT_LIBRARY_NAME_CMAKE}_LINK_DIRECTORIES})
  endif()

    # Adding shared or static library
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_SHARED) # shared library
        message(STATUS "Adding shared library: ${LIBRARY_TARGET_NAME}")
        add_library(${LIBRARY_TARGET_NAME} SHARED ${${DEFAULT_LIBRARY_NAME_CMAKE}_SOURCES})
        # prepare the library specific info (SONAME...)
        set(${LIBRARY_TARGET_NAME}_LIBRARY_PROPERTIES ${${LIBRARY_TARGET_NAME}_LIBRARY_PROPERTIES}
            VERSION   "${CAMITK_VERSION_MAJOR}.${CAMITK_VERSION_MINOR}.${CAMITK_VERSION_PATCH}"
            SOVERSION "${CAMITK_VERSION_MAJOR}"
        )
        # set the library specific info (SONAME...)
        set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ${${LIBRARY_TARGET_NAME}_LIBRARY_PROPERTIES} LINK_INTERFACE_LIBRARIES "")
    elseif (${DEFAULT_LIBRARY_NAME_CMAKE}_STATIC) # static library
        message(STATUS "Adding static library: ${LIBRARY_TARGET_NAME}")
        add_library(${LIBRARY_TARGET_NAME} STATIC ${${DEFAULT_LIBRARY_NAME_CMAKE}_SOURCES})
    else()
        message(FATAL_ERROR "In adding static library ${LIBRARY_TARGET_NAME}.\n   Please specify the library type: SHARED or STATIC")
    endif()
  
  # We are building a static or shared library.
  # Specify, regarding the platform where to build the binaries depending on the library type (public / private)
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_PUBLIC) # Public library => build in the bin folder
        if (MSVC)
            # With Visual Studio, public libraries are built in build\bin directory, else it's in build\lib
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                             LIBRARY_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                             LIBRARY_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                             RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                             RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                             ARCHIVE_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                             ARCHIVE_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX})
        else()
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_PUBLIC_LIB_DIR})
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_PUBLIC_LIB_DIR})
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_PUBLIC_LIB_DIR})
        endif()
    else() # Private library => build in lib folder
        if (MSVC)
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                             LIBRARY_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                             LIBRARY_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                             RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                             RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                             ARCHIVE_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}
                                                             ARCHIVE_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}
            )
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX})
        else() 
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR})
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR})
            set_target_properties(${LIBRARY_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR})
        endif()
    endif()
    
    # change the output library name (without the prefix 'library')
    set_target_properties(${LIBRARY_TARGET_NAME}
                           PROPERTIES OUTPUT_NAME ${LIBRARY_NAME}
    )

    # Update XML Project description adding this target as a subproject of the main CamiTK project with
    # its dependencies (in the CMake target point of view)
    camitk_sub_project_add(CEP_LIBRARY ${LIBRARY_TARGET_NAME})

    # 64bits and other platform with relocation needs -fPIC
    include(TestCXXAcceptsFlag)
    check_cxx_accepts_flag(-fPIC FPIC_FLAG_ACCEPTED)
    # no need to add -fPIC on mingw, otherwise it generates a warning: -fPIC ignored for target (all code is position independent) [enabled by default]
    # msvc is also accepting the flag, but then produce warning D9002 : ignoring unknown option '-fPIC'   cl
    if(FPIC_FLAG_ACCEPTED AND NOT WIN32)
        set_property(TARGET ${LIBRARY_TARGET_NAME} APPEND PROPERTY COMPILE_FLAGS -fPIC)
    endif()
    
    #add libraries needed during the link
    target_link_libraries(${LIBRARY_TARGET_NAME} ${${DEFAULT_LIBRARY_NAME_CMAKE}_TARGET_LINK_LIBRARIES} ${LIBXML2_LIBRARIES} ${XERCESC_LIBRARY})
    
    # generate an install target for headers in include/{camitk-version}/libraries
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_HEADERS_TO_INSTALL)
        export_headers(${${DEFAULT_LIBRARY_NAME_CMAKE}_HEADERS_TO_INSTALL} COMPONENT ${LIBRARY_TARGET_NAME} GROUP libraries)
    endif()
    
    # lib installation (depending if this one is public or not)
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_PUBLIC) # public library
        if(WIN32)
            # public library -> install in bin folder
            install(TARGETS ${LIBRARY_TARGET_NAME}
                    RUNTIME DESTINATION bin
                    LIBRARY DESTINATION lib
                    ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                    COMPONENT ${LIBRARY_NAME_INSTALL}
                    )
        else()
            # other public libraries -> install in lib folder
            install(TARGETS ${LIBRARY_TARGET_NAME}
                    RUNTIME DESTINATION lib
                    LIBRARY DESTINATION lib
                    ARCHIVE DESTINATION lib
                    COMPONENT ${LIBRARY_NAME_INSTALL}
                    )
        endif()
    else() 
         # private library -> install in lib/camitk-version folder
            install(TARGETS ${LIBRARY_TARGET_NAME}
                RUNTIME DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                LIBRARY DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}
                COMPONENT ${LIBRARY_NAME_INSTALL}
                )
    endif()
    
    # Categorized the library in the install shield wizard.
    if(${DEFAULT_LIBRARY_NAME_CMAKE}_CEP_NAME)
        if (${DEFAULT_LIBRARY_NAME_CMAKE}_CEP_NAME MATCHES "SDK")
            # The default SDK extensions are categorized as "required" and are not "unselectable" by the user at installation time
            cpack_add_component(${LIBRARY_TARGET_NAME}
                                DISPLAY_NAME ${LIBRARY_TARGET_NAME}
                                DESCRIPTION ${${DEFAULT_LIBRARY_NAME_CMAKE}_DESCRIPTION}
                                REQUIRED
                                GROUP SDK
                                )
        
        else()
            # Extension is selectable for installation in the wizard of the installer
            cpack_add_component(${LIBRARY_TARGET_NAME}
                                DISPLAY_NAME ${LIBRARY_TARGET_NAME}
                                DESCRIPTION ${${DEFAULT_LIBRARY_NAME_CMAKE}_DESCRIPTION}
                                GROUP ${${DEFAULT_LIBRARY_NAME_CMAKE}_CEP_NAME}
                                )
        endif()
    else()
        # Extension if not categorized for packaging presentation
        cpack_add_component(${LIBRARY_TARGET_NAME}
                            DISPLAY_NAME ${LIBRARY_TARGET_NAME}
                            DESCRIPTION ${${DEFAULT_LIBRARY_NAME_CMAKE}_DESCRIPTION}
                            )
    
    endif()
    
endmacro()

# Old macro to configure a tool
# Use camitk_library macro instead
# TODO CAMITK_OBSOLETE. This macro is marked as obsolete. It is to be removed in CamiTK 4.0
macro(camitk_tool)
        message(WARNING "Warning: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.txt: camitk_tool macro is deprecated, please usecamitk_library(...) instead (use the same parameters).")
#         string(REPLACE "NEEDS_TOOL" "NEEDS_CEP_LIBRARIES" OBSOLETE_ARGN ${ARGN})
#         message(STATUS "**************************************** ${ARGN} ===>>> ${OBSOLETE_ARGN} ")
        camitk_library(${OBSOLETE_ARGN})
endmacro()
