#!
#! \addtogroup CamiTKMacros
#!
#! Macro camitk_extension simplifies writing a camitk extension (component, action)
#!
#! The name of the extension is automatically the name of the directory from where this macro
#! is called, unless it is overriden by the TARGET_NAME parameter.
#!
#!  usage:
#! \code
#!    extension(<type>
#!                     [DISABLED]
#!                     [NEEDS_ITK]
#!                     [NEEDS_LIBXML2]
#!                     [NEEDS_OPENCV]
#!                     [NEEDS_IGSTK]
#!                     [NEEDS_XSD]
#!                     [NEEDS_QTXML]
#!                     [NEEDS_GDCM]
#!                     [NEEDS_CEP_LIBRARIES CEPLib1 CEPLib2 ...]
#!                     [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#!                     [INCLUDE_DIRECTORIES dir1 dir2 ...]
#!                     [DEFINES flag1 flag2 ...]
#!                     [EXTERNAL_SOURCES file1 file2 ...]
#!                     [LIBRARIES lib1 lib2 ...]
#!                     [HEADERS_TO_INSTALL header1.h header2.h ...]
#!                     [INSTALL_ALL_HEADERS]
#!                     [TARGET_NAME non-default-targetname]
#!     )
#! \endcode
#!
#! \param <type>                    = The selected type of extension you wish to build.
#                                     possible values : ACTION_EXTENSION or COMPONENT_EXTENSION.
#! \param DISABLED                  = means this is a not a default extension, it will not be compiled automatically
#! \param NEEDS_ITK                 = add this if your component needs itk.
#!                                    Do not forget to add the needed list of itk libraries in the LIBRARIES parameter
#! \param NEEDS_LIBXML2             = add this if your component needs libxml2
#! \param NEEDS_XSD                 = add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
#! \param NEEDS_OPENCV              = add this if your component needs OpenCV
#! \param NEEDS_IGSTK               = add this if your component needs IgsTK
#! \param NEEDS_XERCESC             = add this if your action / component needs XercesC library
#! \param NEEDS_QTXML               = this application needs QtXML
#! \param NEEDS_CEP_LIBRARIES       = list of needed CEP libraries (not external dependencies)
#! \param NEEDS_COMPONENT_EXTENSION = list of needed component extensions
#! \param NEEDS_ACTION_EXTENSION    = list of needed component extensions
#! \param INCLUDE_DIRECTORIES       = additional include directories
#! \param DEFINES                   = list of define flags to add at compilation time
#! \param EXTERNAL_SOURCES          = list of extra source/headers files (external to the current directory)
#!                                    that needed to be added to the SOURCES variable.
#!                                    Note: EXTERNAL_SOURCES are not installed
#! \param LIBRARIES                 = external libraries to add to the link command
#! \param HEADERS_TO_INSTALL        = list of headers to install, if present this will automatically
#!                                    create an "install-COMPONENT_NAMEcomponent" target, that can be used
#!                                    anywhere else to manage dependencies to this component.
#!                                    The headers are installed ${CAMITK_BUILD_INCLUDE_DIR}/COMPONENT_NAME
#!                                    when the target "install-COMPONENT_NAMEcomponent" is called.
#! \param INSTALL_ALL_HEADERS       = install all of the headers (this is the lazy solution, please consider
#!                                    making a list and using HEADERS_TO_INSTALL parameter instead!
#! \param TARGET_NAME               = specify a target name different than the default (default is the action/component directory name)
#!                                    this can be very useful for example when you have action/component directory in your CEP that has
#!                                    the same name as one in the camitk opensource.
#! \param CEP_NAME                  = specify the CEP_NAME, which is used to categorized the extension for packaging purpose
#!                                    No CEP_NAME provided will result in default categorization (generic extension).
#! \param DESCRIPTION               = Simple description of the extension. Used for packaging presentation for instance.
#! \param NEEDS_GDCM                = Add this, if your extension requires GDCM 2.x library
macro(camitk_extension)

  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} EXTENSION_NAME)

  parse_arguments(${EXTENSION_NAME_CMAKE} #prefix
    "NEEDS_TOOL;NEEDS_CEP_LIBRARIES;NEEDS_COMPONENT_EXTENSION;NEEDS_ACTION_EXTENSION;INCLUDE_DIRECTORIES;LIBRARIES;HEADERS_TO_INSTALL;DEFINES;EXTERNAL_SOURCES;TARGET_NAME;CEP_NAME;DESCRIPTION"  # possible lists
    "ACTION_EXTENSION;COMPONENT_EXTENSION;DISABLED;NEEDS_XERCESC;NEEDS_ITK;NEEDS_LIBXML2;NEEDS_XSD;NEEDS_QTXML;NEEDS_OPENCV;NEEDS_IGSTK;INSTALL_ALL_HEADERS;NEEDS_GDCM" # possible options
    ${ARGN}
  )

  # Get the type of extension
  if (${EXTENSION_NAME_CMAKE}_ACTION_EXTENSION)
        set(TYPE_EXTENSION "action")
        string(TOUPPER ${TYPE_EXTENSION} TYPE_EXTENSION_CMAKE)
  elseif(${EXTENSION_NAME_CMAKE}_COMPONENT_EXTENSION)
        set(TYPE_EXTENSION "component")
        string(TOUPPER ${TYPE_EXTENSION} TYPE_EXTENSION_CMAKE)
  endif()

  # if it is the first cmake run, create the extension variable with a correct initial value
  if(NOT ${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL)
    # add option to enable/disable this extension and set it to true by default
    # Building the extension can be disabled by giving the argument DISABLED to the macro
    # or by passing the flag -D${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_DISABLED:BOOL=TRUE
    if(${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_DISABLED)
        set(${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_ENABLED FALSE)
    else()
        set(${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_ENABLED TRUE)
    endif()
    set(${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE} ${${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_ENABLED} CACHE BOOL "Build extension ${EXTENSION_NAME}")
    set(${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL TRUE CACHE INTERNAL "Is variable ${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME} already created?")
  endif()

  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE})

    # check for target name
    if (${EXTENSION_NAME_CMAKE}_TARGET_NAME)
        set(${TYPE_EXTENSION_CMAKE}_OUTPUT_NAME ${${EXTENSION_NAME_CMAKE}_TARGET_NAME})
    else()
        set(${TYPE_EXTENSION_CMAKE}_OUTPUT_NAME ${EXTENSION_NAME})
    endif()
    if (PACKAGING_NSIS)
        # NSIS requires that cpack component names do not feature space or "-" characters
        set(${TYPE_EXTENSION_CMAKE}_TARGET_NAME ${TYPE_EXTENSION}_${${TYPE_EXTENSION_CMAKE}_OUTPUT_NAME})
    else()
        set(${TYPE_EXTENSION_CMAKE}_TARGET_NAME ${TYPE_EXTENSION}-${${TYPE_EXTENSION_CMAKE}_OUTPUT_NAME})
    endif()
    
    message(STATUS "Building extension ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}")

    # include directories
    include_directories(${CAMITK_INCLUDE_DIRECTORIES})
    include_directories(${CMAKE_CURRENT_BINARY_DIR})
    include_directories(${CMAKE_CURRENT_SOURCE_DIR})
    include_directories(${${EXTENSION_NAME_CMAKE}_INCLUDE_DIRECTORIES})

    # check definitions
    if(${EXTENSION_NAME_CMAKE}_DEFINES)
      foreach (FLAG ${${EXTENSION_NAME_CMAKE}_DEFINES})
        add_definitions(-D${FLAG})
      endforeach()
    endif()

    # Check for obsolete NEEDS_TOOL
    # TODO CAMITK_OBSOLETE. This option is marked as obsolete. It is to be removed in CamiTK 4.0
    if(${EXTENSION_NAME_CMAKE}_NEEDS_TOOL)
        message(WARNING "Warning: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.txt: camitk_extension macro option NEEDS_TOOL is deprecated, please use NEEDS_CEP_LIBRARIES instead..")
        set(${EXTENSION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES ${${EXTENSION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES} ${${EXTENSION_NAME_CMAKE}_NEEDS_TOOL})
    endif()

    # check for needed CEP library
    set(CEP_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES)
      foreach(CEP_LIBRARIES_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES})
        string(TOUPPER ${CEP_LIBRARIES_NEEDED} ${CEP_LIBRARIES_NEEDED}_INTERNAL)
        set(LIBRARY_${${CEP_LIBRARIES_NEEDED}_INTERNAL} ON CACHE BOOL "Required by action extension ${EXTENSION_NAME}" FORCE )
        if (MSVC)
            # if (PACKAGING_NSIS)
                # set(CEP_LIBRARIES ${CEP_LIBRARIES}  debug ${CEP_LIBRARIES_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                    # optimized ${CEP_LIBRARIES_NEEDED}
                # )
            # else()
                set(CEP_LIBRARIES ${CEP_LIBRARIES}  debug ${CEP_LIBRARIES_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                    optimized ${CEP_LIBRARIES_NEEDED}
                )
            # endif()
        else()
            if(PACKAGING_NSIS)
                set(CEP_LIBRARIES ${CEP_LIBRARIES} library_${CEP_LIBRARIES_NEEDED})
            else()
                set(CEP_LIBRARIES ${CEP_LIBRARIES} library-${CEP_LIBRARIES_NEEDED})
            endif()
        endif()
        include_directories(${CAMITK_INCLUDE_DIR}/libraries/${CEP_LIBRARIES_NEEDED})
        include_directories(${CAMITK_USER_INCLUDE_DIR}/libraries/${CEP_LIBRARIES_NEEDED})
        include_directories(${CAMITK_BUILD_INCLUDE_DIR}/libraries/${CEP_LIBRARIES_NEEDED})
      endforeach()
    endif()

    # Looking for ITK
    set(ITK_LIBRARIES "")
    if(${EXTENSION_NAME_CMAKE}_NEEDS_ITK)
      # ITK is required
      find_package(ITK REQUIRED)
      include(${ITK_USE_FILE})
      if (MSVC)
        set(ITK_LIBRARIES
            debug ITKBasicFilters${CAMITK_DEBUG_POSTFIX}
            optimized ITKBasicFilters
            debug ITKCommon${CAMITK_DEBUG_POSTFIX}
            optimized ITKCommon
            debug ITKIO${CAMITK_DEBUG_POSTFIX}
            optimized ITKIO
            debug ITKNrrdIO${CAMITK_DEBUG_POSTFIX}.lib
            debug itkgdcm${CAMITK_DEBUG_POSTFIX}.lib
            debug itkjpeg12${CAMITK_DEBUG_POSTFIX}.lib
            debug itkjpeg16${CAMITK_DEBUG_POSTFIX}.lib
            debug itkopenjpeg${CAMITK_DEBUG_POSTFIX}.lib
            debug itkpng${CAMITK_DEBUG_POSTFIX}.lib
            debug itktiff${CAMITK_DEBUG_POSTFIX}.lib
            debug itkjpeg8${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKSpatialObject${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvnl_inst${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvnl_algo${CAMITK_DEBUG_POSTFIX}.lib
            debug itkv3p_netlib${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvnl${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvcl${CAMITK_DEBUG_POSTFIX}.lib
            debug itkv3p_lsqr${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKMetaIO${CAMITK_DEBUG_POSTFIX}.lib
            debug itksys${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKDICOMParser${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKEXPAT${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKniftiio${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKznz${CAMITK_DEBUG_POSTFIX}.lib
            debug itkzlib${CAMITK_DEBUG_POSTFIX}.lib
            debug snmpapi.lib
            debug rpcrt4.lib
            debug ws2_32.lib
            debug comctl32.lib
            debug wsock32.lib
            debug opengl32.lib
        )
        else()
            set(ITK_LIBRARIES ITKBasicFilters ITKCommon ITKIO)
        endif()
    else()
      set(ITK_LIBRARIES "")
    endif()

    # Looking for LibXML2.0
    set(LIBXML2_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_LIBXML2)
      # LibXml2 is required
      find_package(Xml2)
      if (LIBXML2_FOUND)
        add_definitions(${LIBXML2_DEFINITIONS})
        include_directories(${LIBXML2_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: libxml2 required")
      endif()
    endif()

    # looking for OpenCV
    set(OpenCV_LIBRARIES "")
    if(${EXTENSION_NAME_CMAKE}_NEEDS_OPENCV)
        # OpenCV is required
        find_package( OpenCV REQUIRED )
    else ( )
        set(OpenCV_LIBRARIES "")
    endif()

    #looking for IGSTK libraries ?
    set(IGSTK_LIBRARIES "")
    if(${EXTENSION_NAME_CMAKE}_NEEDS_IGSTK)
        find_package(IGSTK REQUIRED)
        include(${IGSTK_USE_FILE})
    else()
        set(IGSTK_LIBRARIES "")
    endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: xerces-c required")
      endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${EXTENSION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()
    
    # Looking for GDCM 2.2.x library
    set(GDCM_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_GDCM)
        if(NOT GDCM_FOUND)
            # Look for GDCM library only if not found (for instance, ITK has already search for it)
            # Calling find_package(GDCM ..) more than once creates CMake errors.
            find_package(GDCM 2.0 REQUIRED)
        endif()
        if(GDCM_FOUND)
            include(${GDCM_USE_FILE})
            set(GDCM_LIBRARIES vtkgdcm)
        else()
            message(ERROR "${EXTENSION_NAME}: GDCM 2.x library required. Please install GDCM.")
        endif()
    endif()
    
    set(QT_LIBRARIES_WITH_QTXML)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_QTXML)
        set (QT_USE_QTXML ON)
        find_package(Qt4 REQUIRED)
        if(QT_USE_FILE)
            include(${QT_USE_FILE})
            else(QT_USE_FILE)
                set(QT_LIBRARIES ${QT_QT_LIBRARY})
            endif(QT_USE_FILE)
            set(QT_LIBRARIES_WITH_QTXML ${QT_LIBRARIES})
    endif()

    set(COMPONENT_EXTENSION_LIBRARIES)
    # check for intra-component extension
    if(${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION)
      foreach(COMPONENT_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} ${COMPONENT_NEEDED}_INTERNAL)
        set(COMPONENT_${${COMPONENT_NEEDED}_INTERNAL} ON CACHE BOOL "Required by extension ${EXTENSION_NAME}" FORCE )
        if (MSVC)
            set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES}
                                              debug ${CAMITK_BUILD_PRIVATE_LIB_DIR}/components/${COMPONENT_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                              optimized ${COMPONENT_NEEDED}
            )
        else()
            set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} ${COMPONENT_NEEDED})
        endif()
        include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
        include_directories(${CAMITK_USER_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
        include_directories(${CAMITK_BUILD_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
      endforeach()
    endif()

    set(ACTION_EXTENSION_LIBRARIES)
    # check for intra-action extension
    if(${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION)
      foreach(ACTION_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ${ACTION_NEEDED}_INTERNAL)
        set(ACTION_${${ACTION_NEEDED}_INTERNAL} ON CACHE BOOL "Required by extension ${EXTENSION_NAME}" FORCE )
        if (MSVC)
            set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES}
                                           debug ${CAMITK_BUILD_PRIVATE_LIB_DIR}/actions/${ACTION_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                           optimized ${ACTION_NEEDED}
            )
        else()
            set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES} ${ACTION_NEEDED})
        endif()
        include_directories(${CAMITK_INCLUDE_DIR}/actions/${ACTION_NEEDED})
        include_directories(${CAMITK_USER_INCLUDE_DIR}/actions/${ACTION_NEEDED})
        include_directories(${CAMITK_BUILD_INCLUDE_DIR}/actions/${ACTION_NEEDED})        
      endforeach()
    endif()

    # check for libs in all known camitk dir + actions subdir (beware of the plateform bin or lib?)
    link_directories(${CAMITK_LINK_DIRECTORIES})

    # get all headers, sources and do what is needed for Qt
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${EXTENSION_NAME_CMAKE})

    # check for external sources
    if(${EXTENSION_NAME_CMAKE}_EXTERNAL_SOURCES)
            set(${EXTENSION_NAME_CMAKE}_SOURCES ${${EXTENSION_NAME_CMAKE}_SOURCES} ${${EXTENSION_NAME_CMAKE}_EXTERNAL_SOURCES})
    endif()

    # build the action extension module (library)
    add_library(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} SHARED ${${EXTENSION_NAME_CMAKE}_SOURCES})
    
    # 64bits and other platform with relocation needs -fPIC
    include(TestCXXAcceptsFlag)
    check_cxx_accepts_flag(-fPIC FPIC_FLAG_ACCEPTED)
    # no need to add -fPIC on mingw, otherwise it generates a warning: -fPIC ignored for target (all code is position independent) [enabled by default]
    # msvc is also accepting the flag, but then produce warning D9002 : ignoring unknown option '-fPIC'   cl
    if(FPIC_FLAG_ACCEPTED AND NOT WIN32)
        set_property(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} APPEND PROPERTY COMPILE_FLAGS -fPIC)
    endif()

    # Update XML Project description adding this target as a subproject of the main CamiTK project with
    # its dependencies (in the CMake target point of view)
    camitk_sub_project_add(${TYPE_EXTENSION_CMAKE} ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} DEPENDENCIES ${CAMITK_CORE_LIBRARIES} ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${CEP_LIBRARIES})

    # Set the libraries required to link the target
    # Any component or action has to be linked with ${CAMITK_CORE_LIBRARIES} and with all its dependencies
    target_link_libraries(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} ${CAMITK_LIBRARIES} ${QT_LIBRARIES_WITH_QTXML} ${CAMITK_CORE_LIBRARIES} ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${ITK_LIBRARIES} ${LIBXML2_LIBRARIES} ${OpenCV_LIBRARIES} ${IGSTK_LIBRARIES} ${XERCESC_LIBRARY} ${GDCM_LIBRARIES} ${CEP_LIBRARIES}  ${${EXTENSION_NAME_CMAKE}_LIBRARIES})

    # dependencies to core (only set for internal build)
    add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} ${CAMITK_CORE_TARGET_LIB_NAME})
    
    # change the output library name
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                           PROPERTIES OUTPUT_NAME ${${TYPE_EXTENSION_CMAKE}_OUTPUT_NAME}
    )
    # Output directory (all extensions are private)
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTIES LIBRARY_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
                                                                            LIBRARY_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
                                                                            LIBRARY_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s        
    )
    # Output directory (for dll plateform, this is still the same, extensions are private)
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
                                                                            RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
                                                                            RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
    )
    # Output directory (for dll plateform, this is still the same, extensions are private)
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTIES ARCHIVE_OUTPUT_DIRECTORY ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
                                                                            ARCHIVE_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
                                                                            ARCHIVE_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_PRIVATE_LIB_DIR}/${TYPE_EXTENSION}s
    )

    if (MSVC)
        set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTIES DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX})
    endif()
    
    # see http://www.cmake.org/pipermail/cmake/2012-April/049889.html
    # target properties (outputname and remove soname)
    #  set_property(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTY NO_SONAME 1)
    # in CEP the version patch might not have been set
    if (NOT CAMITK_VERSION_PATCH)
        set(CAMITK_VERSION_PATCH 0)
    endif()
    
    # set so name
    set(${TYPE_EXTENSION_CMAKE}_LIBRARY_PROPERTIES ${${TYPE_EXTENSION_CMAKE}_LIBRARY_PROPERTIES}
        VERSION   "${CAMITK_VERSION_MAJOR}.${CAMITK_VERSION_MINOR}.${CAMITK_VERSION_PATCH}"
        SOVERSION "${CAMITK_VERSION_MAJOR}"
    )
    # set the library specific info (SONAME...)
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTIES ${${TYPE_EXTENSION_CMAKE}_LIBRARY_PROPERTIES}                       LINK_INTERFACE_LIBRARIES "")

    # CEP libraries dependencies
    if(${EXTENSION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES)
        foreach(CEP_LIBRARIES_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES})
            # TODO add IMPORTED
            if (PACKAGING_NSIS)
                add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} library_${CEP_LIBRARIES_NEEDED})
            else()
                add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} library-${CEP_LIBRARIES_NEEDED})
            endif()
        endforeach()
    endif()

    # Set the Component(s) dependency
    if(${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION)
      foreach(COMPONENT_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} COMPONENT_NEEDED_CMAKE)
        # if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL)
            # TODO add IMPORTED
            add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} component-${COMPONENT_NEEDED})
        endif()
      endforeach()
    endif()

    # Set the Action(s) dependency
    if(${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION)
      foreach(ACTION_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ACTION_NEEDED_CMAKE)
        # if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL)
            # TODO add IMPORTED
            add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} action-${ACTION_NEEDED})
        endif()
      endforeach()
    endif()

    # generate an install target for headers
    if(${EXTENSION_NAME_CMAKE}_HEADERS_TO_INSTALL)
        export_headers(${${EXTENSION_NAME_CMAKE}_HEADERS_TO_INSTALL} COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} GROUP ${TYPE_EXTENSION}s)
    endif()

    # lazy you!
    if(${EXTENSION_NAME_CMAKE}_INSTALL_ALL_HEADERS)
        export_headers(${${EXTENSION_NAME_CMAKE}_HEADERS} COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} GROUP ${TYPE_EXTENSION}s)
    endif()
    
    # lib installation
    install(TARGETS ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
            # TODO always use private lib, even for runtime
            RUNTIME DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/${TYPE_EXTENSION}s
            LIBRARY DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/${TYPE_EXTENSION}s
            ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/${TYPE_EXTENSION}s
            COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
    )
    
    # Categorized the extension in the install shield wizard.
    if(${EXTENSION_NAME_CMAKE}_CEP_NAME)
        if (${EXTENSION_NAME_CMAKE}_CEP_NAME MATCHES "SDK")
            # The default SDK extensions are categorized as "required" and are not "unselectable" by the user at installation time
            cpack_add_component(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                                DISPLAY_NAME ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                                DESCRIPTION ${${EXTENSION_NAME_CMAKE}_DESCRIPTION}
                                REQUIRED
                                GROUP SDK
                                )
        
        else()
            # Extension is selectable for installation in the wizard of the installer
            cpack_add_component(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                                DISPLAY_NAME ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                                DESCRIPTION ${${EXTENSION_NAME_CMAKE}_DESCRIPTION}
                                GROUP ${${EXTENSION_NAME_CMAKE}_CEP_NAME}
                                )
        endif()
    else()
        # Extension if not categorized for packaging presentation
        cpack_add_component(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                            DISPLAY_NAME ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                            DESCRIPTION ${${EXTENSION_NAME_CMAKE}_DESCRIPTION}
                            )
    
    endif()
    
    # testdata files installation
    if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/testdata)
      # Check test data dir directory
      if (NOT EXISTS ${CAMITK_BUILD_TESTDATA_DIR})
        add_custom_command(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                          POST_BUILD
                          COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_BUILD_TESTDATA_DIR}
                          COMMENT "Creating build-time share directory for test data ${CAMITK_BUILD_TESTDATA_DIR}"
                          VERBATIM
        )
      endif()

      # At build time, copy the files to test data directory
      add_custom_command(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E copy_directory testdata ${CAMITK_BUILD_TESTDATA_DIR}
                        COMMENT "Installing build-time test data for ${EXTENSION_NAME}"
                        WORKING_DIRECTORY ${CMAKE_CURRENT_SOURCE_DIR}
                        VERBATIM
      )


      # during installation, copy the files to install directory
      set (TESTDATA_DEST_DIR share/${CAMITK_SHORT_VERSION_STRING}/testdata)
      install(DIRECTORY testdata/
              #DESTINATION share/testdata
              #DESTINATION share/${CAMITK_SHORT_VERSION_STRING}/testdata
              DESTINATION ${TESTDATA_DEST_DIR}
              # COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
              PATTERN ".svn" EXCLUDE
              PATTERN "*~" EXCLUDE
      )
    endif()

  endif()

endmacro()

# Old macro to configure a new action to the framework
# Use camitk_extension macro instead
# TODO CAMITK_OBSOLETE. This macro is marked as obsolete. It is to be removed in CamiTK 4.0
macro(action_extension)
    message(WARNING "Deprecated macro action_extension detected : please use camitk_extension(ACTION_EXTENSION ...) instead.")
    camitk_extension(ACTION_EXTENSION ${ARGN})
endmacro()

# Old macro to configure a new component to the framework
# Use camitk_extension macro instead
# TODO CAMITK_OBSOLETE. This macro is marked as obsolete. It is to be removed in CamiTK 4.0
macro(component_extension)
    message(WARNING "Deprecated macro component_extension detected : please use camitk_extension(COMPONENT_EXTENSION ...) instead.")
    camitk_extension(COMPONENT_EXTENSION ${ARGN})
endmacro()

# TODO write a viewer_extension macro in CamiTK
