#!
#! \addtogroup CamiTKMacros
#!
#! macro camitk_application simplify writing a CMakeLists.txt for CamiTK application extension
#!
#! The name of the application is automatically the name of the directory from where this macro
#! is called prefixed with "camitk-"
#!
#! usage:
#! \code
#! camitk_application(
#!              [DISABLED]
#!              [NEEDS_ITK]
#!              [NEEDS_QTXML]
#!              [NEEDS_PYTHON]
#!              [NEEDS_CEP_LIBRARIES CEPLib1 CEPLib12 ...]
#!              [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#!              [NEEDS_ACTION_EXTENSION ation1 action2 ...]
#!              [ADDITIONAL_SOURCES source.cxx source.cpp ...]
#!              [INCLUDE_DIRECTORIES dir1 dir2 ...]
#!              [LIBRARIES lib1 lib2 ...]
#!     )
#! \endcode
#!
#! \param DISABLED                      means this is a default application is NOT to be compiled automatically
#! \param NEEDS_ITK                     means this application requires ITK to be compiled / run.
#! \param NEEDS_CEP_LIBRARIES           list of needed camitk CEP libraries
#! \param NEEDS_QTXML                   this application needs QtXML
#! \param NEEDS_COMPONENT_EXTENSION     list of needed component extensions
#! \param NEEDS_ACTION_EXTENSION        list of needed component extensions
#! \param ADDITIONAL_SOURCES            list of additional sources (that cannot be automatically found by gather_headers_and_sources macro)
#! \param CEP_NAME                      specify the CEP_NAME, which is used to categorized the application for packaging purpose
#!                                      No CEP_NAME provided will result in default categorization (generic application).
#! \param DESCRIPTION                   Simple description of the application. Used for packaging presentation for instance.
#! \param INCLUDE_DIRECTORIES           additional include directories
#! \param LIBRARIES                      external libraries to add to the link command
macro(camitk_application)

  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} APPLICATION_NAME)

  parse_arguments(${APPLICATION_NAME_CMAKE}
    "NEEDS_CEP_LIBRARIES;NEEDS_TOOL;NEEDS_COMPONENT_EXTENSION;NEEDS_ACTION_EXTENSION;ADDITIONAL_SOURCES;CEP_NAME;DESCRIPTION;LIBRARIES;INCLUDE_DIRECTORIES"  # possible lists
    "DISABLED;NEEDS_ITK;NEEDS_QTXML;NEEDS_XSD;NEEDS_XERCESC;NEEDS_PYTHON" # possible options
    ${ARGN}
  )

  # if it is the first cmake run, create the application variable with a correct initial value
  if(NOT APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
    # add option to enable/disable this extension and set it to true by default
    # Building the extension can be disabled by giving the argument DISABLED to the macro
    # or by passing the flag -D${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_DISABLED:BOOL=TRUE
    if(${APPLICATION_NAME_CMAKE}_DISABLED)
        set(APPLICATION_${APPLICATION_NAME_CMAKE}_ENABLED FALSE)
    else()
        set(APPLICATION_${APPLICATION_NAME_CMAKE}_ENABLED TRUE)
    endif()
    set(APPLICATION_${APPLICATION_NAME_CMAKE} ${APPLICATION_${APPLICATION_NAME_CMAKE}_ENABLED} CACHE BOOL "Build application ${APPLICATION_NAME}")
    set(APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL TRUE CACHE INTERNAL "Is variable APPLICATION_${APPLICATION_NAME} already created?")
  endif()
  
  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (APPLICATION_${APPLICATION_NAME_CMAKE})
    message(STATUS "Building application ${APPLICATION_NAME}")
    
    # include the directories where to seek for .h files at compilation time.
    include_directories(${CAMITK_INCLUDE_DIRECTORIES}) 
    include_directories(${CMAKE_CURRENT_BINARY_DIR})
    include_directories(${${APPLICATION_NAME_CMAKE}_INCLUDE_DIRECTORIES})

    # check for needed CEP library
    set(CEP_LIBRARIES)
    # check obsolete (and warn)
    # TODO CAMITK_OBSOLETE. This option is marked as obsolete. It is to be removed in CamiTK 4.0
    if(${APPLICATION_NAME_CMAKE}_NEEDS_TOOL)
        message(WARNING "Warning: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.txt: camitk_application macro option NEEDS_TOOL is deprecated, please use NEEDS_CEP_LIBRARIES instead..")
        set(CEP_LIBRARIES ${CEP_LIBRARIES} ${${APPLICATION_NAME_CMAKE}_NEEDS_TOOL})
    endif()
    
    # check the option
    if(${APPLICATION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES)
      foreach(CEP_LIBRARIES_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES})
        message(STATUS "${CEP_LIBRARIES_NEEDED} required by application ${APPLICATION_NAME}")
        string(TOUPPER ${CEP_LIBRARIES_NEEDED} ${CEP_LIBRARIES_NEEDED}_INTERNAL)
        set(LIBRARY_${${CEP_LIBRARIES_NEEDED}_INTERNAL} ON CACHE BOOL "Required by application ${APPLICATION_NAME}" FORCE )

        # Add the libraries to link against according to the dependencies
        if (MSVC)
            # if (PACKAGING_NSIS)
                # set(CEP_LIBRARIES ${CEP_LIBRARIES}  debug ${CEP_LIBRARIES_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                    # optimized ${CEP_LIBRARIES_NEEDED}
                # )
            # else()
                set(CEP_LIBRARIES ${CEP_LIBRARIES}  debug ${CEP_LIBRARIES_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                    optimized ${CEP_LIBRARIES_NEEDED}
                )
            # endif()
        else()
            if(PACKAGING_NSIS)
                set(CEP_LIBRARIES ${CEP_LIBRARIES} library_${CEP_LIBRARIES_NEEDED})
            else()
                set(CEP_LIBRARIES ${CEP_LIBRARIES} library-${CEP_LIBRARIES_NEEDED})
            endif()
        endif()
        include_directories(${CAMITK_INCLUDE_DIR}/libraries/${CEP_LIBRARIES_NEEDED})
      endforeach()
    endif()

    # Looking for ITK
    set(ITK_LIBRARIES "")
    if(${APPLICATION_NAME_CMAKE}_NEEDS_ITK)
      # ITK is required
      find_package(ITK REQUIRED)
      include(${ITK_USE_FILE})
      if (MSVC)
        set(ITK_LIBRARIES
            debug ITKBasicFilters${CAMITK_DEBUG_POSTFIX}
            optimized ITKBasicFilters
            debug ITKCommon${CAMITK_DEBUG_POSTFIX}
            optimized ITKCommon
            debug ITKIO${CAMITK_DEBUG_POSTFIX}
            optimized ITKIO
            debug ITKNrrdIO${CAMITK_DEBUG_POSTFIX}.lib
            debug itkgdcm${CAMITK_DEBUG_POSTFIX}.lib
            debug itkjpeg12${CAMITK_DEBUG_POSTFIX}.lib
            debug itkjpeg16${CAMITK_DEBUG_POSTFIX}.lib
            debug itkopenjpeg${CAMITK_DEBUG_POSTFIX}.lib
            debug itkpng${CAMITK_DEBUG_POSTFIX}.lib
            debug itktiff${CAMITK_DEBUG_POSTFIX}.lib
            debug itkjpeg8${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKSpatialObject${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvnl_inst${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvnl_algo${CAMITK_DEBUG_POSTFIX}.lib
            debug itkv3p_netlib${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvnl${CAMITK_DEBUG_POSTFIX}.lib
            debug itkvcl${CAMITK_DEBUG_POSTFIX}.lib
            debug itkv3p_lsqr${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKMetaIO${CAMITK_DEBUG_POSTFIX}.lib
            debug itksys${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKDICOMParser${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKEXPAT${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKniftiio${CAMITK_DEBUG_POSTFIX}.lib
            debug ITKznz${CAMITK_DEBUG_POSTFIX}.lib
            debug itkzlib${CAMITK_DEBUG_POSTFIX}.lib
            debug snmpapi.lib
            debug rpcrt4.lib
            debug ws2_32.lib
            debug comctl32.lib
            debug wsock32.lib
            debug opengl32.lib
        )
        else()
            set(ITK_LIBRARIES ITKBasicFilters ITKCommon ITKIO)
        endif()      
    else()
      set(ITK_LIBRARIES "")
    endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${APPLICATION_NAME}: xerces-c required")
      endif()
    endif()

    set(QT_LIBRARIES_WITH_QTXML)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_QTXML)
        set (QT_USE_QTXML ON)
        find_package(Qt4 REQUIRED)
        if(QT_USE_FILE)
            include(${QT_USE_FILE})
        else(QT_USE_FILE)
            set(QT_LIBRARIES ${QT_QT_LIBRARY})
        endif(QT_USE_FILE)
        set(QT_LIBRARIES_WITH_QTXML ${QT_LIBRARIES})
    endif()

    # Looking for codesynthesis XSD CXX
    if(${APPLICATION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${APPLICATION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()

    # Looking for required component that this application needs
    set(COMPONENT_EXTENSION_LIBRARIES)
    # check for intra-component extension
    if(${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION)
        foreach(COMPONENT_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
            string(TOUPPER ${COMPONENT_NEEDED} ${COMPONENT_NEEDED}_INTERNAL)
            set(COMPONENT_${${COMPONENT_NEEDED}_INTERNAL} ON CACHE BOOL "Required by application ${APPLICATION_NAME}" FORCE )
            if (MSVC)
                set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES}
                                                debug ${CAMITK_BUILD_PRIVATE_LIB_DIR}/components/${COMPONENT_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                optimized ${COMPONENT_NEEDED}
                )
            else()
                set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} ${COMPONENT_NEEDED})
            endif()
            include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
            include_directories(${CAMITK_USER_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
            include_directories(${CAMITK_BUILD_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
        endforeach()
    endif()

    set(ACTION_EXTENSION_LIBRARIES)
    # check for intra-action extension
    if(${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION)
      foreach(ACTION_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ${ACTION_NEEDED}_INTERNAL)
        set(ACTION_${${ACTION_NEEDED}_INTERNAL} ON CACHE BOOL "Required by extension ${APPLICATION_NAME}" FORCE )
        if (MSVC)
            set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES}
                                           debug ${CAMITK_BUILD_PRIVATE_LIB_DIR}/actions/${ACTION_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                           optimized ${ACTION_NEEDED}
            )
        else()
            set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES} ${ACTION_NEEDED})
        endif()
        include_directories(${CAMITK_INCLUDE_DIR}/actions/${ACTION_NEEDED})
        include_directories(${CAMITK_USER_INCLUDE_DIR}/actions/${ACTION_NEEDED})
        include_directories(${CAMITK_BUILD_INCLUDE_DIR}/actions/${ACTION_NEEDED})
      endforeach()
    endif()
    
    # Python dependencies
    set(PYTHON_LIBRARIES "")
    if(${APPLICATION_NAME_CMAKE}_NEEDS_PYTHON)
        message(STATUS "Python needed by ${APPLICATION_NAME}")
        find_package(PythonLibs 2.7 REQUIRED)
        if(PYTHONLIBS_FOUND)
            message(STATUS "Python found (needed by ${APPLICATION_NAME})")
            include_directories(${PYTHON_INCLUDE_DIRS})
            # PYTHON_LIRARIES is automatically and correctly set by find_package
        else()
            message(FATAL_ERROR "Python (2.7 or above) is required by ${APPLICATION_NAME} : please add your python installation dir to your PATH environment variable")
        endif()
    endif()
    
    # set the directories where to seek for static and dynamic libraries at linking time.
    link_directories(${CAMITK_LINK_DIRECTORIES})

    # check for target name
    if (PACKAGING_NSIS)
        # NSIS requires that cpack component names do not feature space or "-" characters
        set(APPLICATION_TARGET_NAME application_${APPLICATION_NAME})
    else()
        set(APPLICATION_TARGET_NAME application-${APPLICATION_NAME})
    endif()
    

    # get all headers, sources and do what is needed for Qt
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${APPLICATION_NAME})

    # additional sources
    set(${APPLICATION_NAME}_SOURCES ${${APPLICATION_NAME}_SOURCES} ${${APPLICATION_NAME_CMAKE}_ADDITIONAL_SOURCES})
    
    # add the executable target
    add_executable(${APPLICATION_TARGET_NAME} ${${APPLICATION_NAME}_SOURCES})

    # Update XML Project description adding this target as a subproject of the main CamiTK project with
    # its dependencies (in the CMake target point of view)
    camitk_sub_project_add(APPLICATION ${APPLICATION_TARGET_NAME} DEPENDENCIES ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${CAMITK_CORE_LIB_NAME} ${CEP_LIBRARIES})

    # target properties
    if (MSVC)
        set_target_properties(${APPLICATION_TARGET_NAME}
                              PROPERTIES OUTPUT_NAME camitk-${APPLICATION_NAME}
                              DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX}
        )
    else()
        set_target_properties(${APPLICATION_TARGET_NAME}
                              PROPERTIES OUTPUT_NAME camitk-${APPLICATION_NAME}
        )
    endif()

    # output directory always bin
    set_target_properties(${APPLICATION_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                                RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                                RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
    )
    
    # Application requires at least camitkcore library
    add_dependencies(${APPLICATION_TARGET_NAME} ${CAMITK_CORE_TARGET_LIB_NAME})

    # CEP library dependencies
    if(${APPLICATION_TARGET_NAME}_NEEDS_CEP_LIBRARIES)
      foreach(CEP_LIBRARIES_NEEDED ${${APPLICATION_TARGET_NAME}_NEEDS_CEP_LIBRARIES})
        if(PACKAGING_NSIS)
            add_dependencies(${APPLICATION_TARGET_NAME} library_${CEP_LIBRARIES_NEEDED})
        else()
            add_dependencies(${APPLICATION_TARGET_NAME} library-${CEP_LIBRARIES_NEEDED})
        endif()
      endforeach()
    endif()

    # Set the Component(s) dependency
    if(${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION)
        foreach(COMPONENT_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
            string(TOUPPER ${COMPONENT_NEEDED} COMPONENT_NEEDED_CMAKE)
            # if this is not true, then the dependencies is from an external build, do not use add_dependencies
            if (APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
                # TODO add IMPORTED
                add_dependencies(${APPLICATION_TARGET_NAME} component-${COMPONENT_NEEDED})
            endif()
        endforeach()
    endif()

    # Set the Action(s) dependency
    if(${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION)
      foreach(ACTION_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ACTION_NEEDED_CMAKE)
        # if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
            # TODO add IMPORTED
            add_dependencies(${APPLICATION_TARGET_NAME} action-${ACTION_NEEDED})
        endif()
      endforeach()
    endif()
    
    # Set the libraries needed to build the application target
    target_link_libraries(${APPLICATION_TARGET_NAME} ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${CAMITK_LIBRARIES} ${QT_LIBRARIES_WITH_QTXML} ${CAMITK_CORE_LIBRARIES} ${CEP_LIBRARIES} ${XERCESC_LIBRARY} ${ITK_LIBRARIES} ${${APPLICATION_NAME_CMAKE}_LIBRARIES} ${PYTHON_LIBRARIES})

    # installation
    install(TARGETS ${APPLICATION_TARGET_NAME}
            RUNTIME DESTINATION bin
            COMPONENT ${APPLICATION_TARGET_NAME}
    )
    
    # Categorized the extension in the install shield wizard.
    if(${APPLICATION_NAME_CMAKE}_CEP_NAME)
        if (${APPLICATION_NAME_CMAKE}_CEP_NAME MATCHES "SDK")
            # The default SDK extensions are categorized as "required" and are not "unselectable" by the user at installation time
            cpack_add_component(${APPLICATION_TARGET_NAME}
                                DISPLAY_NAME ${APPLICATION_TARGET_NAME}
                                DESCRIPTION ${${APPLICATION_NAME_CMAKE}_DESCRIPTION}
                                REQUIRED
                                GROUP SDK
                                )
        
        else()
            # Extension is selectable for installation in the wizard of the installer
            cpack_add_component(${APPLICATION_TARGET_NAME}
                                DISPLAY_NAME ${APPLICATION_TARGET_NAME}
                                DESCRIPTION ${${APPLICATION_NAME_CMAKE}_DESCRIPTION}
                                GROUP ${${APPLICATION_NAME_CMAKE}_CEP_NAME}
                                )
        endif()
    else()
        # Extension if not categorized for packaging presentation
        cpack_add_component(${APPLICATION_TARGET_NAME}
                            DISPLAY_NAME ${APPLICATION_TARGET_NAME}
                            DESCRIPTION ${${APPLICATION_NAME_CMAKE}_DESCRIPTION}
                            )
    
    endif()

    # man page installation
    if(UNIX)
        set(${APPLICATION_NAME_CMAKE}_MAN_PAGE ${CMAKE_CURRENT_SOURCE_DIR}/camitk-${APPLICATION_NAME}.1)
        if (EXISTS ${${APPLICATION_NAME_CMAKE}_MAN_PAGE})
            message(STATUS "Found man page for ${APPLICATION_TARGET_NAME}")
            install(FILES ${${APPLICATION_NAME_CMAKE}_MAN_PAGE}
                    DESTINATION ${CAMITK_APPLICATION_MAN_INSTALL_DIR}
            )
        else()
            # check if .in exists
            if (EXISTS ${${APPLICATION_NAME_CMAKE}_MAN_PAGE}.in)
                message(STATUS "Found man page configuration file for ${APPLICATION_TARGET_NAME}")
                install(CODE "message(STATUS \"Updating ${APPLICATION_NAME} man page (version ${CAMITK_VERSION_MAJOR}.${CAMITK_VERSION_MINOR} as of ${CURRENT_DATE})\")")
                # set variable in sub-cmake shell
                install(CODE "set(APPLICATION_NAME ${APPLICATION_NAME})")
                install(CODE "set(CURRENT_DATE ${CURRENT_DATE})")
                install(CODE "set(CAMITK_VERSION_MAJOR ${CAMITK_VERSION_MAJOR})")
                install(CODE "set(CAMITK_VERSION_MINOR ${CAMITK_VERSION_MINOR})")
                # remove previous version
                install(CODE "execute_process(COMMAND ${CMAKE_COMMAND} -E remove -f ${CMAKE_CURRENT_BINARY_DIR}/camitk-${APPLICATION_NAME}.1)")
                # configure current version
                install(CODE "configure_file(${CMAKE_CURRENT_SOURCE_DIR}/camitk-${APPLICATION_NAME}.1.in ${CMAKE_CURRENT_BINARY_DIR}/camitk-${APPLICATION_NAME}.1 @ONLY)")
                # install file
                install(FILES ${CMAKE_CURRENT_BINARY_DIR}/camitk-${APPLICATION_NAME}.1 DESTINATION ${CAMITK_APPLICATION_MAN_INSTALL_DIR})
            endif()
        endif()

        set(${APPLICATION_NAME_CMAKE}_DESKTOP ${CMAKE_CURRENT_SOURCE_DIR}/camitk-${APPLICATION_NAME}.desktop)
        if (EXISTS ${${APPLICATION_NAME_CMAKE}_DESKTOP})
            message(STATUS "Found desktop file for ${APPLICATION_TARGET_NAME}")
            install(FILES ${${APPLICATION_NAME_CMAKE}_DESKTOP} DESTINATION ${CAMITK_APPLICATION_DESKTOP_INSTALL_DIR})
        endif()
    endif()

endif() #APPLICATION_${APPLICATION_NAME_CMAKE}

endmacro()
