/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file dataMonitoredList
 * @brief Create an object which contained mml data monitored from csv file 
 * @author Johan Sarrazin
 * @date November 27 2012
 */

#include "dataMonitoredList.h"
#include <vector>
#include <cmath>
#include <algorithm>
#include <iostream>


//-----------------constructor-------------------------------------------------------------
dataMonitoredList::dataMonitoredList(std::string monitorName, std::vector< double> resultData)
{
    this->monitorName = monitorName;
    this->resultData = resultData;
}

//-----------------constructor--------------------------
dataMonitoredList::dataMonitoredList(std::string monitorName)
{
    this->monitorName=monitorName;
}

//------------------------getMonitorName--
std::string dataMonitoredList::getMonitorName()
{
    return monitorName;
}

//-------------------------------------getResultData---
std::vector< double > dataMonitoredList::getResultData()
{
    return resultData;
}

//----------------------addResults-----------------------------
void dataMonitoredList::addResults(std::vector< double > results)
{
    // add a vector to the end of the resultData vector.
    if (getResultData().empty()==1){
      resultData = results;
    }else{
      resultData.insert(resultData.end(),results.begin(),results.end());
    }
}

//------------------------getMax---
double dataMonitoredList::getMax()
{
//     std::vector<double>::iterator it=resultData.begin();
//     double max=*it;
    double max = *resultData.begin();
    for (std::vector<double>::iterator it= resultData.begin(); it != resultData.end (); ++it) {
        if (*it>max) {
            max=*it;
        }
    }
    return max;
}

//------------------------getMin--
double dataMonitoredList::getMin()
{
//     std::vector<double>::iterator it=resultData.begin();
//     double min=*it;
    double min = *resultData.begin();
    for (std::vector<double>::iterator it= resultData.begin(); it != resultData.end() ;++it) {
        if (*it<min) {
            min=*it;
        }
    }
    return min;
}

//---------------firstQuartile-----------------------------
double dataMonitoredList::firstQuartile()
{
    // From a sorted vector, the first quartile is the value 
    // separating a vector of data from first 25% to the rest of data.
    std::vector<double> r(resultData.size());
    copy(resultData.begin(),resultData.end(),r.begin());
    //sort the vector
    std::sort (r.begin(),r.end());
    double quart = ceil(r.size()/4) -1;
    return r[quart];
}

//---------------median-----------------------------
double dataMonitoredList::median()
{
    std::vector<double> r(resultData.size());
    copy(resultData.begin(),resultData.end(),r.begin());
    std::sort (r.begin(),r.end());
    double pair= r.size()%2;
    if (pair==0) {
        return (r[r.size()/2]+r[(r.size()/2)-1])/2.0;
    } else {
        return r[r.size()-0.5];
    }
}

//---------------thirdQuartile-----------------------------
double dataMonitoredList::thirdQuartile()
{
    // From a sorted vector, the third quartile is the value 
    // separating a vector of data from last 25% to the rest of data.
    std::vector<double> r(resultData.size());
    copy(resultData.begin(),resultData.end(),r.begin());
    std::sort (r.begin(),r.end());
    double quart3 = ceil(3*(r.size())/4) -1;
    return r[quart3];
}

std::vector< double > dataMonitoredList::outliers()
{
    //WARNING: outlier definition => totally empiric; 
    //For most of software, it is data which are <1.5*Q1 or >1.5*Q3
    std::vector<double> outliers;
    std::vector<double>::iterator ite;
    double IQR = thirdQuartile()-firstQuartile();
    double belowOut = firstQuartile() - (IQR*1.5);
    double aboveOut = thirdQuartile() + (IQR*1.5);
    for (ite = resultData.begin(); ite!= resultData.end() ; ite++){
      if ( (*ite<belowOut) || (*ite>aboveOut)){
	outliers.push_back(*ite);
      }
    }
    return outliers;
}


//------------------destructor----------
dataMonitoredList::~dataMonitoredList()
{

}
