/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file csv2StatsVal
 * @brief class to compute stats from a csv file obtained with mml2Csv
 * @author Johan Sarrazin
 * @date November 27 2012
 */

#ifndef CSV2STATSVAL_H
#define CSV2STATSVAL_H

#include "../../mml2Csv/dataMonitoredList.h"
#include "../csv2Sthing.h"

/**
 * @class csv2StatsVal
 * Class to compute stats from a csv file obtained with mml2Csv to a NEW csv file.
 * @see mml2Csv
 */
class csv2StatsVal : public csv2Sthing{

public:
    /**
     * Constructor
     * @param saveFile path where the stats csv file will be saved
     * @param file path to the input csv file @see mml2Csv
     * @param statsName list of given stats to compute
     * @param results list of @see dataMonitoredList
     */
    csv2StatsVal(std::string saveFile, std::string file, std::list<std::string> statsName, std::list<dataMonitoredList> results);
    /**
     * Constructor
     * @param saveFile path where the stats csv file will be saved
     * @param file path to the input csv file @see mml2Csv
     * @param statsName list of given stats to compute
     */
    csv2StatsVal(std::string saveFile, std::string file, std::list<std::string> statsName);
    
    /// destructor
    virtual ~csv2StatsVal();

    /**
    * generate the rms error of a given data list.
    * WARNING: use this method when you're initial values in mml are not exact (Apparent error is calculated)
    * @param l data vector
    */
    double RMSCalculation(std::vector<double> l);
  
    /**
     * generate the rms error of a given data list.
     * WARNING: use this method when you're initial values in mml are exact (Real Error is calculated)
     * @param l data vector
    */
    double RMSCalculationExact(std::vector<double> l);
 
    /**
     * generate the mean error of a given data list
     * @param l data vector
     */
    double mean(std::vector<double> l);
  
    /**
     * find the max error in a given list
     * @param l the data vector
     */
    double getMax(std::vector<double> l);
   
    /**
     *  find the min error in a given list
     * @param l the data vector
     */
    double getMin(std::vector<double> l);

    /**
     * find the standard deviation of a given list
     * @param l the data vector
     */
    double standardDeviation(std::vector<double> l);

    /**
     * Allows you to write some stats in a csv file @param saveFile to a csv file @param filename.
     * @param filename the name of the csv file which contains data
     * @param saveFile the path where the csv file must be saved
     * @param statsName name of the stats to calculate
     * @param values object type @class dataMonitoredList which contains all data to calculate stats.
     */
    void writeStatsCSV(std::string filename, std::string saveFile, std::list<std::string>statsName,std::list<dataMonitoredList> values );
    
    /**
     * Allows to write computed statistics as a text
     * @param filename the name of the csv file 
     * @param statsName statistics will be computed
     * @param values values which allow to compute statistics  
     */
    void writeStatsText(std::string filename, std::list<std::string>statsName, std::list<dataMonitoredList> values );

    /**
     * Get the results as text 
     * @return results in pre-formated string
     */
    std::string getResultsAsText();
    
    /**
     * Set the string which contains results under a simple pre-formated text
     * @param file the new text
     */
    void setResultsAsText(std::string file);
private:

    /// path where generated csv file will be saved
    std::string saveFile;
    
    /// method to match the name of a stat with its method.
    double statsCalculation(std::string name, std::vector< double > l);
    
    /// string which contains results in csv format
    std::string resultsAsText;

};

#endif // CSV2STATSVAL_H
