/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file csv2StatsVal.cpp
 * @brief class to compute stats from a csv file obtained with mml2Csv
 * @author Johan Sarrazin
 * @date November 27 2012
 */

#include "csv2StatsVal.h"
#include <iostream>
#include <fstream>
#include <string>
#include <complex>
using namespace std;

//------------constructor----------------------------------------------------
csv2StatsVal::csv2StatsVal(string saveFile, string file, list< string > statsName, list< dataMonitoredList > results): csv2Sthing(file, statsName, results)
{
    this->saveFile=saveFile;
    writeStatsCSV(file,saveFile,statsName,results);
    writeStatsText(file,statsName,results);
}

//------------constructor-------------------------------------------------------------------------------------
csv2StatsVal::csv2StatsVal(string saveFile, string file, list< string > statsName): csv2Sthing(file, statsName)
{
    this->saveFile=saveFile;
    writeStatsCSV(file,saveFile,statsName,results);
    writeStatsText(file,statsName,results);
}


//-------------------getMAx------------------------
double csv2StatsVal::getMax(std::vector< double > l)
{
    vector<double>::iterator it = l.begin();
    double max=*it;
    for (it = l.begin (); it != l.end (); ++it) {
        if (*it>max) {
            max=*it;
        }
    }
    return max;
}

//-------------------getMin----------------------
double csv2StatsVal::getMin(std::vector< double > l)
{
    vector<double>::iterator it=l.begin();
    double min=*it;
    for (it=l.begin(); it != l.end() ;++it) {
        if (*it<min) {
            min=*it;
        }
    }
    return min;
}

//------------------mean-------------------------
double csv2StatsVal::mean(std::vector< double > l)
{
    double c=0.0;
    for (vector<double>::iterator it= l.begin(); it != l.end() ;it++) {
        c=c+abs(*it);
    }
    double s = l.size();
    double result = c/s;
    return result;
}

//-------------------RMSCalculation----------------------
double csv2StatsVal::RMSCalculation(std::vector< double > l)
{
    double c=0.0;
    for (vector<double>::iterator it= l.begin(); it != l.end() ;it++) {
        c=c+pow(*it,2.0);
    }
    //the denominator is n-1 because we don't work with real data but with a mean of sample data
    //Difference between real errors and estimated errors (here)
    double s = l.size()-1;
    double result = sqrt(c/s);
    return result;
}

//-------------------RMSCaluclationExact-----------------------
double csv2StatsVal::RMSCalculationExact(std::vector< double > l)
{
    double c=0.0;
    for (vector<double>::iterator it= l.begin(); it != l.end() ;it++) {
        c=c+pow(*it,2.0);
    }
    //the denominator is the size of the list because it's the formula with the real errors and
    //not estimated errors.
    double s = l.size();
    double result = sqrt(c/s);
    return result;
}

//-------------------standardDeviation-----------------------
double csv2StatsVal::standardDeviation(std::vector< double > l)
{
    double m = mean(l);
    double std = 0.0;
    for (vector<double>::iterator it = l.begin();it!=l.end();it++) {
        std+=pow(((*it)-m),2.0);
    }
    return sqrt(std/(l.size()-1));
}

//-----------Destructor------
csv2StatsVal::~csv2StatsVal()
{
}


//-----------------writeStatsCSV---------------------------------------------------------------------------------------------
void csv2StatsVal::writeStatsCSV(string filename, string saveFile, list< string > statsName, list< dataMonitoredList > values)
{
    ofstream myFile;
    if (saveFile.length()==0) {
        saveFile=filename.substr(0,filename.find_last_of(".csv")-3)+"_Stats.csv";
    } else {
        if (!saveFile.find_last_of(".csv")) {
            saveFile+=".csv";
        }
    }
    // open the file to write
    char separator=',';
    myFile.open (const_cast<char*>(saveFile.c_str()));
    //iterator to write the CSV
    list<dataMonitoredList>::iterator listDataResults;
    vector<double>::iterator dataResults;
    list<string>::iterator mathName;

    //write filename used
    myFile <<filename.substr(filename.find_last_of("/")+1)<<separator;
    for (listDataResults=values.begin(); listDataResults!=values.end();listDataResults++) {
        myFile<<(*listDataResults).getMonitorName()<<separator;
    }
    myFile<<endl;

    //write results  in 2D tab
    for (mathName=statsName.begin(); mathName!= statsName.end();mathName++) {
        myFile << (*mathName) << separator;
        for (listDataResults = values.begin() ; listDataResults != values.end(); listDataResults++) {
            myFile << statsCalculation((*mathName),(*listDataResults).getResultData()) << separator;
        }
        myFile<<endl;
    }
    myFile << endl;
    // close after
    myFile.close();
}

//-----------------writeStatsText---------------------------------------------------------------------------
void csv2StatsVal::writeStatsText(string filename,list< string > statsName, list< dataMonitoredList > values)
{
    string myFile="";
    //iterator to write the text
    list<dataMonitoredList>::iterator listDataResults;
    vector<double>::iterator dataResults;
    list<string>::iterator mathName;

    myFile += "File studied : "+filename +"\n\n";
    //write results  in 2D tab
    for (mathName=statsName.begin(); mathName!= statsName.end();mathName++) {
        myFile += (*mathName) +" :\n";
        for (listDataResults = values.begin() ; listDataResults != values.end(); listDataResults++) {
            std::ostringstream s;
            s << statsCalculation((*mathName),(*listDataResults).getResultData());
            myFile +=  "--> " + (*listDataResults).getMonitorName() + " : "+ s.str() +"\n";
        }
        myFile+="\n";
    }
    setResultsAsText(myFile);
}

//------------------ getResultsAsText---
string csv2StatsVal::getResultsAsText()
{
    return resultsAsText;
}

//-----------------setResultsAsText---------------
void csv2StatsVal::setResultsAsText(string file)
{
    this->resultsAsText=file;
}


//-------------------statsCalculation-----------------------------
double csv2StatsVal::statsCalculation(string name,vector< double > l)
{
    if ((name)=="MAX") {
        return getMax(l);
    } else if (name=="MIN") {
        return getMin(l);
    } else if ((name)=="STD") {
        return standardDeviation(l);
    } else if ((name)=="MEAN") {
        return mean(l);
    } else if ((name)=="RMSe") {
        return RMSCalculationExact(l);
    } else if ((name)=="RMS") {
        return RMSCalculation(l);
    }
    cout<<"ERROR: "<<(name)<<" is not yet implemented !"<<endl;
    return 0.0;
}



