/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef REFRENCE_REFERENCE_H
#define REFRENCE_REFERENCE_H

#include <string>

#include "manager/MonitoringManager.h"

class Monitor;

using namespace std;

/**
 * A reference is used to compare to simulated data
 */
class Reference{

  public:
    /// constructor
    Reference(mml::Reference reference, MonitoringManager* monitoringManager);
    /// destructor
    virtual ~Reference();
    
    /**
     * get the nearest reference point in the "target" using "Position" monitor at given time
     * @param pos atom's position (with eventual offset)
     * @param time time to get the Position
     * @param ref a table where positon of the nearest atom in reference will be stored
     * @return true if method succeded (a nearest atom was found in reference)
     */
    bool getNearest(double pos[3], double time, double ref[3]);
    
    /**
     * get the nearest reference point in the "target" using "Position" monitor at final time
     * this is an optimized method to get nearest position where there is only one final time step (real reference) using final pml
     * @param pos atom's position (with eventual offset)
     * @param ref a table where positon of the nearest atom in reference will be stored
     * @return true if method succeded (a nearest atom was found in reference)
     */
    bool getNearest(double pos[3], double ref[3]);
    
    /**
     * get distance to a triagular mesh, the target of the reference must contain triangles.
     * This is an optimized method to get distance to triangular mesh where there is only one final time step (real reference) using final pml
     * @param pos atom's position (with eventual offset)
     * @param dist the distance
     * @return true if method succeded (a triangular mesh was found)
     */
    bool getDistanceToTriangularMesh(double pos[3],double& dist);
    
    /**
     * get the the values of a given monitor which do not depend of time or an atom (e.g. geometrical data)
     * @param type name of the monitor, must be a existing monitor in the mmlOut document
     * @param ref a table where data will be stored
     * @return true if method succeded (an existing monitor named "type" was found)
     */
    bool getMonitoredData(std::string type, double ref[]);
    
    /**
     * get the the values of a given monitor at a given time which do not depend an atom (e.g. volume)
     * if there is no corresponding time in reference, it return data for the first higher time.
     * @param type name of the monitor, must be a existing monitor in the mmlOut document
     * @param time time to get the values
     * @param realTime the nearest time found if realTime=infinity, the nearest is the final state
     * @param ref a table where data will be stored
     * @return true if method succeded
     */
    bool getMonitoredData(std::string type, double time, double &realTime, double ref[]);
    
    /**
     * get the the values of a given monitor at a given time and a given atom's index (e.g. position)
     * @param type name of the monitor, must be a existing monitor in the mmlOut document
     * @param time time to get the values
     * @param index atom's index
     * @param realTime the nearest time found if realTime=infinity, the nearest is the final state
     * @param ref a table where data will be stored
     * @return true if method succeded
     */
    bool getMonitoredData(std::string type, double time, int index, double &realTime, double ref[]);
    
    /// return a string relative to Reference type
    std::string toString();

  private:
    
    /// private structure with monitors for a time step
    struct timeStep{
      double time;
      std::multimap<std::string,Monitor*> monitorsMap;
    };
    
    /// the mmlOut document
    std::string mmlOutFile;
    /// Object in the file generated by xsdcxx
    std::auto_ptr<mml::MonitoringOut> mmlOut;
    /// monitoring manager
    MonitoringManager* monitoringManager;
    /// target
    std::string target;
    /// pml of the reference
    PhysicalModel* pml;
    /** vector which contain alls monitors for each time step
     */
    std::vector<timeStep*> data;
    /// current data index, stored to avoid search from beginning in the data vector
    int CurrentIndex;
    
};

#endif // REFRENCE_REFERENCE_H