/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file csv2Plot.cpp
 * @brief class to plot mml data
 * @author Johan Sarrazin
 * @date November 27 2012
 */

#include "csv2Plot.h"
#include <iostream>
#include <fstream>
#include <complex>
#include <stdlib.h>
#include <vector>
#include "plotWindow.h"
#include <bits/stl_algo.h>

//--------Constructor----------------------------------------------------------------------
csv2Plot::csv2Plot(string file, list< string > whichPlot, list< dataMonitoredList > results): csv2Sthing(file, whichPlot, results)
{
}

//------------constructor----------------------------------------------------
csv2Plot::csv2Plot(string file, list< string > whichPlot): csv2Sthing(file, whichPlot)
{
}

//---------destructor--
csv2Plot::~csv2Plot() {
}

//-------------drawMeMyPlots-----------------------------------------------------------
void csv2Plot::drawMeMyPlots()
{
    list<dataMonitoredList> monitoredValues =results;
    // Iterator
    list<dataMonitoredList>::iterator listDataResults;
    list<string>::iterator plotName;

    // for each type of given plot
    for (plotName=item.begin(); plotName!= item.end();plotName++) {
        //for each monitored value in the csv
        for (listDataResults = monitoredValues.begin() ; listDataResults != monitoredValues.end(); listDataResults++) {
            // plot a graph
            plotWindow* p= new plotWindow;
            if ((*plotName)=="BOX") {
                drawBoxChart(p->getCustomPlot(),(*listDataResults).getMonitorName());

            } else if ((*plotName)=="BARCHART") {
                drawBarChart(p->getCustomPlot(),(*listDataResults).getMonitorName());
            }
            string title = (*listDataResults).getMonitorName()+"-"+(*plotName);
            p->getCustomPlot()->replot();
            p->setWindowTitle(QString(title.c_str()));
            p->show();
        }
    }
}

//--------------drawBarChart-------------------------------------------------------------
bool csv2Plot::drawBarChart(QCustomPlot *plot, string monitor)
{
    list<dataMonitoredList>::iterator listDR =results.begin();
    while (listDR != results.end()) {
        if ((*listDR).getMonitorName() == monitor) {
            // create empty bar chart objects:
            QCPBars *dataToMonitor= new QCPBars(plot->xAxis, plot->yAxis);
            plot->addPlottable(dataToMonitor);
            vector<double> v= (*listDR).getResultData();

            //calculate number of classes k=1+ (10*log(size)/3)
            double k = 1 + (10 * log10((*listDR).getResultData().size())/3);
            double decimal = k-floor(k);
            if (decimal <0.5) {
                k = floor (k);
            } else {
                k = ceil (k);
            }
            //interval inter= max - min
            double inter = (*listDR).getMax() - (*listDR).getMin();
            //amplitude h=inter/k
            //mml precision 10^-7
            double h = (inter/k)+0.0000001;

            QVector<double> xData (k,0);
            QVector<double> yData(k,0);
            QVector<double>::iterator ite;
            double sup,inf = 0.0;
            for (int i =0; i<k ; i++) {
                inf = (*listDR).getMin() + (i*h);
                sup = (*listDR).getMin()+((i+1)*h);
                xData[i]=(inf+sup)/2.0;

                vector<double>::iterator ite;
                for (ite = v.begin(); ite != v.end();ite++) {
                    if ((*ite >= inf) && (*ite < sup)) {
                        yData[i] += 1;
                    }
                }
            }
            dataToMonitor->setName((*listDR).getMonitorName().c_str());
            dataToMonitor->setWidth(h);
            dataToMonitor->setData(xData, yData);

            // setup X-Axis:
            plot->xAxis->setRange((*listDR).getMin()-h, (*listDR).getMax()+h);
            plot->xAxis->setTicks(true);
            plot->xAxis->setSubGrid(true);
	    //auto label scale in x axis
	    plot->xAxis->setAutoTickLabels(true);
            plot->xAxis->setAutoTicks(true);
            plot->xAxis->setAutoTickStep(true);
            //plot->xAxis.setTickStep(0.1); if decomment: put false to line above
            plot->xAxis->setAutoSubTicks(false);
            plot->xAxis->setSubTickCount(9);
            // setup Y-Axis:
            int max  = (int)*max_element(yData.begin(),yData.end());
            plot->yAxis->setRange(0, max+1);
            plot->yAxis->setAutoTickStep(false);
            plot->yAxis->setTickStep(1);
            plot->yAxis->setSubGrid(true);
            plot->yAxis->setAutoSubTicks(false);
            plot->yAxis->setSubTickCount(3);
            plot->yAxis->setPadding(5); // a bit more space to the left border
            plot->yAxis->setLabel("Histogram of "+ QString::fromStdString((*listDR).getMonitorName()));

            // setup legend:
            plot->legend->setVisible(true);
            plot->legend->setPositionStyle(QCPLegend::psTopRight);
            plot->legend->setBrush(QColor(255, 255, 255, 0));
            QPen legendPen;
            legendPen.setColor(QColor(130, 130, 130, 200));
            plot->legend->setBorderPen(legendPen);

            //allows to move in the plot
            plot->setRangeDrag(Qt::Horizontal|Qt::Vertical);
            //allows to zoom in the plot
            plot->setRangeZoom(Qt::Horizontal|Qt::Vertical);
            return true;
        }
        listDR++;
    }
    return false;
}


//-------------drawBox-------------------------------------
bool csv2Plot::drawBoxChart(QCustomPlot *plot, string monitor)
{
    list<dataMonitoredList>::iterator listDR =results.begin();
    while (listDR != results.end()) {
        if ((*listDR).getMonitorName() == monitor) {
            // create empty statistical box plottables:
            QCPStatisticalBox *sample1 = new QCPStatisticalBox(plot->xAxis, plot->yAxis);
            plot->addPlottable(sample1);
            QBrush boxBrush(QColor(60, 60, 255, 100));
            boxBrush.setStyle(Qt::Dense6Pattern); // make it look oldschool
            sample1->setBrush(boxBrush);

            // set data:
            sample1->setKey(1);
            sample1->setLowerQuartile((*listDR).firstQuartile());
            sample1->setMedian((*listDR).median());
            sample1->setUpperQuartile((*listDR).thirdQuartile());
            vector<double> v  = (*listDR).outliers();
            int up=0,down = 0;
            if (v.size()>0) {
                sample1->setOutliers(QVector<double>::fromStdVector(v));
                for (vector<double>::iterator it = v.begin(); it != v.end();it++) {
                    if (*it < (*listDR).firstQuartile()) {
                        down++;
                    } else if (*it > (*listDR).thirdQuartile()) {
                        up++;
                    }
                }
            }

            vector<double> temp = (*listDR).getResultData();
            std::sort(temp.begin(),temp.end());
            if (up!=0) {
                int val = temp.size()-up-1;
                sample1->setMaximum(temp[val]);
            } else {
                sample1->setMaximum((*listDR).getMax());
            }
            if (down!=0) {
                sample1->setMinimum(temp[down]);
            } else {
                sample1->setMinimum((*listDR).getMin());
            }
            plot->rescaleAxes();
            // prepare manual x axis labels:
            plot->xAxis->setSubTickCount(0);
            plot->xAxis->setTickLength(0, 4);
            plot->xAxis->setTickLabelRotation(20);
            plot->xAxis->setAutoTicks(false);
            plot->xAxis->setAutoTickLabels(false);
            plot->xAxis->setTickVector(QVector<double>() << 1);
            plot->xAxis->setTickVectorLabels(QVector<QString>()<<(*listDR).getMonitorName().c_str());
            plot->xAxis->scaleRange(1,plot->xAxis->range().center());
            plot->xAxis->setRange(0.5,1.5);

            // prepare axes:
            plot->yAxis->setRange(0, (*listDR).getMax()+(*listDR).firstQuartile());
            plot->yAxis->setAutoTickStep(true);
            plot->yAxis->setSubGrid(true);
            plot->yAxis->setAutoSubTicks(false);
            plot->yAxis->setSubTickCount(5);
            plot->yAxis->setAutoTickCount(15);
            plot->yAxis->setLabel("Box plot");


            plot->legend->setVisible(false);
            //allows to move in the plot
            plot->setRangeDrag(Qt::Horizontal|Qt::Vertical);
            //allows to zoom in the plot
            plot->setRangeZoom(Qt::Horizontal|Qt::Vertical);
            return true;
        }
        listDR++;
    }
    return false;
}

