/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/Cell.h>
#include <pml/CellProperties.h>

#include "CellDCProperties.h"
#include "CellDC.h"


#include <Application.h>

#include <QMessageBox>

// ---------------------- Constructor ----------------------
CellDCProperties::CellDCProperties(CellDC *dc) : QObject(NULL) {
  setObjectName("Cell Properties");
  updateProperties(dc);
  // install a filter to get the modification of the dynamic properties
  installEventFilter(this);
}


// ---------------------- updateProperties ----------------------
void CellDCProperties::updateProperties(CellDC *dc) {

  // get the DC (who will get the messages)
  myDC = dc;

  if (myDC != NULL) {
    // message on the statusbar
    Application::showStatusBarMessage("Cell #" + QString::number(dc->getCell()->getIndex()) + " selected.", 2000); // for 2sec
  }

  // check for extra properties...
  CellProperties *cProp = dc->getCell()->getProperties();

  std::string propName;

  for (unsigned int i = 0; i < cProp->numberOfFields(); i++) {
    propName = cProp->getField(i);

    if (propName != "") {
      setProperty(propName.c_str(), QVariant(cProp->getString(propName).c_str()));
    }
  }

}


// ---------------------- setName ----------------------
void CellDCProperties::setName(QString id) {
  if (myDC) {
    if (dynamic_cast<Structure *>(myDC->getCell())->getName()!= "") {
      dynamic_cast<Structure *>(myDC->getCell())->setName(id.toStdString());
      myDC->setName(id);
      myDC->setModified();

      myDC->refresh();
    }
    else {
      QMessageBox::warning(NULL, "Apply cell properties",
                           "This cell has no name, it is designated by its index.\nYou can not create a name.");
    }
  }
}

// ---------------------- getName ----------------------
QString CellDCProperties::getName() const {
  if (myDC)
    return myDC->getName();
  else
    return "";
}

// ---------------------- getGeometry ----------------------
CellDCProperties::GeometricType CellDCProperties::getGeometry() const {

  if (myDC) {
    CellDCProperties::GeometricType t;
    switch (myDC->getCell()->getType()) {
      case StructureProperties::ATOM:
        t = ATOM;
        break;
      case StructureProperties::TETRAHEDRON:
        t = TETRAHEDRON;
        break;
      case StructureProperties::HEXAHEDRON:
        t = HEXAHEDRON;
        break;
      case StructureProperties::WEDGE:
        t = WEDGE;
        break;
      case StructureProperties::PYRAMID:
        t = PYRAMID;
	break;
      case StructureProperties::LINE:
        t = LINE;
        break;
      case StructureProperties::POLY_LINE:
        t = POLY_LINE;
        break;
      case StructureProperties::POLY_VERTEX:
        t = POLY_VERTEX;
        break;
      case StructureProperties::TRIANGLE:
        t = TRIANGLE;
        break;
      case StructureProperties::QUAD:
        t = QUAD;
        break;
      default:
        t = INVALID;
        break;
    }
    return t;
  }
  else {
    return INVALID;
  }
}

// ---------------------- event ----------------------------
bool CellDCProperties::event(QEvent * e) {
  if (e->type() == QEvent::DynamicPropertyChange) {
    e->accept();
    QDynamicPropertyChangeEvent *chev = dynamic_cast<QDynamicPropertyChangeEvent *>(e);

    if (! chev)
      return false;

    // set the corresponding property (field)
    myDC->getCell()->getProperties()->set(QString(chev->propertyName()).toStdString(), std::string(property(chev->propertyName()).toString().toStdString()));

    return true;
  }

  return QObject::event(e);
}
