/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file MshExtension.cpp
 * @brief Component class implementation for .msh files.
 * @author Mathieu BAILET
 * @date July 5 2012
 */

#include "MshComponent.h"

#include <sstream>

//#include <Geometry.h>
#include <InteractiveViewer.h>
#include <Log.h>

#include <QFileInfo>
#include <QVector3D>

#include <vtkOBJReader.h>
#include <vtkProperty.h>
#include <vtkUnstructuredGrid.h>
#include <vtkTetra.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkPyramid.h>
#include <vtkCellArray.h>

using namespace camitk;

// -------------------- default constructor  --------------------
MshComponent::MshComponent(const QString & file) throw(AbortException) : MeshComponent(file) {

    // use the file name without extension as component name
    setName(QFileInfo(file).baseName());

    std::ifstream inputFile(file.toStdString().c_str(), std::ios::in);
    std::string line;
    int n, eltType, nbNodes;
    vtkIdType vtkEltTypeId;
    double pt[3];
    vtkSmartPointer<vtkUnstructuredGrid> mesh = vtkSmartPointer<vtkUnstructuredGrid>::New();

    nbElements = 0;

    //build the vtkPoinSet from the file
    vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();

    // get points from file
    while (std::getline(inputFile, line)) {
        if ((!line.compare("$Nodes")) || (!line.compare("$NOD")))
            break;
    }
    std::getline(inputFile, line); // nb vertices
    while (std::getline(inputFile, line)) {
        std::istringstream stream(line, std::istringstream::in);
        if ((!line.compare("$EndNodes")) || (!line.compare("$ENDNOD")))
            break;
        stream >> n;
        for (int i = 0; i < 3; i++)
            stream >> pt[i];
        points->InsertNextPoint (pt);
    }
    mesh->SetPoints(points);

    // get elements
    while (std::getline(inputFile, line)) {
        if ((!line.compare("$Elements")) || (!line.compare("$ELM")))
            break;
    }

    std::getline(inputFile, line); // nb elements
    std::istringstream stream(line, std::istringstream::in);
    stream >> n;

    while (std::getline(inputFile, line)) {
        std::istringstream stream(line, std::istringstream::in);
        if ((!line.compare("$EndElements")) || (!line.compare("$ENDELM")))
            break;
        // element number
        stream >> n;
        // element type
        stream >> eltType;

        for (int i = 0; i < 3; i++)
            stream >> n;

        switch (eltType) {
        case 2 :
            vtkEltTypeId = VTK_TRIANGLE;
            nbNodes = 3;
            break;
        case 4 :
            vtkEltTypeId = VTK_TETRA;
            nbNodes = 4;
            break;
        case 5 :
            vtkEltTypeId = VTK_HEXAHEDRON;
            nbNodes = 8;
            break;
        case 6 :
            vtkEltTypeId = VTK_WEDGE;
            nbNodes = 6;
            break;
        case 7 :
            vtkEltTypeId = VTK_PYRAMID;
            nbNodes = 5;
            break;
        default :
            std::cerr << "unknown element type : " << eltType << std::endl;
            continue;
        }

        vtkSmartPointer<vtkIdList> idList = vtkSmartPointer<vtkIdList>::New();
        for (int j=0; j<nbNodes; j++) {
            stream >> n;
            idList->InsertNextId(n-1);
        }

        mesh->InsertNextCell(vtkEltTypeId, idList);
        nbElements++;
    }

    mesh->Update();

    // instanciate the Geometry
    initRepresentation(mesh);

    // add the represention in the 3D viewer
    setVisibility(InteractiveViewer::get3DViewer(), true);

    // initialize the dynamic properties
    initDynamicProperties();
}

// -------------------- initDynamicProperties --------------------
void MshComponent::initDynamicProperties() {
    // add a dynamic property to manage the surface color
    //setProperty("position point #1", QVector3D(1.0,0.0,0.0));
    vtkUnstructuredGrid* mesh = dynamic_cast< vtkUnstructuredGrid* >(getPointSet().GetPointer());
    vtkSmartPointer<vtkUnsignedCharArray> types = mesh->GetCellTypesArray();
    std::map<unsigned char, int> elementsMap;
    std::map<unsigned char, int>::iterator elementsMapIt;

    for (int i=0; i<types->GetNumberOfTuples(); i++) {
        if (!elementsMap.count(types->GetValue(i)))
            elementsMap[types->GetValue(i)] = 0;
        elementsMap[types->GetValue(i)]++;
    }

    for (elementsMapIt = elementsMap.begin(); elementsMapIt != elementsMap.end(); elementsMapIt++) {
        switch(elementsMapIt->first) {
        case VTK_TRIANGLE :
            setProperty("Triangle", elementsMapIt->second);
            break;
        case VTK_TETRA :
            setProperty("Tetra", elementsMapIt->second);
            break;
        case VTK_HEXAHEDRON :
            setProperty("Hexa", elementsMapIt->second);
            break;
        case VTK_WEDGE :
            setProperty("Prism", elementsMapIt->second);
            break;
        case VTK_PYRAMID :
            setProperty("Pyramid", elementsMapIt->second);
            break;
        default :
            setProperty("Prob", elementsMapIt->second);
            break;
        }
    }
}

// -------------------- getNbPoints --------------------
int MshComponent::getNbPoints() {
    return getPointSet()->GetNumberOfPoints();
}

// -------------------- getNbElements --------------------
int MshComponent::getNbElements() {
    return nbElements;
}

