#
# CamiTK Macros
#


# MACRO get_subdirectories 
#
# usage:
# get_subdirectories(Name)
#
# get the names of all the subdirectories that contains a CMakeLists.txt
macro(get_subdirectories Name)
  set(Name)
  file(GLOB LS *)
  foreach(FILENAME ${LS})
    if(IS_DIRECTORY ${FILENAME} AND EXISTS ${FILENAME}/CMakeLists.txt)
      get_directory_name(${FILENAME} DIRNAME)
      # ignore .svn
      if (NOT ${DIRNAME} STREQUAL ".svn")
          set(${Name} ${${Name}} ${DIRNAME})
      endif()
    endif()
  endforeach()
endmacro()

# MACRO gather_headers_and_sources 
#
# usage:
# gather_headers_and_sources(Name [BaseDirectory])
#
# find all headers, sources, including the qt ui, moc and resources
# and create two variables from it: ${Name}_HEADERS ${Name}_SOURCES, and define all needed commands for qt
# if BaseDirectory is specified, the glob recurse starts in directory DIR

macro(gather_headers_and_sources Name)
  # gather all possible C++ and Qt sources
  if (${ARGC} EQUAL 1)
    file(GLOB_RECURSE HEADERS *.h)
    file(GLOB_RECURSE SRCS *.cpp *.c)
    file(GLOB_RECURSE File_UI *.ui)
    file(GLOB_RECURSE File_QRC *.qrc)
  else()
    # if an optional parameter is used, gather everything from BaseDirectory
    file(GLOB_RECURSE HEADERS ${ARGV1}/*.h )
    file(GLOB_RECURSE SRCS ${ARGV1}/*.cpp *.c)
    file(GLOB_RECURSE File_UI ${ARGV1}/*.ui )
    file(GLOB_RECURSE File_QRC ${ARGV1}/*.qrc )
  endif()

  # manage Qt ui
  qt4_wrap_ui (UI ${File_UI})
  
  # manage Qt resources
  qt4_add_resources(QRC ${File_QRC})

  # find Q_OBJECT derived class
  foreach(HEADER ${HEADERS})
    file(READ ${HEADER} stream)
    if(stream MATCHES "Q_OBJECT")
      set(MOC_SOURCES ${MOC_SOURCES} ${HEADER})
    endif(stream MATCHES "Q_OBJECT")
  endforeach(HEADER)

  # manage Qt inherited sources
  qt4_wrap_cpp(QT_SRCS ${MOC_SOURCES})

  # name all headers
  set (${Name}_HEADERS
      ${HEADERS}
      ${UI}
      ${QRC}
  )

  # name all sources
  set (${Name}_SOURCES
      ${HEADERS}
      ${UI}
      ${QRC}
      ${SRCS}
      ${QT_SRCS}
  )
endmacro()

# MACRO parse_arguments
#
# usage, see: http://www.cmake.org/Wiki/CMakeMacroParseArguments
macro(parse_arguments prefix arg_names option_names)
  set(DEFAULT_ARGS)
  foreach(arg_name ${arg_names})    
    set(${prefix}_${arg_name})
  endforeach()
  foreach(option ${option_names})
    set(${prefix}_${option} FALSE)
  endforeach()

  set(current_arg_name DEFAULT_ARGS)
  set(current_arg_list)
  foreach(arg ${ARGN})            
    set(larg_names ${arg_names})    
    list(FIND larg_names "${arg}" is_arg_name)                   
    if (is_arg_name GREATER -1)
      set(${prefix}_${current_arg_name} ${current_arg_list})
      set(current_arg_name ${arg})
      set(current_arg_list)
    else()
      set(loption_names ${option_names})    
      list(FIND loption_names "${arg}" is_option)            
      if (is_option GREATER -1)
       set(${prefix}_${arg} TRUE)
      else()
       set(current_arg_list ${current_arg_list} ${arg})
      endif()
    endif()
  endforeach()
  set(${prefix}_${current_arg_name} ${current_arg_list})
endmacro()

# MACRO get_directory_name
#
# usage:
# directory_name(FullPathDirectory VariableName)
#
# set the variable VariableName to the name of the last directory of FullPathDirectory
# set the variable VariableName_CMAKE to be the same but in uppercase

macro(get_directory_name FullPathDirectory VariableName)
  string (REGEX REPLACE ".*/([^/]*)$" "\\1" ${VariableName} ${FullPathDirectory})

  # set the variable name to uppercase
  string(TOUPPER ${${VariableName}} ${VariableName}_CMAKE)  
endmacro()

# MACRO export_headers
#
# Duplicate headers installation:
# - one is used at compiled time and puts everything in 
#   ${CAMITK_BUILD_DIR}/include/${GroupName}/${ComponentName}/${SubdirName} 
# - the other one is used at installation time and puts everything in 
#   ${CAMITK_INSTALL_ROOT}/include/${GroupName}/${ComponentName}/${SubdirName}
#
# usage:
# export_headers(HeaderFile1.h HeaderFile2.h ...
#                COMPONENT ComponentName
#                [GROUP GroupName]
#                [SUBDIRECTORY SubdirName]
# )
# 
# FILES        = list of files to install
# COMPONENT    = name of the component to use. This is also the include subdirectory name
#                used for copying the file
# GROUP        = the name of the group this install should be using group will be
#                prepend to the component name.
# SUBDIRECTORY = optional subdirectory to use in ${CAMITK_INCLUDE_DIR}/include/${ComponentName}
macro(export_headers)

  parse_arguments(EXPORT_HEADER 
    "COMPONENT;SUBDIRECTORY;GROUP"  # possible lists
    "" # possible options
    ${ARGN}
  )

  # special case for components: remove prefix for the destination directory
  string(REGEX REPLACE "^component-|^action-" "" EXPORT_HEADER_DESTINATION ${EXPORT_HEADER_COMPONENT})
  
  # Check group
  if(EXPORT_HEADER_GROUP)
    # check directory
    if (NOT EXISTS ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_GROUP})
      add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_GROUP}
                        COMMENT "Creating build-time group include dir ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_GROUP}"
                        VERBATIM
      )
    endif()
    set(EXPORT_HEADER_DESTINATION ${EXPORT_HEADER_GROUP}/${EXPORT_HEADER_DESTINATION})
  endif()

  # check that the directory exists otherwise create it
  if (NOT EXISTS ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION})
    add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                      POST_BUILD
                      COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}
                      COMMENT "Creating build-time include dir ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}"
                      VERBATIM
    )
  endif()
  
  # check the SUBDIRECTORY parameter
  if(EXPORT_HEADER_SUBDIRECTORY)
    set(EXPORT_HEADER_DESTINATION ${EXPORT_HEADER_DESTINATION}/${EXPORT_HEADER_SUBDIRECTORY})
    # create subdirectory if it does not exists
    if (NOT EXISTS ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION})
      add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}
                        COMMENT "Creating build-time include subdir ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}"
                        VERBATIM
      )
    endif()
  endif()
  

  # at build time, copy the files to build directory include files when the target is built
  foreach(HEADER ${EXPORT_HEADER_DEFAULT_ARGS})
    # check if file name is relative or not
    set(FILE_TO_COPY ${HEADER})
   
	# copy after build, only if there was a change
    add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                       POST_BUILD
                       COMMAND ${CMAKE_COMMAND} -E copy_if_different ${FILE_TO_COPY} ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}/
                       COMMENT "Installing build-time header ${HEADER}"
                       WORKING_DIRECTORY ${CMAKE_CURRENT_SOURCE_DIR}
                       VERBATIM
    )
  endforeach()

  # during installation, copy the files to install directory
  install(FILES ${EXPORT_HEADER_DEFAULT_ARGS} 
          DESTINATION ${CAMITK_INSTALL_ROOT}/include/${CAMITK_SHORT_VERSION_STRING}/${EXPORT_HEADER_DESTINATION}/	
	      COMPONENT ${EXPORT_HEADER_COMPONENT}
  )

endmacro()


# MACRO camitk_extension
#
# Simplify writing a camitk extension (component, action) 
# 
# usage:
# 			extension(<type> 
#					  [DEFAULT]
#                     [NEEDS_ITK]
#                     [NEEDS_LIBXML2]
#                     [NEEDS_OPENCV]
#                     [NEEDS_IGSTK]
#                     [NEEDS_XSD]
#                     [NEEDS_TOOL tool1 tool2 ...]
#                     [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#                     [INCLUDE_DIRECTORIES dir1 dir2 ...]
#                     [DEFINES flag1 flag2 ...]
#                     [EXTERNAL_SOURCES file1 file2 ...]
#                     [LIBRARIES lib1 lib2 ...]
#                     [HEADERS_TO_INSTALL header1.h header2.h ...]
#                     [INSTALL_ALL_HEADERS]
#                    )
#
# The name of the extension is automatically the name of the directory from where this macro
# is called.
#
# <type>					= The selected type of extension you wish to build.
#							  possible values : ACTION_EXTENSION or COMPONENT_EXTENSION.
# DEFAULT                   = means this is a default component to be compiled automatically
# NEEDS_ITK                 = add this if your component needs itk. 
#                             Do not forget to add the needed list of itk libraries in the LIBRARIES parameter
# NEEDS_LIBXML2             = add this if your component needs libxml2 
# NEEDS_XSD                 = add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
# NEEDS_OPENCV              = add this if your component needs OpenCV
# NEEDS_IGSTK               = add this if your component needs IgsTK
# NEEDS_XERCESC	 			= add this if your action / component needs XercesC library
# NEEDS_TOOL                = list of needed camitk tools
# NEEDS_COMPONENT_EXTENSION = list of needed component extensions
# NEEDS_ACTION_EXTENSION 	= list of needed component extensions
# INCLUDE_DIRECTORIES       = additional include directories
# DEFINES                   = list of define flags to add at compilation time
# EXTERNAL_SOURCES          = list of extra source/headers files (external to the current directory)
#                             that needed to be added to the SOURCES variable. 
#                             Note: EXTERNAL_SOURCES are not installed
# LIBRARIES                 = libraries to add to the link command
# HEADERS_TO_INSTALL        = list of headers to install, if present this will automatically
#                             create an "install-COMPONENT_NAMEcomponent" target, that can be used
#                             anywhere else to manage dependencies to this component.
#                             The headers are installed ${CAMITK_INCLUDE_DIR}
#                             when the target "install-COMPONENT_NAMEcomponent" is called.
# INSTALL_ALL_HEADERS       = install all of the headers (this is the lazy solution, please consider
#                             making a list and using HEADERS_TO_INSTALL parameter instead!   
macro(camitk_extension)

  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} EXTENSION_NAME)
  
  parse_arguments(${EXTENSION_NAME_CMAKE} #prefix
    "NEEDS_TOOL;NEEDS_COMPONENT_EXTENSION;NEEDS_ACTION_EXTENSION;INCLUDE_DIRECTORIES;LIBRARIES;HEADERS_TO_INSTALL;DEFINES;EXTERNAL_SOURCES"  # possible lists
    "ACTION_EXTENSION;COMPONENT_EXTENSION;DEFAULT;NEEDS_XERCESC;NEEDS_ITK;NEEDS_LIBXML2;NEEDS_XSD;NEEDS_OPENCV;NEEDS_IGSTK;INSTALL_ALL_HEADERS" # possible options
    ${ARGN}
  )

  # Get the type of extension
  if (${EXTENSION_NAME_CMAKE}_ACTION_EXTENSION)
	set(TYPE_EXTENSION "action")	
	string(TOUPPER ${TYPE_EXTENSION} TYPE_EXTENSION_CMAKE)
  elseif(${EXTENSION_NAME_CMAKE}_COMPONENT_EXTENSION)
	set(TYPE_EXTENSION "component")	
	string(TOUPPER ${TYPE_EXTENSION} TYPE_EXTENSION_CMAKE)
  endif()

  # if it is the first cmake run, create the extension variable with a correct initial value
  if(NOT CAMITK_${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL)
    # add option to enable/disable this extension and set it to true if either
    # it was marked as default or a -D???_${EXTENSION_NAME_CMAKE}:BOOL=TRUE was given on the command line
    set(${EXTENSION_NAME_CMAKE}_DEFAULT_IS_SET "FALSE")
    if(DEFINED ${EXTENSION_NAME_CMAKE}_DEFAULT)
        set(${EXTENSION_NAME_CMAKE}_DEFAULT_IS_SET ${${EXTENSION_NAME_CMAKE}_DEFAULT})
    endif()
    if(DEFINED ${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE})
        set(${EXTENSION_NAME_CMAKE}_DEFAULT_IS_SET ${${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}})
    endif()
    set(${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE} ${${EXTENSION_NAME_CMAKE}_DEFAULT} CACHE BOOL "Build extension ${EXTENSION_NAME}")
    set(CAMITK_${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL TRUE CACHE BOOL "Is variable ${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME} already created?")
    mark_as_advanced(CAMITK_${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE}_INTERNAL)
  endif()
  
  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (${TYPE_EXTENSION_CMAKE}_${EXTENSION_NAME_CMAKE})   

    # check for target name
    set(${TYPE_EXTENSION_CMAKE}_TARGET_NAME ${TYPE_EXTENSION}-${EXTENSION_NAME})
    message(STATUS "Building extension ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}")
    
    # actions are compiled as shared object 
    set(CMAKE_LIBRARY_OUTPUT_DIRECTORY ${CMAKE_LIBRARY_OUTPUT_DIRECTORY}/${TYPE_EXTENSION}s) 

    # extension are compiled as shared object (for dll plateform put it in BIN)
    set(CMAKE_RUNTIME_OUTPUT_DIRECTORY ${CMAKE_RUNTIME_OUTPUT_DIRECTORY}/${TYPE_EXTENSION}s) 

    # include directories
    include_directories(${CAMITK_INCLUDE_DIRECTORIES})
    include_directories(${CMAKE_CURRENT_BINARY_DIR})
    include_directories(${CMAKE_CURRENT_SOURCE_DIR})
    include_directories(${${EXTENSION_NAME_CMAKE}_INCLUDE_DIRECTORIES})

    # check definitions
    if(${EXTENSION_NAME_CMAKE}_DEFINES)
      foreach (FLAG ${${EXTENSION_NAME_CMAKE}_DEFINES})
        add_definitions(-D${FLAG})
      endforeach()
    endif()

    # check for needed tool
    set(TOOL_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_TOOL)
      foreach(TOOL_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_TOOL})
        string(TOUPPER ${TOOL_NEEDED} ${TOOL_NEEDED}_INTERNAL)
        set(TOOL_${${TOOL_NEEDED}_INTERNAL} ON CACHE BOOL "Required by action extension ${EXTENSION_NAME}" FORCE )
        set(TOOL_LIBRARIES ${TOOL_LIBRARIES} ${TOOL_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/${TOOL_NEEDED})
      endforeach()
    endif()

 	# Looking for ITK
    set(ITK_LIBRARIES "")
    if(${EXTENSION_NAME_CMAKE}_NEEDS_ITK)
      # ITK is required
      find_package(ITK REQUIRED)
      include(${ITK_USE_FILE})
      set(ITK_LIBRARIES ITKBasicFilters ITKCommon ITKIO)
    else()
      set(ITK_LIBRARIES "")
    endif()

   # Looking for LibXML2.0
    set(LIBXML2_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_LIBXML2) 
      # LibXml2 is required
      find_package(Xml2)
      if (LIBXML2_FOUND)
        add_definitions(${LIBXML2_DEFINITIONS})
        include_directories(${LIBXML2_INCLUDE_DIR})      
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: libxml2 required")
      endif()
    endif()
	
    # looking for OpenCV
    set(OpenCV_LIBRARIES "")
    if(${EXTENSION_NAME_CMAKE}_NEEDS_OPENCV) 
		# OpenCV is required
		find_package( OpenCV REQUIRED )
	else ( )
		set(OpenCV_LIBRARIES "")
	endif()

    #looking for IGSTK libraries ?
    set(IGSTK_LIBRARIES "")
    if(${EXTENSION_NAME_CMAKE}_NEEDS_IGSTK)
		find_package(IGSTK REQUIRED)
		include(${IGSTK_USE_FILE})	
	else()
		set(IGSTK_LIBRARIES "")
	endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: xerces-c required")
      endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${EXTENSION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()

    set(COMPONENT_EXTENSION_LIBRARIES)
    # check for intra-component extension
    if(${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} ${COMPONENT_NEEDED}_INTERNAL)
        set(COMPONENT_${${COMPONENT_NEEDED}_INTERNAL} ON CACHE BOOL "Required by extension ${EXTENSION_NAME}" FORCE )
        set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} component-${COMPONENT_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
      endforeach()
    endif()

    set(ACTION_EXTENSION_LIBRARIES)
    # check for intra-action extension
    if(${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION) 
      foreach(ACTION_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ${ACTION_NEEDED}_INTERNAL)
        set(ACTION_${${ACTION_NEEDED}_INTERNAL} ON CACHE BOOL "Required by extension ${EXTENSION_NAME}" FORCE )
        set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES} action-${ACTION_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/actions/${ACTION_NEEDED})
      endforeach()
    endif()

    # check for libs in all known camitk dir + actions subdir (beware of the plateform bin or lib?)
    link_directories( ${CAMITK_LIB_DIRECTORIES}
                      ${CAMITK_BIN_DIR}
                      ${CAMITK_BIN_DIR}/components
                      ${CAMITK_BIN_DIR}/actions
                      ${CAMITK_INSTALL_ROOT}/lib/${CAMITK_SHORT_VERSION_STRING}/actions
                      ${CAMITK_INSTALL_ROOT}/lib/${CAMITK_SHORT_VERSION_STRING}/components
                      ${CAMITK_INSTALL_ROOT}/bin/components
                      ${CAMITK_INSTALL_ROOT}/bin/actions
                      ${CMAKE_CURRENT_BINARY_DIR}/../../bin/components
                      ${CMAKE_CURRENT_BINARY_DIR}/../../bin/actions
    )

    # get all headers, sources and do what is needed for Qt 
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${EXTENSION_NAME_CMAKE})
	
    # check for external sources
    if(${EXTENSION_NAME_CMAKE}_EXTERNAL_SOURCES)
            set(${EXTENSION_NAME_CMAKE}_SOURCES ${${EXTENSION_NAME_CMAKE}_SOURCES} ${${EXTENSION_NAME_CMAKE}_EXTERNAL_SOURCES})
    endif()

    # build the action extension module (library)
    add_library(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} SHARED ${${EXTENSION_NAME_CMAKE}_SOURCES})
    
    # Update XML Project description adding this target as a subproject of the main CamiTK project with
    # its dependencies (in the CMake target point of view)
    addSubProject(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} ${CAMITK_CORE_LIB} ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${TOOL_LIBRARIES})

    # Set the libraries required to link the target
    # An action has to be linked with ${CAMITK_CORE_LIB}. As CAMITK_CORE_LIB is always a DLL/shared lib, there should
    # be no symbols loaded multiple times
    target_link_libraries(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} ${CAMITK_LIBRARIES} ${CAMITK_CORE_LIB} ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${ITK_LIBRARIES} ${LIBXML2_LIBRARIES} ${OpenCV_LIBRARIES} ${IGSTK_LIBRARIES} ${XERCESC_LIBRARY} ${TOOL_LIBRARIES}  ${${EXTENSION_NAME_CMAKE}_LIBRARIES})

    # dependencies to core (only set for internal build)
    if (CAMITK_INTERNAL_BUILD)
      add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} ${CAMITK_CORE_LIB})
    endif()

    # see http://www.cmake.org/pipermail/cmake/2012-April/049889.html
    # target properties (outputname and remove soname)
    #  set_property(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTY NO_SONAME 1)
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                           PROPERTIES OUTPUT_NAME ${EXTENSION_NAME}
    )
    set(${TYPE_EXTENSION_CMAKE}_LIBRARY_PROPERTIES ${${TYPE_EXTENSION_CMAKE}_LIBRARY_PROPERTIES}
        VERSION   "${CAMITK_VER_MAJ}.${CAMITK_VER_MIN}.${CAMITK_VER_PAT}"
        SOVERSION "${CAMITK_VER_MAJ}"
    )
    # set the library specific info (SONAME...)
    set_target_properties(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} PROPERTIES ${${TYPE_EXTENSION_CMAKE}_LIBRARY_PROPERTIES} LINK_INTERFACE_LIBRARIES "")

    # tools dependencies
    if(${EXTENSION_NAME_CMAKE}_NEEDS_TOOL) 
      foreach(TOOL_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_TOOL})
        add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} ${TOOL_NEEDED})
      endforeach()
    endif()

    # Set the Component(s) dependency
    if(${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} COMPONENT_NEEDED_CMAKE)
        # if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (CAMITK_COMPONENT_${COMPONENT_NEEDED_CMAKE}_INTERNAL)
            add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} component-${COMPONENT_NEEDED})
#                 message(STATUS "............................................................ ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} depends on component-${COMPONENT_NEEDED}")
#                 if(NOT TARGET "component-${COMPONENT_NEEDED}")
#                   message(STATUS "............................................................ component-${COMPONENT_NEEDED} not existing (yet)")
#                 endif()
        endif()
      endforeach()
    endif()

	# Set the Action(s) dependency
    if(${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION) 
      foreach(ACTION_NEEDED ${${EXTENSION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ACTION_NEEDED_CMAKE)
	# if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (CAMITK_ACTION_${ACTION_NEEDED_CMAKE}_INTERNAL)
		add_dependencies(${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} action-${ACTION_NEEDED})
#                 message(STATUS "............................................................ ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} depends on action-${ACTION_NEEDED}")
#                 if(NOT TARGET "action-${ACTION_NEEDED}")
#                   message(STATUS "............................................................ action-${ACTION_NEEDED} not existing")
#                 endif() 
        endif()
      endforeach()
    endif()

    # generate an install target for headers
    if(${EXTENSION_NAME_CMAKE}_HEADERS_TO_INSTALL)
		export_headers(${${EXTENSION_NAME_CMAKE}_HEADERS_TO_INSTALL} COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} GROUP ${TYPE_EXTENSION}s)
    endif()

    # lazy you!
    if(${EXTENSION_NAME_CMAKE}_INSTALL_ALL_HEADERS)
      export_headers(${${EXTENSION_NAME_CMAKE}_HEADERS} COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME} GROUP ${TYPE_EXTENSION}s)
    endif()

    # lib installation
    install(TARGETS ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
            RUNTIME DESTINATION bin/${TYPE_EXTENSION}s
            LIBRARY DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/${TYPE_EXTENSION}s
            ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/${TYPE_EXTENSION}s
            COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
    )

	 # testdata files installation
    if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/testdata)
      # Check test data dir directory
      if (NOT EXISTS ${CAMITK_TESTDATA_DIR})
        add_custom_command(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                          POST_BUILD
                          COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_TESTDATA_DIR}
                          COMMENT "Creating build-time share directory for test data ${CAMITK_TESTDATA_DIR}"
                          VERBATIM
        )
      endif()

      # At build time, copy the files to test data directory
      add_custom_command(TARGET ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E copy_directory testdata ${CAMITK_TESTDATA_DIR}
                        COMMENT "Installing build-time test data for ${EXTENSION_NAME}"
                        WORKING_DIRECTORY ${CMAKE_CURRENT_SOURCE_DIR}
                        VERBATIM
      )
      

      # during installation, copy the files to install directory
	  set (TESTDATA_DEST_DIR)
	  if ( MSVC )
		# Do NOT add ${CAMITK_SHORT_VERSION_STRING} to the testdata path for Win32 MSVC !
		set (TESTDATA_DEST_DIR share/testdata)
	  else ()
		set (TESTDATA_DEST_DIR share/${CAMITK_SHORT_VERSION_STRING}/testdata)
	  endif ()
      install(DIRECTORY testdata/
			  #DESTINATION share/testdata
              #DESTINATION share/${CAMITK_SHORT_VERSION_STRING}/testdata
			  DESTINATION ${TESTDATA_DEST_DIR}
              COMPONENT ${${TYPE_EXTENSION_CMAKE}_TARGET_NAME}
              PATTERN ".svn" EXCLUDE
              PATTERN "*~" EXCLUDE
      )
    endif()

  endif()

endmacro()

# Old macro to configure a new action to the framework
# Use camitk_extension macro instead
macro(action_extension)
        message(WARNING "Deprecated macro action_extension detected : please use camitk_extension(ACTION_EXTENSION ...) instead.")
        camitk_extension(ACTION_EXTENSION ${ARGN})
endmacro()

# Old macro to configure a new component to the framework
# Use camitk_extension macro instead
macro(component_extension)
        message(WARNING "Deprecated macro component_extension detected : please use camitk_extension(COMPONENT_EXTENSION ...) instead.")
        camitk_extension(COMPONENT_EXTENSION ${ARGN})
endmacro()

# TODO write a viewer_extension macro in CamiTK


# MACRO camitk_tool
#
# This macro simplifies the declaration of a library inside CamiTK
# and is to be used for all inner project tools
#
# The name of the tool/library is deduced from the name of the directory, unless LIBNAME is provided
#
# If the library is compiled as shared, it adds the needed library properties
# If it needs relocation flag fPIC (on some plateforms), this macro should determine
# if this is the case and specify this flag
#
# usage:
# camitk_tool(
#     SHARED|STATIC
#     SOURCES source1.cpp source1.h...
#     TARGET_LINK_LIBRARIES lib1 lib2...
#     [INCLUDE_DIRECTORIES dir1 dir2...]
#     [NEEDS_LIBXML2]
#     [NEEDS_XERCESC]
#     [NEEDS_XSD]
#     [LINK_DIRECTORIES dir1 dir2...]
#     [LIBNAME libname]
# )
# where:
# SHARED                        optional, if the tool should be compiled as shared
# SOURCES                       all sources to be included in the library
# INCLUDE_DIRECTORIES           all needed include directories (${CMAKE_CURRENT_SOURCE_DIR} and ${CMAKE_CURRENT_BINARY_DIR} are automatically added)
# TARGET_LINK_LIBRARIES         all the libraries to be used during the link
# NEEDS_LIBXML2                 add this if the tool needs libxml2
# NEEDS_XERCESC                 add this if your action / component needs XercesC library
# NEEDS_XSD                     add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
# LIBNAME                       force the library/tool name to be different from the directory it is in
# LINK_DIRECTORIES              additional directories to use in link_directories(...)
macro(camitk_tool)
  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} DEFAULT_LIBRARY_NAME)

  parse_arguments(${DEFAULT_LIBRARY_NAME_CMAKE}
    "SOURCES;TARGET_LINK_LIBRARIES;INCLUDE_DIRECTORIES;LIBNAME;LINK_DIRECTORIES"  # possible lists
    "SHARED;STATIC;NEEDS_LIBXML2;NEEDS_XERCESC;NEEDS_XSD" # possible options
    ${ARGN}
  )

  # if the libname is forced to be different (for string comparison problem, and
  # why there is a double quote here, see http://www.cmake.org/pipermail/cmake/2011-October/046939.html)
  if ("${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME}" STREQUAL "")
    set(LIBRARY_NAME ${DEFAULT_LIBRARY_NAME})
  else()
    set(LIBRARY_NAME ${${DEFAULT_LIBRARY_NAME_CMAKE}_LIBNAME})
  endif()
  
  # includes
  include_directories ( ${${DEFAULT_LIBRARY_NAME_CMAKE}_INCLUDE_DIRECTORIES}
    ${CMAKE_CURRENT_SOURCE_DIR}
    ${CMAKE_CURRENT_BINARY_DIR}
  )

  # Looking for LibXML2.0
  set(LIBXML2_LIBRARIES)
  if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_LIBXML2)
    # LibXml2 is required
    find_package(Xml2)
    if (LIBXML2_FOUND)
      add_definitions(${LIBXML2_DEFINITIONS})
      include_directories(${LIBXML2_INCLUDE_DIR})
    else()
      # most probably win32 or crosscompiling
      message(WARNING "${LIBRARY_NAME}: libxml2 required")
    endif()
  endif()
  
  # Looking for XercesC
  set(XERCESC_LIBRARIES)
  if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
      # most probably win32 or crosscompiling
        message(STATUS "${LIBRARY_NAME}: xerces-c required")
      endif()
  endif()

  # Looking for codesynthesis XSD CXX
  if(${DEFAULT_LIBRARY_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${LIBRARY_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
  endif()

  # some additional link directories are needed
  if (NOT "${${DEFAULT_LIBRARY_NAME_CMAKE}_LINK_DIRECTORIES}" STREQUAL "")
    link_directories(${${DEFAULT_LIBRARY_NAME_CMAKE}_LINK_DIRECTORIES})
  endif()
  
  # Add shared or static library
  if (${DEFAULT_LIBRARY_NAME_CMAKE}_SHARED)
    message(STATUS "Adding shared library: ${LIBRARY_NAME}")
    add_library(${LIBRARY_NAME} SHARED ${${DEFAULT_LIBRARY_NAME_CMAKE}_SOURCES})
    # prepare the library specific info (SONAME...)
    set(${LIBRARY_NAME}_LIBRARY_PROPERTIES ${${LIBRARY_NAME}_LIBRARY_PROPERTIES}
        VERSION   "${CAMITK_VER_MAJ}.${CAMITK_VER_MIN}.${CAMITK_VER_PAT}"
        SOVERSION "${CAMITK_VER_MAJ}"
    )
    # set the library specific info (SONAME...)
    set_target_properties(${LIBRARY_NAME} PROPERTIES ${${LIBRARY_NAME}_LIBRARY_PROPERTIES} LINK_INTERFACE_LIBRARIES "")
  else()
    if (${DEFAULT_LIBRARY_NAME_CMAKE}_STATIC)
        message(STATUS "Adding static library: ${LIBRARY_NAME}")
        add_library(${LIBRARY_NAME} STATIC ${${DEFAULT_LIBRARY_NAME_CMAKE}_SOURCES})
    else()
        message(FATAL_ERROR "In adding static library ${LIBRARY_NAME}.\n   Please specify the library/tool type: SHARED or STATIC")
    endif()
  endif()
  
  # Update XML Project description adding this target as a subproject of the main CamiTK project with
  # its dependencies (in the CMake target point of view)
  addSubProject(${LIBRARY_NAME})
  
  # 64bits and other platform with relocation needs -fPIC
  include(TestCXXAcceptsFlag)
  check_cxx_accepts_flag(-fPIC FPIC_FLAG_ACCEPTED)
  # no need to add -fPIC on mingw, otherwise it generates a warning: -fPIC ignored for target (all code is position independent) [enabled by default]
  if(FPIC_FLAG_ACCEPTED AND NOT MINGW)
    set_property(TARGET ${LIBRARY_NAME} APPEND PROPERTY COMPILE_FLAGS -fPIC)
  endif()

  #add libraries needed during the link
  target_link_libraries(${LIBRARY_NAME} ${${DEFAULT_LIBRARY_NAME_CMAKE}_TARGET_LINK_LIBRARIES} ${LIBXML2_LIBRARIES} ${XERCESC_LIBRARY})  
endmacro()

# MACRO application_extension
#
# Simplify writing application extension CMakeLists.txt
#
# usage:
# application_extension(
#        [DEFAULT]
#        [NEEDS_ITK]
#        [NEEDS_TOOL tool1 tool2 ...]
#        [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#        [NEEDS_ACTION_EXTENSION ation1 action2 ...]
#     )
#
# where :
#     DEFAULT                     = means this is a default application to be compiled automatically
#     NEEDS_ITK                   = means this application requires ITK to be compiled / run.
#     NEEDS_TOOL                  = list of needed camitk tools
#     NEEDS_COMPONENT_EXTENSION   = list of needed component extensions
#     NEEDS_ACTION_EXTENSION      = list of needed component extensions
#
# The name of the application is automatically the name of the directory from where this macro
# is called prefixed with "camitk-"
#
# 
macro(application_extension)



  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} APPLICATION_NAME)
  
  parse_arguments(${APPLICATION_NAME_CMAKE} 
    "NEEDS_TOOL;NEEDS_COMPONENT_EXTENSION;NEEDS_ACTION_EXTENSION"  # possible lists
    "DEFAULT;NEEDS_ITK;ACTION_EXTENSION;COMPONENT_EXTENSION;NEEDS_XSD;NEEDS_XERCESC" # possible options
    ${ARGN}
  )
  
  
  # if it is the first cmake run, create the application variable with a correct initial value
  if(NOT CAMITK_APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
    message(STATUS "Checking ${APPLICATION_NAME}")
    # add option to enable/disable this application extension and set it to true if either
    # it was marked as default or a -DAPPLICATION_${APPLICATION_NAME_CMAKE}:BOOL=TRUE was given on the command line
    set(APPLICATION_${APPLICATION_NAME_CMAKE}_IS_SET "FALSE")
    if(DEFINED ${APPLICATION_NAME_CMAKE}_DEFAULT)
        set(APPLICATION_${APPLICATION_NAME_CMAKE}_IS_SET ${${APPLICATION_NAME_CMAKE}_DEFAULT})
    endif()
    if(DEFINED APPLICATION_${APPLICATION_NAME_CMAKE})
        set(APPLICATION_${APPLICATION_NAME_CMAKE}_IS_SET ${APPLICATION_${APPLICATION_NAME_CMAKE}})
    endif()
    set(APPLICATION_${APPLICATION_NAME_CMAKE} ${APPLICATION_${APPLICATION_NAME_CMAKE}_IS_SET} CACHE BOOL "Build application ${APPLICATION_NAME}")
    set(CAMITK_APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL TRUE CACHE BOOL "Is variable APPLICATION_${APPLICATION_NAME} already created?")
    mark_as_advanced(CAMITK_APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
  endif()

  message(STATUS "Building application ${APPLICATION_NAME}: ${APPLICATION_${APPLICATION_NAME_CMAKE}}") 

  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (APPLICATION_${APPLICATION_NAME_CMAKE})
    # check for needed tool
    set(TOOL_LIBRARIES)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_TOOL)
      foreach(TOOL_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_TOOL})
        message(STATUS "${TOOL_NEEDED} required by application ${APPLICATION_NAME}")
        string(TOUPPER ${TOOL_NEEDED} ${TOOL_NEEDED}_INTERNAL)
        set(TOOL_${${TOOL_NEEDED}_INTERNAL} ON CACHE BOOL "Required by application ${APPLICATION_NAME}" FORCE )
        set(TOOL_LIBRARIES ${TOOL_LIBRARIES} ${TOOL_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/${TOOL_NEEDED})
      endforeach()
    endif()
	
	# Looking for ITK
    set(ITK_LIBRARIES "")
    if(${APPLICATION_NAME_CMAKE}_NEEDS_ITK)
      # ITK is required
      find_package(ITK REQUIRED)
      include(${ITK_USE_FILE})
    else()
      set(ITK_LIBRARIES "")
    endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${APPLICATION_NAME}: xerces-c required")
      endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${APPLICATION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${APPLICATION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()
	
	# Looking for required component that this application needs
	set(COMPONENT_EXTENSION_LIBRARIES)
    # check for intra-component extension
    if(${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
		string(TOUPPER ${COMPONENT_NEEDED} ${COMPONENT_NEEDED}_INTERNAL)
		set(COMPONENT_${${COMPONENT_NEEDED}_INTERNAL} ON CACHE BOOL "Required by application ${APPLICATION_NAME}" FORCE )
        set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} component-${COMPONENT_NEEDED})
		include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
      endforeach()
    endif()

    # set the directories where to seek for .h files at compilation time.
    include_directories(${CAMITK_INCLUDE_DIRECTORIES} ${CAMITK_INCLUDE_DIR}/${CAMITK_CORE_LIB} ${CAMITK_INCLUDE_DIR}/component ${CMAKE_CURRENT_BINARY_DIR})

    # set the directories where to seek for static and dynamic libraries at linking time.
    link_directories(${CAMITK_LIB_DIRECTORIES} ${CAMITK_BIN_DIR}/components)

    # check for target name
    set(APPLICATION_TARGET_NAME application-${APPLICATION_NAME})

    # get all headers, sources and do what is needed for Qt 
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${APPLICATION_NAME})

    # add the executable target
    add_executable(${APPLICATION_TARGET_NAME} ${${APPLICATION_NAME}_SOURCES})

	# Update XML Project description adding this target as a subproject of the main CamiTK project with
	# its dependencies (in the CMake target point of view)
	addSubProject(${APPLICATION_TARGET_NAME} ${COMPONENT_EXTENSION_LIBRARIES} ${CAMITK_CORE_LIB} ${TOOL_LIBRARIES})

    # target properties
    set_target_properties(${APPLICATION_TARGET_NAME} 
                          PROPERTIES OUTPUT_NAME camitk-${APPLICATION_NAME}
    )

    # tool dependencies
    if(${APPLICATION_TARGET_NAME}_NEEDS_TOOL)
      foreach(TOOL_NEEDED ${${APPLICATION_TARGET_NAME}_NEEDS_TOOL})
        add_dependencies(${APPLICATION_TARGET_NAME} ${TOOL_NEEDED})
      endforeach()
    endif()
	
	# Set the Component(s) dependency
    if(${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} COMPONENT_NEEDED_CMAKE)
		# if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (CAMITK_COMPONENT_${COMPONENT_NEEDED_CMAKE}_INTERNAL)
			add_dependencies(${APPLICATION_TARGET_NAME} component-${COMPONENT_NEEDED})
        endif()
      endforeach()
    endif()
	
	# Set the Action(s) dependency
	set(ACTION_EXTENSION_LIBRARIES)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION) 
      foreach(ACTION_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
		string(TOUPPER ${ACTION_NEEDED} ${ACTION_NEEDED}_INTERNAL)
		set(ACTION_${${ACTION_NEEDED}_INTERNAL} ON CACHE BOOL "Required by application ${APPLICATION_NAME}" FORCE )
        set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES} action-${ACTION_NEEDED})
		include_directories(${CAMITK_INCLUDE_DIR}/actions/${ACTION_NEEDED})
		# if this is not true, then the dependencies is from an external build, do not use add_dependencies
	  endforeach()
    endif()

    # dependences to core (only set for internal build)
    if (CAMITK_INTERNAL_BUILD)
		add_dependencies(${APPLICATION_TARGET_NAME} ${CAMITK_CORE_LIB})
    endif()
	
	# Set the libraries needed to build the application target
    target_link_libraries(${APPLICATION_TARGET_NAME} ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${CAMITK_LIBRARIES} ${CAMITK_CORE_LIB} ${TOOL_LIBRARIES} ${XERCESC_LIBRARY} ${ITK_LIBRARIES})
	
    # installation
    install(TARGETS ${APPLICATION_TARGET_NAME}
            RUNTIME DESTINATION bin
            COMPONENT ${APPLICATION_TARGET_NAME}
    )
	
	# For Microsoft Visual C++, sets the default application to the corresponding project
	# (i.e. launches this application when we you click on "Debug" or "Start Without Debugging" button on Visual)
	# In addition, but not the least, sets the environment to the debug dll directory for VTK (and ITK)
	#     to solve the dll incompatibility between debug and relase version of QVTK.dll and ITKCommon.dll
	# Edit, don't know why, but macro does not work correctly inside application_extension()
	if ( MSVC )
		include(CreateLaunchers)
		create_target_launcher(
			${APPLICATION_TARGET_NAME}
			RUNTIME_LIBRARY_DIRS "%VTK_DIR%/bin/$(OutDir)" "%ITK_DIR%/bin/$(OutDir)"
		)
	endif( MSVC )

endif() #APPLICATION_${APPLICATION_NAME_CMAKE}
  
endmacro()




# MACRO initXmlSubProject
#
# Initialize the xml_subprojects variable to gather CamiTK subprojects
#
# usage:
# initXmlSubProject()
macro(initXmlSubProject)

	set(CAMITK_XML_PROJECT_DESCRIPTION "<Project name=\"CamiTK\">" CACHE INTERNAL "")
	set(CAMITK_SUBPROJECTS "" CACHE INTERNAL "")
	
endmacro()

# MACRO addSubProject
#
# Add a subproject definition from targets and its dependencies (if any) to the variable
# Typically recursively call this function accross all targets to generate Project.xml file.
#
# usage:
# addSubProject(TARGET
#		DEPENDENCIES)
# TARGET : the library / exe program targeted for CMake compilation
# DEPENDENCIES : (optional) the library dependencies to build the target
macro(addSubProject target)

	# Create the xml node for this current subproject
	set(xml_subproject "\n  <SubProject name=\"${target}\">")
	foreach(dependency ${ARGN})
		set(xml_subproject ${xml_subproject} "\n    <Dependency name=\"${dependency}\"/>")
	endforeach()
	set(xml_subproject ${xml_subproject} "  </SubProject>")
	# Please note that no "\n" character is used, avoiding cmake cache to throw an "Offending entry"
	# At the moment I did not find a way to solve this one.
	
	# Add it to the list of different nodes
	set(CAMITK_XML_PROJECT_DESCRIPTION ${CAMITK_XML_PROJECT_DESCRIPTION} ${xml_subproject} CACHE INTERNAL "")
	set(CAMITK_SUBPROJECTS ${CAMITK_SUBPROJECTS} ${target} CACHE INTERNAL "")
	
endmacro()
 
# MACRO validateXmlSubProjects
#
# Validate the xml_subprojects variable by ending its xml tag
# i.e. By adding the closing </Project> xml node, this will close the xml data
# Then create the 'dir'/Project.xml file 
# Use this file to send to the DashBoard to get the CamiTK project and subprojects definition
#
# usage:
# validateXmlSubProjects(DIR)
# DIR: The path to the directory where to store the xml data file
macro(validateXmlSubProjects dir)

	# Add apidoc subproject at the end
	set(CAMITK_XML_PROJECT_DESCRIPTION ${CAMITK_XML_PROJECT_DESCRIPTION} "\n  <SubProject name=\"apidoc\">\n  </SubProject>")
	
	# Add packaging subproject at the end
	set(CAMITK_XML_PROJECT_DESCRIPTION ${CAMITK_XML_PROJECT_DESCRIPTION} "\n  <SubProject name=\"package\">\n  </SubProject>")

	# Write down the Project.xml file 
	set(CAMITK_XML_PROJECT_DESCRIPTION ${CAMITK_XML_PROJECT_DESCRIPTION} "\n</Project>")
	set(filename "${dir}/Project.xml")
	file(WRITE ${filename} ${CAMITK_XML_PROJECT_DESCRIPTION})
	message(STATUS "Generated ${filename} for project description")

	#Write down subprojects.cmake filename 
	set(filename "${dir}/Subprojects.cmake")
	file(WRITE ${filename} "set(CAMITK_SUBPROJECTS ${CAMITK_SUBPROJECTS})")
	message(STATUS "Generated ${filename} for subprojects listing")

endmacro()
