/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
// because of namespace problems with Component, put these include first

#include "MeshQuality.h"
#include <MeshComponent.h>
#include <Log.h>
#include <Application.h>
#include <InteractiveViewer.h>

#include <vtkMeshQuality.h>
#include <vtkDoubleArray.h>
#include <vtkCellData.h>

// Qt
#include <QPushButton>
#include <QVBoxLayout>
#include <QHeaderView>

// -------------------- MeshQuality --------------------
MeshQuality::MeshQuality(ActionExtension *extension) : Action(extension) {
    setName("Mesh Quality");
    setDescription("Display basic mesh quality information");
    setComponent("MeshComponent");
    setFamily("Basic Mesh");
    addTag("Quality");
    addTag("Distort");
    addTag("Degenerated");

    //-- widget lazy instanciation
    informationFrame = NULL;

    // TODO add the management of a property for selecting a quality measure (enum ?)
}

// --------------- getWidget -------------------
QWidget* MeshQuality::getWidget() {
    if (!informationFrame) {
        //-- the frame
        informationFrame = new QFrame();
        informationFrame->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
        informationFrame->setLineWidth(3);

        //-- Create the specific widgets
        informationLabel = new QLabel();
        qualityInfo = new QTableWidget();

        //-- the vertical layout, put every GUI elements in it
        QVBoxLayout * informationFrameLayout = new QVBoxLayout();

        //-- add the widgets
        informationFrameLayout->addWidget(informationLabel);
        informationFrameLayout->addWidget(qualityInfo);

        //-- set the layout for the action widget
        informationFrame->setLayout(informationFrameLayout);
    }

    apply();
    
    return informationFrame;
}

// --------------- apply -------------------
Action::ApplyStatus MeshQuality::apply() {
    // set waiting cursor (not really needed here as the action is very fast, but after all this is a demo)
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    // get the last selected image component
    MeshComponent * meshComponent = dynamic_cast<MeshComponent *>(getTargets().last());

    // the information message (in rich text, i.e., html)
    QString message = "<b>Selected Component:</b> " + meshComponent->getName() + "<br/>";

    message += "Number of Cells: " + QString("%1").arg(meshComponent->getPointSet()->GetNumberOfCells(),6) + "<br/>";

    //-- compute quality
    vtkSmartPointer<vtkMeshQuality> qualityFilter = vtkSmartPointer<vtkMeshQuality>::New();
    qualityFilter->SetInputConnection(meshComponent->getDataPort());
    qualityFilter->SetHexQualityMeasureToDistortion(); //SetHexQualityMeasureToMedAspectFrobenius(); // SetHexQualityMeasureToJacobian(); //SetTriangleQualityMeasureToArea();
    qualityFilter->Update();

    vtkDataSet* qualityMesh = qualityFilter->GetOutput();
    vtkSmartPointer<vtkDoubleArray> qualityArray = vtkDoubleArray::SafeDownCast(qualityMesh->GetCellData()->GetArray("Quality"));

    message += "Number of Hex Jacobian values:" + QString("%1").arg(qualityArray->GetNumberOfTuples(),6) + "<br/>";

    // resize the table
    qualityInfo->setRowCount(qualityArray->GetNumberOfTuples());
    qualityInfo->setColumnCount(2);
    qualityInfo->setSortingEnabled(false); // temporarily for insertion!
    QStringList headerTitles;
    headerTitles << "Cell Id" << "Quality";
    qualityInfo->setHorizontalHeaderLabels(headerTitles);
    /*  qualityInfo->horizontalHeader()->setStretchLastSection(false);
      qualityInfo->horizontalHeader()->setResizeMode(0, QHeaderView::Stretch); // First column stretch to what is needed
      qualityInfo->horizontalHeader()->setResizeMode(1, QHeaderView::Fixed); // Last column never resize
      */
    qualityInfo->horizontalHeader()->setVisible(true);
    qualityInfo->verticalHeader()->setVisible(false);

    double min = VTK_DOUBLE_MAX, max = VTK_DOUBLE_MIN, avg = 0.0;

    for(vtkIdType i = 0; i < qualityArray->GetNumberOfTuples(); i++) {
        double val = qualityArray->GetValue(i);
        qualityInfo->setItem(i, 0, new QTableWidgetItem(tr("%1").arg(i)));
        qualityInfo->setItem(i, 1, new QTableWidgetItem(tr("%1").arg(val)));
        avg += val;

        if (val<min)
            min = val;

        if (val>max)
            max = val;
    }

    qualityInfo->resizeColumnsToContents();
    qualityInfo->resizeRowsToContents();
    qualityInfo->setSortingEnabled(true);
    qualityInfo->sortByColumn(1, Qt::AscendingOrder);

    avg /= qualityArray->GetNumberOfTuples();
    message +=  "Min " + QString("%1").arg(min,6) + ", Max " + QString("%1").arg(max,6)+ ", Avg " + QString("%1").arg(avg,6)+"<br/>";

    //-- update the information label
    informationLabel->setText(message);

    // restore the normal cursor
    QApplication::restoreOverrideCursor();

    return SUCCESS;
}

