/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "VtkImageComponent.h"
#include <ImageComponent.h>

//-- Qt
#include <QFileInfo>

//-- Vtk
#include <vtkImageData.h>
// vtk image readers
#include <vtkImageReader2.h>
#include <vtkJPEGReader.h>
#include <vtkPNGReader.h>
#include <vtkTIFFReader.h>
#include <vtkBMPReader.h>
#include <vtkPNMReader.h>
#include <vtkMetaImageReader.h>
#include <vtkImageFlip.h>

using namespace camitk;

// -------------------- constructor --------------------
VtkImageComponent::VtkImageComponent(const QString& fileName) throw(AbortException)
        : ImageComponent(fileName) {
    createComponent(fileName);
}

// -------------------- createComponent --------------------
void VtkImageComponent::createComponent(const QString& filename) {
    if (!filename.isNull()) {
        try {
            // create the ImageComponent (it will register it self in the list of Components)
            // use file basename as default name
            setName(QFileInfo(filename).baseName());

            // Reader and image variables
            vtkSmartPointer<vtkImageReader2> reader;
            vtkSmartPointer<vtkImageData> image = NULL;

            // filename extension
            QString fileExt = QFileInfo(filename).suffix();
            
            // Reader initialization, depending on file extension
            if (QString::compare(fileExt, "jpg", Qt::CaseInsensitive) == 0)
                reader = vtkSmartPointer<vtkJPEGReader>::New();
            else
                if (QString::compare(fileExt, "png", Qt::CaseInsensitive) == 0)
                    reader = vtkSmartPointer<vtkPNGReader>::New();
                else
                    if ((QString::compare(fileExt, "tiff", Qt::CaseInsensitive) == 0) ||
                            (QString::compare(fileExt, "tif",  Qt::CaseInsensitive) == 0))
                        reader = vtkSmartPointer<vtkTIFFReader>::New();
                    else
                        if (QString::compare(fileExt, "bmp", Qt::CaseInsensitive) == 0)
                            reader = vtkSmartPointer<vtkBMPReader>::New();
                        else
                            if ((QString::compare(fileExt, "pbm", Qt::CaseInsensitive) == 0) ||
                                    (QString::compare(fileExt, "pgm", Qt::CaseInsensitive) == 0) ||
                                    (QString::compare(fileExt, "ppm", Qt::CaseInsensitive) == 0))
                                reader = vtkSmartPointer<vtkPNMReader>::New();
                            else
                                if ((QString::compare(fileExt, "mhd", Qt::CaseInsensitive) == 0) ||
                                        (QString::compare(fileExt, "mha", Qt::CaseInsensitive) == 0)) {
                                    reader = vtkSmartPointer<vtkMetaImageReader>::New();
                                }
                                else {
                                    throw AbortException("File format " + fileExt.toStdString() + " not supported yet.");
                                }

            if (reader) {
                reader->SetFileName(filename.toStdString().c_str());
                try {
                    reader->Update();
                }
                catch (...) {
                    throw AbortException("VTK was unable to read file " + filename.toStdString());
                }
                image = reader->GetOutput();
            }

            setImageData(image, false);
        }

        catch (AbortException e) {
            throw (e);
        }
    }
}
