/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
// pml
#include <pml/Atom.h>
#include <pml/Component.h>
#include <pml/StructuralComponent.h>
#include <pml/MultiComponent.h>
#include <pml/RenderingMode.h>
#include <pml/PhysicalModel.h>

// Core stuff
#include <InteractiveViewer.h>
#include <Geometry.h>

// vtk stuff
#include <vtkUnstructuredGrid.h>
#include <vtkPoints.h>
#include <vtkTransformFilter.h>
#include <vtkPointData.h>
#include <vtkCellData.h>
#include <vtkSmartPointer.h>

#include <Log.h>


#include "PMManagerDC.h"
#include "AtomDC.h"
#include "AtomDCWidget.h"
#include "AtomDCProperties.h"
#include "AtomDCPopup.h"
#include "StructuralComponentDC.h"
#include "AtomDecoration.h"


// -------------------- constructor --------------------
AtomDC::AtomDC ( camitk::Component *parent, PMManagerDC * pmManagerDC, Atom *a ) : camitk::Component ( parent, "default atom", camitk::Component::GEOMETRY) {
    // store the represented atom
    myAtom = a;
    myPMManagerDC = pmManagerDC;

    QString name = QString::number ( myAtom->getIndex() );

    if ( name.isEmpty() ) {
        name = myAtom->getName().c_str();
    }

    camitk::Component::setName ( name );

    // associate the DC with the atom in the map
    myPMManagerDC->addAtomDCPair ( std::AtomDCPair ( myAtom, this ) );

    // by default, the atom is not moved, of course !
    alreadyMoved = false;

    // initialize the popup
    myPopupMenu = NULL;
    myProp = NULL;

    initRepresentation();
    // add it in the InteractiveViewer
    setVisibility ( InteractiveViewer::get3DViewer(), true );
}

// -------------------- destructor --------------------
AtomDC::~AtomDC() {
    delete myPopupMenu;
    myPopupMenu = NULL;

    delete myProp;
    myProp = NULL;

    // delete decoration
    foreach ( AtomDecoration *deco, decorations ) {
        delete deco;
    }
    decorations.clear();
}

// -------------------- setParent --------------------
void AtomDC::setParent ( InterfaceNode * ) {
    // do nothing (i.e. do not change parentDC and do not ask the previous
    // parent to remove this AtomDC)...
}

// -------------------- registerIndexInSCDC --------------------
unsigned int AtomDC::registerIndexInSCDC ( std::IndexInParentItemPair p ) {
    // remove the old one
    unregisterIndexInSCDC ( dynamic_cast<StructuralComponentDC *> ( p.first ) );
    // add the new one
    mySCDCindexes.insert ( p );

    return mySCDCindexes.size();
}

// -------------------- getOrderNumberInSCDC --------------------
int AtomDC::getOrderNumberInSCDC ( StructuralComponentDC * parent ) {
    std::IndexInParentItemMapIterator result = mySCDCindexes.find ( dynamic_cast<InterfaceNode *> ( parent ) );

    if ( result == mySCDCindexes.end() )
        return ( -1 );
    else
        return result->second;
}

// -------------------- unregisterIndexInSCDC --------------------
unsigned int AtomDC::unregisterIndexInSCDC ( StructuralComponentDC * parent ) {
    mySCDCindexes.erase ( dynamic_cast<InterfaceNode *> ( parent ) );
    return mySCDCindexes.size();
}

// -------------------- getPopupMenu --------------------
QMenu * AtomDC::getPopupMenu ( QWidget* parent ) {
    if ( !myPopupMenu ) {
        myPopupMenu = new AtomDCPopup ( this, parent );
    }

    return myPopupMenu;
}

// -------------------- getPropertyWidget --------------------
QWidget * AtomDC::getPropertyWidget ( QWidget* parent ) {
    // ask the pmManagerDC
    return myPMManagerDC->getAtomDCWidget ( this, parent );
}

// -------------------- getPropertyObject --------------------
QObject * AtomDC::getPropertyObject() {
    if ( !myProp ) {
        myProp = new AtomDCProperties ( this );
    }

    return myProp;
}


// -------------------- initRepresentation --------------------
void AtomDC::initRepresentation() {
    // create the point for the atom (only one atom here)
    vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();
    vtkIdType *vtkPointIndex = new vtkIdType [1];

    // NOTE : the index of each atom in the vtk structure is in fact its order number in the atoms structure (not its actual index)
    double pos[3];
    myAtom->getPosition ( pos );
    thePoints->InsertPoint ( 0, pos[0], pos[1], pos[2] );
    vtkPointIndex[0] = 0;

    // create the cell (of type VTK_VERTEX) to represent this cell
    vtkSmartPointer<vtkUnstructuredGrid> uGrid = vtkSmartPointer<vtkUnstructuredGrid>::New();
    uGrid->Allocate ( 2 );
    uGrid->InsertNextCell ( VTK_VERTEX, (vtkIdType) 1, vtkPointIndex );
    uGrid->SetPoints ( thePoints );
    uGrid->Update();

    delete [] vtkPointIndex;

    // create the new geometry
    myGeometry = new Geometry ( this->getName(), uGrid, InterfaceGeometry::Points );

    if ( isSelectedFlag ) {
        // add the glyph now
        isSelectedFlag = false;
        setSelected ( true );
    }
}


// -------------------- getPosition --------------------
void AtomDC::getPosition ( double & x, double & y, double & z ) {
    double pos[3];
    myAtom->getPosition ( pos );

    x = pos[0];
    y = pos[1];
    z = pos[2];
}

// -------------------- setPosition --------------------
void AtomDC::setPosition ( double x, double y, double z ) {
    if ( !alreadyMoved ) {
        // change the position
        myAtom->setPosition ( x, y, z );

        // update the position and start the up-cascade process chains of update
        updatePosition();

        // set the flag
        alreadyMoved = true;
    }
}

// -------------------- updatePosition --------------------
void AtomDC::updatePosition() {

    // advise the manager, so that the file is tagged as modified (pop up when closed...)
    myPMManagerDC->setModified();

    // get the new atom position
    double pos[3];
    myAtom->getPosition ( pos );

    // update the 3D representation

    if ( myGeometry ) {
        myGeometry->setPointPosition ( 0, pos[0], pos[1], pos[2] );
    }

    /*  // CAll the updateAtomPos of the myExtension that launch the cascade/chained reaction of
      // update calls
      dynamic_cast<PMManager *>(getExtension())->updateAtomPosition(this);
    */

    // look at all the DCs that are using this atom,
    // for each of them
    //     - get its DC
    //     - tell it that this particular atom has been modified
    for ( unsigned int i = 0;i < myAtom->getNumberOfStructuralComponents();i++ ) {
        // tell the SCDC the atom position has changed
        myPMManagerDC->getDC ( myAtom->getStructuralComponent ( i ) )->updatePosition ( this );
    }

    // Look now at all the SC containing a cell containing this atom, and tell
    // them this atom position has changed
    std::IndexInParentItemMapIterator it;

    for ( it = mySCDCindexes.begin(); it != mySCDCindexes.end(); it++ ) {
        // tell the DC the atom position has changed (and even give it the correct order nr !)
        dynamic_cast<StructuralComponentDC *> ( it->first )->updatePosition ( this, it->second );
    }

}

// -------------------- setPointSet --------------------
void AtomDC::setPointSet ( vtkSmartPointer<vtkPointSet> newPointSet ) {
    // tag this DC as modified
    getTopLevelComponent()->setModified();

    // update the atom's position
    vtkSmartPointer<vtkPoints> vtkPts = newPointSet->GetPoints();
    double pos[3];

    // get the new position
    vtkPts->GetPoint ( 0, pos );

    // set the new position for the atom (it will automatically update the DC, the atom, and all the cells, SC containing this atom)
    setPosition ( pos[0], pos[1], pos[2] );
}

// -------------------- setName --------------------
void AtomDC::setName ( const QString & n ) {
    myAtom->setName ( n.toStdString() );
    camitk::Component::setName ( n );
}

//-------------------------- setEnhancedModes --------------------------
void AtomDC::setEnhancedModes ( const EnhancedModes em ) {
//   if (isSelectedFlag)
//     camitk::Component::setEnhancedModes(InterfaceGeometry::Normal);
//   else
//     camitk::Component::setEnhancedModes(em);
}

//------------------------pointPicked---------------------
void AtomDC::pointPicked ( vtkIdType /* always 0 */, bool pickingIsSelecting ) {
    setSelected(pickingIsSelecting);
}

//------------------------setSelected---------------------
void AtomDC::setSelected ( const bool d, const bool /*recursive (not used)*/ ) {
    if ( myGeometry ) {
        if ( d && !isSelectedFlag ) {
            // always display the sphere
            float includedSphereRadius = getTopLevelComponent()->getBoundingRadius();
            if ( includedSphereRadius==0.0 )
                includedSphereRadius = myPMManagerDC->getBoundingRadius();

            myGeometry->setGlyphType( Sphere, includedSphereRadius / 75.0 );
            myGeometry->getProp("glyph")->VisibilityOn();
        }

        if ( !d && isSelectedFlag ) {
            myGeometry->getProp("glyph")->VisibilityOff();
        }

    }

    camitk::Component::setSelected ( d );
}

//------------------------resetAlreadyMovedFlag---------------------
void AtomDC::resetAlreadyMovedFlag() {
    // reset the flat
    alreadyMoved = false;
}

//------------------------ getPixmap ---------------------
#include "atom_20x20.xpm"
QPixmap * AtomDC::myPixmap = NULL;
QPixmap AtomDC::getIcon() {
    if ( !myPixmap ) {
        myPixmap = new QPixmap ( atom_20x20 );
    }

    return ( *myPixmap );
}


//------------------------ doubleClicked ---------------------
bool AtomDC::doubleClicked() {
    CAMITK_INFO ( "AtomDC","doubleClicked","AtomDC #" << myAtom->getIndex() << " got double-clicked..." );
    return false;
    /*
        double xyz[3];

        // get the first display point
        if (InteractiveViewer::get3DViewer()->getDisplayPoints(xyz, 0)) {
            setPosition(xyz[0], xyz[1], xyz[2]);
            return true; // the point was modified
        }
        else
            return false;
    */
}

//------------------------addPointData---------------------
void AtomDC::addPointData ( StructuralComponentDC *sc, double * ptr ) {
    pointData.push_back ( ptr );
    pointDataSC.insert ( sc );
}

//------------------------clearPointData---------------------
void AtomDC::clearPointData() {
    pointData.clear();
}

//------------------------updatePointData---------------------
void AtomDC::updatePointData ( const double value ) {
    if ( value >= 0.0 ) {
        // make sure the point data is created
        for ( std::set<StructuralComponentDC *>::iterator it = pointDataSC.begin(); it != pointDataSC.end(); it++ ) {
            ( *it )->createPointData();
        }

        for ( unsigned int i = 0; i < pointData.size(); i++ ) {
            *pointData[i] = value;
        }
    }
}

//------------------------getDecoration---------------------
AtomDecoration *AtomDC::getDecoration ( const QString & name, GeometricObject::Geometry type ) {
    // check for the asked decoration (name and type must match)
    QMap<QString, AtomDecoration*>::const_iterator it = decorations.find ( name );
    while ( it != decorations.end() && it.key() == name && it.value()->getType() != type ) {
        ++it;
    }

    if ( it != decorations.end() && it.key() == name && it.value()->getType() == type ) {
        // if already in the list...
        return it.value();
    } else {
        // create a new decoration
        AtomDecoration *deco = new AtomDecoration ( this, type );
        // insert it
        decorations.insert ( name, deco );
        // return it
        return deco;
    }
}
