(* $Id: dep_view.ml,v 1.6 2005/09/27 15:02:20 zoggy Exp $ *)

(** Caml modules dependencies view.*)

module V = Cam_plug.View
open Odoc_info

module C = Cam_plug.Commands

let default_dot_ppi = 72.0

let node_color = "deepskyblue"
let top_color = "coral1"

let p_dbg s = ()
(*
let p_dbg = prerr_endline
*)

let gen_dot f modules =
  let oc = open_out f in
  let header = "digraph G {\n"^
(*    "  size=\"10,7.5\";\n"^*)
    "  ratio=\"fill\";\n"^
(*    "  rotate=90;\n"^*)
    "  fontsize=\"10pt\";\n"^
    "  rankdir = TB ;\n"
  in
  output_string oc header;
  let t = Hashtbl.create 13 in
  List.iter
    (fun m ->
      List.iter (fun m -> Hashtbl.replace t m true) m.Module.m_top_deps
    )
    modules;
  let f_node m =
    let color =
      try ignore (Hashtbl.find t m.Module.m_name); node_color
      with Not_found -> top_color
    in
    Printf.fprintf oc "\"%s\" [ shape=rectangle, style=filled, color=%s];\n" m.Module.m_name color
  in
  List.iter f_node modules;
  let print_one_dep src dest =
    Printf.fprintf oc "\"%s\" -> \"%s\";\n" src dest
  in
  let f_dep m =
    List.iter (print_one_dep m.Module.m_name) m.Module.m_top_deps
  in
  List.iter f_dep modules;
  output_string oc "}\n";
  close_out oc

(*c==v=[File.string_of_file]=1.0====*)
let string_of_file name =
  let chanin = open_in_bin name in
  let len = 1024 in
  let s = String.create len in
  let buf = Buffer.create len in
  let rec iter () =
    try
      let n = input chanin s 0 len in
      if n = 0 then
        ()
      else
        (
         Buffer.add_substring buf s 0 n;
         iter ()
        )
    with
      End_of_file -> ()
  in
  iter ();
  close_in chanin;
  Buffer.contents buf
(*/c==v=[File.string_of_file]=1.0====*)

(*c==v=[Misc.md5sum_of_string]=1.0====*)
let md5sum_of_string s =
  let com = Printf.sprintf "echo %s | md5sum | cut -d\" \" -f 1"
      (Filename.quote s)
  in
  let ic = Unix.open_process_in com in
  let s = input_line ic in
  close_in ic;
  ignore (Unix.wait ());
  s
(*/c==v=[Misc.md5sum_of_string]=1.0====*)

let show image file zoom_file zoom =
  let com = Printf.sprintf "convert -resize %d%% %s %s"
      zoom
      (Filename.quote file)
      (Filename.quote zoom_file)
  in
  match Sys.command com with
    0 -> image#set_file zoom_file
  | n -> failwith (Printf.sprintf "Exec error %d: %s" n com)

let analyse_annot_dot_file modules f =
  let s = string_of_file f in
  let re_bb = Str.regexp "bb=\"\\([0-9]+\\),\\([0-9]+\\),\\([0-9]+\\),\\([0-9]+\\)\"" in
  try
    let pos = Str.search_forward re_bb s 0 in
    let width = int_of_string (Str.matched_group 3 s) in
    let height = int_of_string (Str.matched_group 2 s) in
    p_dbg (Printf.sprintf "width=%d,height=%d" width height);
    let f acc m =
      let s_re =
	Printf.sprintf
	  "%s[ \t\n\t]\\[.*pos=\"\\([0-9]+\\),\\([0-9]+\\)\".*width=\"\\([0-9]+\\.[0-9]+\\)\".*height=\"\\([0-9]+\\.[0-9]+\\)\".*$"
	  m.Module.m_name
      in
      let re = Str.regexp s_re in
      try
	let pos =  Str.search_forward re s 0 in
	let (x,y,w,h) =
	  (int_of_string (Str.matched_group 1 s),
	   int_of_string (Str.matched_group 2 s),
	   float_of_string (Str.matched_group 3 s),
	   float_of_string (Str.matched_group 4 s)
	  )
	in
	let w = w *. default_dot_ppi in
	let h = h *. default_dot_ppi in
	let x1 = (float x) -. w /. 2.0 in
	let y1 = (float y) -. h /. 2.0 in
	let x2 = (float x) +. w /. 2.0 in
	let y2 = (float y) +. h /. 2.0 in
(*
	let x = int_of_float ((float x) *. 1.34) in
	let y = int_of_float ((float y) *. 1.34) in
	let x2 = x2 *. 1.34 in
	let y2 = y2 *. 1.34 in
*)
	p_dbg (Printf.sprintf "module %s: x=%f y=%f x2=%f y2=%f" m.Module.m_name x1 y1 x2 y2);
	(x1,y1,x2,y2, m) :: acc
      with
	Not_found ->
	  p_dbg (Printf.sprintf "Module %s not found; re=%s" m.Module.m_name s_re);
	  acc
    in
    (width, height, List.fold_left f [] modules)
  with
    _ ->
      (1, 1, [])

class box view_name ?root_module dir =
  let hash = md5sum_of_string
      (Printf.sprintf "%s%s%s" Cam_plug.login dir
	 (match root_module with None -> "" | Some s -> s)
      )
  in
  let dot_file = Printf.sprintf "/tmp/depview_%s.dot" hash in
  let annot_dot_file = Printf.sprintf "/tmp/depview_%s.dot_annot" hash in
  let png_file = Printf.sprintf "/tmp/depview_%s.png" hash in
  let vbox = GPack.vbox () in
  let hbox = GPack.hbox ~spacing:5 ~packing:(vbox#pack ~expand: false) () in
  let _ = GMisc.label ~text: "Zoom:" ~packing: (hbox#pack ~padding: 4 ~expand: false) () in
  let zooms =
      [ 10 ; 20 ; 30 ; 40 ; 50 ; 60 ; 70 ; 80 ; 90 ; 100 ; 120 ]
  in
  let wcombo = GEdit.combo
      ~popdown_strings: (List.map (fun s -> Printf.sprintf "%d%%" s) zooms)
      ~allow_empty:false
      ~enable_arrow_keys:true
      ~value_in_list:true
      ~packing: (hbox#pack ~expand: false)
      ()
  in
  let wb_refresh = GButton.button ~label: "Refresh"
      ~packing: (hbox#pack ~expand: false ~padding: 4) ()
  in
  let wscroll = GBin.scrolled_window
      ~vpolicy: `AUTOMATIC
      ~hpolicy: `AUTOMATIC
      ~packing: (vbox#pack ~expand: true)
      ()
  in
  let evt_box = GBin.event_box ~packing: wscroll#add_with_viewport () in
  let image = GMisc.image ~file: png_file ~packing:evt_box#add () in
  let _ = image#set_xalign 0.0 in
  let _ = image#set_yalign 0.0 in
  object(self)
    val mutable current_zoom = 100.0
    val mutable dot_width = 1
    val mutable dot_height = 1
    val mutable modules = []

    method zoom_file_of_zoom zoom =
      Printf.sprintf "%s_%d%%.png" (Filename.chop_extension png_file) zoom

    method box = vbox

    method zoom () =
      let z =
	try Scanf.sscanf wcombo#entry#text "%d%%" (fun a -> Some a)
	with _ -> None
      in
      match z with
	None -> ()
      |	Some 100 ->
	  current_zoom <- 100.0;
	  image#set_file png_file
      |	Some z ->
	  let f = self#zoom_file_of_zoom z in
	  if Sys.file_exists f then
	    image#set_file f
	  else
	    show image png_file f z;
	  current_zoom <- float z

    method modules_to_handle =
      let mods =
        match (Unix.lstat dir).Unix.st_kind with
          Unix.S_DIR -> Cam_plug.Modules_view.get_modules_from_dir dir
        | _ -> Odoc_info.load_modules dir
      in
      let mods =
	match root_module with
	  None -> mods
	| Some name ->
	    let rec iter acc = function
		[] -> acc
              | m_name :: q ->
		  if List.exists (fun m -> m.Module.m_name = m_name) acc then
                    iter acc q
		  else
                    let m_opt =
                      try Some (List.find (fun m -> m.Module.m_name = m_name) mods)
                      with Not_found -> None
                    in
                    match m_opt with
                      None -> iter acc q
                    | Some m -> iter (m::acc) (m.Module.m_top_deps @ q)
            in
            try iter [] [String.capitalize name]
            with Not_found ->
              Cam_hooks.error_message ("Could not find module "^name);
	      []
      in
      Odoc_info.Dep.kernel_deps_of_modules mods;
      let pred =
	let l = List.map (fun m -> m.Module.m_name) mods in
	fun s -> List.mem s l
      in
      List.iter
	(fun m ->
	  m.Module.m_top_deps <- List.filter pred m.Module.m_top_deps
	)
	mods;
      mods

    method update_modules_info =
      let mods = self#modules_to_handle in
      let (w,h,l) = analyse_annot_dot_file mods annot_dot_file in
      dot_width <- w;
      dot_height <- h;
      modules <- l

    method clean_files =
      List.iter (fun f -> try Sys.remove f with _ -> ())
	[ dot_file ; annot_dot_file ; png_file];
      List.iter (fun z -> try Sys.remove (self#zoom_file_of_zoom z) with _ -> ()) zooms;

    method refresh () =
      self#clean_files;
      let mods = self#modules_to_handle in
      gen_dot dot_file mods;
      let com = Printf.sprintf
	  "dot -s%d -y %s > %s && dot -s%d -T png -o %s %s "
	  (int_of_float default_dot_ppi)
	  (Filename.quote dot_file)
	  (Filename.quote annot_dot_file)
	  (int_of_float default_dot_ppi)
	  (Filename.quote png_file)
	  (Filename.quote dot_file)
      in
      (
       match Sys.command com with
	 0 ->
	   let s = string_of_file annot_dot_file in
	   p_dbg s;
	   self#update_modules_info ;
	   self#zoom ()
       | n -> GToolbox.message_box "Error"
	     (Printf.sprintf "Exec error %d: %s" n com)
      );

    method on_button1_press x y =
      p_dbg (Printf.sprintf "Button press ! x=%d y=%d" x y);
      let px = image#pixbuf in
      let dc =
	{
   	  Gobject.kind = `INT ;
   	  Gobject.proj = (function `INT n -> n | _ -> assert false) ;
	  Gobject.inj = (fun n -> `INT n);
	}
      in
      let image_width = Gobject.Property.get px
	  { Gobject.name = "width" ; Gobject.conv = dc }
      in
      let image_height = Gobject.Property.get px
	  { Gobject.name = "height" ; Gobject.conv = dc }
      in
      p_dbg (Printf.sprintf "image width=%d height=%d" image_width image_height);
      let ratio_x = (float image_width) /. (float dot_width) in
      let ratio_y = (float image_height) /. (float dot_height) in
      try
	let x = float x in
	let y = float y in
	let (x1,y1,x2,y2,m) = List.find
	    (fun (x1,y1,x2,y2,_) ->
	      x1 *. ratio_x <= x && x <= x2 *. ratio_x &&
	      y1 *. ratio_y <= y && y <= y2 *. ratio_y
	    )
	    modules
	in
	p_dbg (Printf.sprintf "Module %s clicked pixels: x1=%f x2=%f y1=%f y2=%f ratio_x=%f ratio_y=%f"
			 m.Module.m_name
			 (x1 *. ratio_x) (x2 *. ratio_x)
			 (y1 *. ratio_y) (y2 *. ratio_y)
			 ratio_x ratio_y
		      );
	let files =
	  (match m.Module.m_loc.Odoc_info.loc_inter with
	    None -> []
	  | Some (f,_) -> [f]
	  ) @
	  (match m.Module.m_loc.Odoc_info.loc_impl with
	    None -> []
	  | Some (f,_) -> [f]
	  )
	in
	let entries =
	  (`I ("view only this module's dependencies",
	       let res = Printf.sprintf "%s#%s" dir m.Module.m_name in
	       fun () -> ignore(Cam_view.open_ressource res view_name [| |]))
	  ) ::
	  (
	   List.map
	     (fun f ->
	       `M (Cam_files.escape_menu_label (Filename.basename f),
		   Cam_files.edition_commands_menu_entries f)
	     )
	     files
	  )
	in
	GToolbox.popup_menu ~entries ~button: 1 ~time: Int32.zero
      with
	Not_found ->
	  ()

    method on_button3_press x y =
      let entries = List.map
	  (fun z ->
	    let t = Printf.sprintf "%d%%" z in
	    `I (t, fun () -> wcombo#entry#set_text t)
	  )
	  zooms
      in
      GToolbox.popup_menu ~entries ~button: 3 ~time: Int32.zero

    initializer
      wcombo#entry#set_editable false;
      wcombo#entry#set_text "100%";
      ignore (wcombo#entry#connect#changed self#zoom );
      ignore (wb_refresh#connect#clicked self#refresh);
      ignore
	(evt_box#event#connect#button_press ~callback:
	   (fun evt ->
	     match GdkEvent.Button.button evt with
	       1 ->
		 GdkEvent.get_type evt = `BUTTON_PRESS &&
		 (
		  let x = int_of_float (GdkEvent.Button.x evt) in
		  let y = int_of_float (GdkEvent.Button.y evt) in
		  self#on_button1_press x y;
		  true
		 )
	     | 3 ->
		 GdkEvent.get_type evt = `BUTTON_PRESS &&
		 (
		  let x = int_of_float (GdkEvent.Button.x evt) in
		  let y = int_of_float (GdkEvent.Button.y evt) in
		  self#on_button3_press x y;
		  true
		 )
	     | n -> true
	   )
	);
      if not (Sys.file_exists annot_dot_file) then
	self#refresh ()
      else
	self#update_modules_info;

  end

class view
    (name : V.view_name)
    (dir : V.ressource_name)
    (box : box)
    (close_window_on_close : bool) =
  object (self)
    method changed = false
    method close : bool = close_window_on_close
    method name = name
    method refresh = box#refresh ()
    method ressource = dir
    method ressource_kind : V.ressource_kind = `Dir
  end

class factory : V.view_factory =
  object (self)
    method private dir_and_root_module_of_string s =
      try
	let len = String.length s in
	let p = String.rindex s '#' in
	let dir = String.sub s 0 p in
	let root_module =
	  if len > p + 1 then
	    Some (String.sub s (p+1) (len-p-1))
	  else
	    None
	in
	(dir,root_module)
      with
	Not_found ->
	  (s, None)

    method create res_name args =
      let (dir,root_module) = self#dir_and_root_module_of_string res_name in
      let box = new box self#name ?root_module dir in
      let v = new view (self#name) res_name box true in
      let w = V.create_view_window
          ~title: (Printf.sprintf "%s [%s]" res_name self#name)
          v
      in
      let _ = w#vbox#pack ~expand: true box#box#coerce in
      (v, w#window)

    method create_no_window window res_name args =
      let (dir,root_module) = self#dir_and_root_module_of_string res_name in
      let box = new box self#name ?root_module dir in
      let v = new view (self#name) res_name box false in
      (v, box#box#coerce)

    method known_ressource_kinds = [`Dir ; `File]
    method name = "dependencies"
  end

let _ = V.register_factory (new factory)
